/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ExceptionTest.C
 *    Test exceptions
 *
 * @author Christof Schröter
 * @date   2020/03/27
 */

#include <boost/test/unit_test.hpp>

#include <error/Exceptions.h>
#include <rpc/RPCError.h>

#include <platform/Typename.h>

#include <serialization/BinarySerializer.h>
#include <serialization/XMLSerializer.h>

#include <serialization/adapters/boost/shared_ptr.hpp>

using namespace mira;
using namespace std;

/*
template <typename Nested, typename Outer>
void throw_nested() {
  try {
    throw Nested(typeName<Nested>() + " nested");
  } catch (const std::exception& e) {
    std::throw_with_nested(Outer(typeName<Outer>() + " outer"));
  }
}

template<typename T>
bool has_nested(const std::exception& ex) {
	try {
		std::cout << "outer: " << ex.what() << std::endl;
		std::rethrow_if_nested(ex);
	}
	catch(T& ex) {
		std::cout << "nested: " << ex.what() << std::endl;
		return true;
	}

	return false;
}


BOOST_AUTO_TEST_CASE(NestedExceptionTest)
{
	try {
		throw_nested<XLogical, XRPC>();
	}
	catch(XRPC& ex) {
		BOOST_REQUIRE(has_nested<XLogical>(ex));
		return;		
	}

	BOOST_FAIL("Expected XRPC");
}
*/

BOOST_AUTO_TEST_CASE(ExceptionSerializationTest)
{
	XMLDom xml;
	Buffer<uint8> buffer;

	std::string what;
	std::string callStack;
	ThreadID threadID;

	try {
		try {
			try {
				MIRA_THROW(XLogical, "XLogical test")
			}
			catch(XLogical& ex) {
				MIRA_RETHROW(ex, "Additional info");
			}		
		}
		catch(XLogical& ex) {
			MIRA_RETHROW(ex, "In ExceptionSerializationTest");
		}
	}
	catch(SerializableException& ex) {
		// store properties of original execption for reference
		what = ex.what();
		std::stringstream ss;
		ss << ex.callStack();
		callStack = ss.str();
		threadID = ex.getThreadID();

		SerializableException* exPtr = &ex;

		XMLSerializer xs(xml, XMLSerializer::NO_COMMENTS);
		xs.serialize("Exception", exPtr, "Exception pointer");

//		std::cout << "original: " << std::endl;
//		std::cout << xml.saveToString() << std::endl;

		BinaryBufferSerializer bs(&buffer);
		bs.serialize(exPtr);
	}

	//
	// Now the exception is preserved in xml and buffer (and only there)
	//

	for (;;) {
		try {
			SerializableException* exPtr;
			XMLDeserializer xd(xml);

			xd.deserialize("Exception", exPtr);

			XMLDom xml2;
			XMLSerializer xs(xml2, XMLSerializer::NO_COMMENTS);
			xs.serialize("Exception", exPtr, "Exception pointer");

	//		std::cout << "xml restored: " << std::endl;
	//		std::cout << xml2.saveToString() << std::endl;

			exPtr->raise();
		}
		catch(XLogical& ex) {
			BOOST_CHECK_EQUAL(ex.what(), what);

			std::stringstream ss;
			ss << ex.callStack();
			BOOST_CHECK_EQUAL(ss.str(), callStack);

			BOOST_CHECK_EQUAL(ex.getThreadID(), threadID);
			break;
		}

		BOOST_FAIL("Expected XLogical");
		break;
	}

	for (;;) {
		try {
			SerializableException* exPtr;
			BinaryBufferDeserializer bd(&buffer);

			bd.deserialize(exPtr);

			XMLDom xml2;
			XMLSerializer xs(xml2, XMLSerializer::NO_COMMENTS);
			xs.serialize("Exception", exPtr, "Exception pointer");

//			std::cout << "binary restored: " << std::endl;
//			std::cout << xml2.saveToString() << std::endl;

			exPtr->raise();
		}
		catch(XLogical& ex) {
			BOOST_CHECK_EQUAL(ex.what(), what);

			std::stringstream ss;
			ss << ex.callStack();
			BOOST_CHECK_EQUAL(ss.str(), callStack);

			BOOST_CHECK_EQUAL(ex.getThreadID(), threadID);
			break;
		}

		BOOST_FAIL("Expected XLogical");
		break;
	}
}
