/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CallStack.C
 *    Implementation of CallStack.h
 *
 * @author Erik Einhorn, Christian Martin
 * @date   2009/12/26
 */

#include <error/CallStack.h>

using namespace boost;
using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ostream& operator<<(ostream& os, const CallStack::Symbol& s)
{
	// max path length. if path is longer than this constant
	// the library name will be extracted from path and used instead
	const std::size_t extractLibraryLength = 32;
	// extract the library name from the whole file-path, if path is too long:
	string file = s.file;
	if(file.size()>extractLibraryLength) {
		size_t pos = s.file.find_last_of("/\\");
		if(pos!=string::npos)
			file = s.file.substr(pos+1);
	}

	os <<  s.addressStr << " in " << s.nameDemangled << " from " << file;
	return os;
}

///////////////////////////////////////////////////////////////////////////////

const CallStack& CallStack::operator=(const CallStack& other)
{
	mStack = other.mStack;
	mStackSize = other.mStackSize;
	mStackStart = other.mStackStart;
	mSymbols = other.mSymbols;
	return *this;
}

const CallStack::Symbol& CallStack::operator[](size_t i) const
{
	static CallStack::Symbol emptySymbol;
	obtainSymbols();
	i+=mStackStart;
	if(i<mSymbols.size())
		return mSymbols[i];
	else
		return emptySymbol;
}

ostream& operator<<(ostream& os, const CallStack& stack)
{
	for (size_t i = 0; i < stack.size(); ++i) {
		const CallStack::Symbol& s = stack[i];
		os << "#" << i << "  " << s << endl;
		string loc = s.sourceLocation();
		if(!loc.empty())
			os << "    at " << loc << endl;
	}
	return os;
}

///////////////////////////////////////////////////////////////////////////////

const std::vector<CallStack::Symbol>& CallStack::getSymbols()
{
	obtainSymbols();
	return mSymbols;
}

void CallStack::setSymbols(const std::vector<CallStack::Symbol>& symbols)
{
	mSymbols = symbols;
}

///////////////////////////////////////////////////////////////////////////////

}
