/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RecursiveMemberReflector.h
 *    Description.
 *
 * @author Erik Einhorn
 * @date   2010/09/22
 */

#ifndef _MIRA_RECURSIVEMEMBERREFLECTOR_H_
#define _MIRA_RECURSIVEMEMBERREFLECTOR_H_

#include <type_traits>

#ifndef Q_MOC_RUN
#include <boost/mpl/eval_if.hpp>
#include <boost/mpl/identity.hpp>
#endif

#include <error/LoggingCore.h>

#include <factory/Factory.h>

#include <serialization/Accessor.h>

#include <serialization/ReflectMemberMeta.h>
#include <serialization/AbstractReflector.h>
#include <serialization/PolymorphicPointerReflector.h>

#include <serialization/Array.h>

#include <serialization/IsAtomicSerializable.h>
#include <serialization/IsObjectTrackable.h>
#include <serialization/IsCollection.h>
#include <serialization/IsTransparentSerializable.h>

#include <serialization/IgnoreMissing.h>

#include <serialization/VoidCast.h>

#include <utils/Singleton.h>
#include <utils/ToString.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace serialization {

//////////////////////////////////////////////////////////////////////////////

/**
 * For internal use only:
 * Holds value=true, if the type T is ever reflected as pointer with Reflector
 * within this translation unit (C file), otherwise false.
 * If it is used as pointer it is set to true in
 * RecursiveMemberReflectorBase::pointer() via instantiating
 * SetReflectedAsPointer as EagerSingleton. This somewhat complicated
 * mechanism makes sure, that the value is correctly set to true immediately
 * at the start of the process, even before the first serialization operation
 * is performed.
 */
template <typename T, typename Reflector>
struct ReflectedAsPointer {
	ReflectedAsPointer() : value(false) {}
	bool value;
};

template <typename T, typename Reflector>
struct SetReflectedAsPointer {
	SetReflectedAsPointer() {
		LazySingleton<ReflectedAsPointer<T,Reflector>>::instance().value = true;
	}
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

/**
 * Can be thrown by subclasses of RecursiveMemberReflectorBase to indicate
 * that they did not find a certain member, e.g. when deserializing a class.
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XMemberNotFound, XIO)

/**
 * Adds additional marker to XMemberNotFound, to indicate that there was
 * no default value available and the XMemberNotFound exception must not
 * be caught in surrounding classes to handle the issue.
 */
class XMemberNotFound_NoDefault : public XMemberNotFound
{
	MIRA_OBJECT(XMemberNotFound_NoDefault)
protected:
	friend class ClassFactoryDefaultConstClassBuilder;
	XMemberNotFound_NoDefault() MIRA_NOEXCEPT_OR_NOTHROW {}

public:
	XMemberNotFound_NoDefault(XMemberNotFound& ex) MIRA_NOEXCEPT_OR_NOTHROW :
		XMemberNotFound(ex) {}

	virtual void raise(bool recursive = false) { throw *this; }
};

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * The RecursiveMemberReflectorBase is a base class for all Reflectors that
 * are used to visit the reflect methods of the classes recursively in order to
 * collect information on the members of these classes. It therefore is the
 * base class of most @ref Serializer and @ref Deserializer classes of the
 * Serialization Framework.
 *
 * If you implement your own derived Reflector, you can overwrite the
 * following visiting methods:
 *
 * - atomic()
 * - enumeration()
 * - object()
 * - collection()
 * - pointer()
 * - pointerNormal()
 * - pointerPolymorphic()
 * - pointerAbstract()
 *
 * These visit-methods are called for each reflected member based on the
 * member type. (See documentation of each method for details!) The type
 * of the member is inferred automatically at compile time.
 *
 *
 * Implementation Detail:
 *
 * The following types are differentiated:
 *
 * - A1: Members that are no classes or structs and just atomic
 *         types like int, float, ...
 * - A2: Members that are enumerations.
 * - A3: Members are arrays.
 *
 * - B1: Members that are complex classes and that contain a
 *       reflect method (i.e. classes that were built to work with
 *       the serialization framework).
 * - B2: Members that are complex classes but that do not have a
 *       intrusive reflect method. For those classes a non-intrusive
 *       reflect method must be available.
 * - B?c: Members that are complex classes that have the IsCollection trait
 *
 * - C:  Members that are pointers
 * - C1: Members that are pointers to "normal" objects.
 * - C2: Members that are pointers to polymorphic objects that are derived from
 *       mira::Object.
 * - C3: Members that are pointers to objects of abstract classes that are NOT
 *       derived from mira::Object.
 *
 * Depending on the type, the calling sequences are as follows:
 *
 * For type A1:
 *   - atomic() is called. (Should be overwritten and implemented)
 *
 * For type A2:
 *   - enumeration() is called. The default implementation resolves the
 *     enumeration to an atomic type and will invoke a call of atomic().
 *     Hence you don't need to overwrite this method, if you implement atomic().
 *
 * For type A3:
 *   - collection(a, size) is called. The default implementation creates an Array
 *     object and delegates to its reflect method. Hence, you don't need to
 *     overwrite this method.
 *
 * For type B1 and B2:
 *   - object() is called. The default implementation reflects the members of
 *     the object recursively
 *
 * For type C1-3:
 *   - pointer() is called. The default implementation inspects the class of
 *     the object the pointer is pointing to. Depending on the type of that
 *     class (C1-C3) pointerNormal(), pointerPolymorphic() or pointerAbstract()
 *     are called afterwards. Hence, you don't need to overwrite this method,
 *     if you implement these three methods.
 *
 * <b>Methods that should be implemented:</b>
 * atomic(), pointerNormal(), pointerPolymorphic(), pointerAbstract()
 *
 * <b>Methods that can be overwritten:</b>
 * invokeOverwrite(), object(). enumeration(), pointer()
 *
 * <b>Implementation Detail:</b>
 *
 * Some methods in this class and derived class are called by
 * explicitly typing this->... . This is necessary to avoid compiler
 * errors like: "... does not depend on template parameters".
 *
 */
template <typename Derived>
class RecursiveMemberReflectorBase : public AbstractReflector<Derived>
{
	typedef AbstractReflector<Derived> Base;
public:

	/// The constructor.
	RecursiveMemberReflectorBase() :
		mCurrentMemberMeta(NULL) {}

public:

	/** @name Visiting methods that are called by the RecursiveMemberReflector
	 *        while reflecting the members
	 */
	//@{

	/**
	 * Is called if the member is an atomic type (int, float, etc).
	 * (Type A1)
	 *
	 * As for all the other visit-methods the parameter
	 * is a reference to the actual visited/reflected member.
	 *
	 * This method usually will be implemented in concrete derived classes.
	 */
	template<typename T>
	void atomic(T& member) {}

	/**
	 * Is called by the if the member is an enumeration.
	 * (Type A2)
	 *
	 * As for all the other visit-methods the parameter
	 * is a reference to the actual visited/reflected member.
	 *
	 * The default implementation casts the enum to an uint32 and calls
	 * atomic().
	 */
	template<typename T>
	void enumeration(T& member) {
		int32 val;
		if(Derived::isReadOnly::value) {
			val = static_cast<int32>(member);
			this->This()->atomic(val);
		} else {
			this->This()->atomic(val);
			member = static_cast<T>(val);
		}
	}

	/**
	 * Is called for each complex object.
	 * See documentation of RecursiveMemberReflectorBase class for detailed
	 * calling sequences.
	 * If you overwrite this method in a derived class you should call the
	 * method of the Base class to allow recursive reflection of the complex
	 * object's members.
	 */
	template<typename T>
	void object(T& member) {
		this->This()->invoke(member); // invoke us recursively on the object
	}

	/**
	 * Is called for each complex object or array, where the IsCollection<T>
	 * trait is true_type.
	 * The default implementation calls object().
	 */
	template<typename T>
	void collection(T& member) {
		this->This()->object(member); // default impl calls object()
	}


	template<typename T>
	void trackObject(T& member) {}

	/**
	 * Is called if the member is a pointer. (Type C)
	 * The default implementation detects the specific pointer
	 * type and calls pointerNormal(), pointerPolymorphic() or pointerAbstract().
	 */
	template<typename T>
	void pointer(T* &pointer) {
		// this will instantiate the SetReflectedAsPointer during startup
		// which will set ReflectedAsPointer<T,Derived>::value to true,
		// and mark that T was reflected as pointer by Derived at least once.
		Singleton<serialization::SetReflectedAsPointer<T,Derived>,singleton::EagerInstantiation,singleton::CreateStatic>::instance();

		// Selects the correct pointer ReflectPointerXXX struct using boost::mpl.
		typedef typename
		boost::mpl::eval_if<std::is_base_of<mira::Object, T>,
			boost::mpl::identity< ReflectPointerPolymorphic >,   // T is a polymorphic class derived from Object
		//else
			boost::mpl::eval_if<std::is_abstract<T>,
				boost::mpl::identity< ReflectPointerAbstract >,  // T is abstract
			//else
				boost::mpl::identity< ReflectPointerNormal >     // T is a "normal" class (maybe polymorphic but not derived from Object)
			>
		>::type type;

		type::invoke(*this->This(), pointer);
	}

	/**
	 * Is called if a reflected pointer is a "normal" pointer.
	 * As second parameter the typeId of the concrete objects class
	 * is passed.
	 * The default implementation recursively reflects the object,
	 * the pointer points to (if the pointer is not NULL).
	 */
	template<typename T>
	void pointerNormal(T* &pointer, int typeId) {
		if(pointer!=NULL) // delegate serialization of the actual object recursively
			this->This()->invokePointerObject(*pointer);
	}

	/**
	 * Is called if a reflected pointer is a polymorphic pointer
	 * to an object that is derived from mira::Object.
	 * As second parameter the typeId of the concrete objects class
	 * is passed (may be -1, if pointer is a NULL pointer)
	 * The default implementation recursively reflects the object,
	 * the pointer points to (if the pointer is not NULL).
	 */
	template<typename T>
	void pointerPolymorphic(T* &pointer, int typeId) {
		if(pointer!=NULL) { // delegate serialization of the actual object recursively

			// obtain a special polymorph pointer serializer for the derived type
			typedef serialization::PolymorphicPointerReflectorRegistrar<Derived> OurRegistrar;

			typename OurRegistrar::ReflectorPtr r  =
					 OurRegistrar::instance().getPolymorphicPointerReflector(typeId);

			if(r==NULL) {
				const ReflectMemberMeta& meta = getCurrentMemberMeta();
				const Class& c = pointer->getClass();
				MIRA_THROW(XIO, "Cannot reflect the polymorphic member pointer '"
				          << meta.name << "'. The class '" << c.getIdentifier()
				          << "' was not registered with this serializer '"
				          << typeName<Derived>());
			}

			// deserialize the content recursively
			r->invoke(*this->This(), serialization::void_upcast(pointer));
		}
	}

	/**
	 * Is called if a reflected pointer is a pointer to an abstract class.
	 * As second parameter the typeId of the objects class is passed.
	 * The default implementation recursively reflects the object,
	 * the pointer points to (if the pointer is not NULL).
	 */
	template<typename T>
	void pointerAbstract(T* &pointer, int typeId) {
		if(pointer!=NULL) // delegate serialization of the actual object recursively
			this->This()->invokePointerObject(*pointer);
	}

	//@}

public:

	/**
	 * Registers a new polymorphic class at the PolymorphPointerReflector.
	 * Each RecursiveMemberReflector only is able to process pointers to
	 * polymorphic classes that are registered using this method.
	 *
	 * The MIRA_CLASS_SERIALIZATION macro provides a simple mechanism to register
	 * a class in all known MemberReflectors and calls their registerClass()
	 * method. If you implement a new MemberReflector you should modify the
	 * MIRA_CLASS_SERIALIZATION macro and add your reflector there.
	 *
	 * @see MIRA_CLASS_SERIALIZATION
	 */
	template<typename T>
	static void registerClass()
	{
		typedef serialization::PolymorphicPointerReflectorRegistrar<Derived> OurRegistrar;

		if(!OurRegistrar::isDestroyed())
			OurRegistrar::instance().template registerClass<T>();
	}

	/**
	 * Unregisters the class.
	 *
	 * The MIRA_CLASS_SERIALIZATION macro provides a simple mechanism to register
	 * a class in all known MemberReflectors and calls their registerClass()
	 * method. If you implement a new MemberReflector you should modify the
	 * MIRA_CLASS_SERIALIZATION macro and add your reflector there.
	 *
	 * @see MIRA_CLASS_SERIALIZATION
	 */
	template<typename T>
	static void unregisterClass()
	{
		typedef serialization::PolymorphicPointerReflectorRegistrar<Derived> OurRegistrar;
		if(!OurRegistrar::isDestroyed())
			OurRegistrar::instance().template unregisterClass<T>();
	}
public:

	/**
	 * Returns the meta-information of the current member that is reflected.
	 */
	const ReflectMemberMeta& getCurrentMemberMeta() const
	{
		assert(mCurrentMemberMeta!=NULL);
		return *mCurrentMemberMeta;
	}

	/**
	 * Returns the full human readable object id / name of the current member
	 * being reflected. This method must not be called if the Reflector has set
	 * useHumanReadableIDs to false. If it is called anyway we will assert here.
	 */
	const std::string& getCurrentMemberFullID() const
	{
		assert(this->usesHumanReadableIDs());
		return mCurrentMemberFullID;
	}

public:

	/**
	 * For internal use only:
	 * Returns true, if the type T is ever reflected as pointer within the
	 * current translation unit (C file).
	 */
	template <typename T>
	static bool isReflectedAsPointer() {
		return LazySingleton<serialization::ReflectedAsPointer<T,Derived>>::instance().value;
	}

protected:

	/// tracks the given object (if pointer tracking is enabled for type T)
	template<typename T>
	void invokeTrackObject(T& member)
	{
		if(IsObjectTrackable<T>::value)
			this->This()->trackObject(member);
	}


	// The following structs implement different invoke methods for each of
	// the member types (A1-3, B1-2, C). The member types are detected in
	// the invoke method above which delegates to the invoke method
	// of the correct struct below.

	struct ReflectUnknown {
		template<typename T>
		static void invoke(Derived& r, T& member) {
			static_assert(sizeof(T)==0, "Trying to reflect unknown type.");
		}
	};

	/// Type A1: for atomic members (float,int,etc.)
	struct ReflectAtomic {
		template<typename T>
		static void invoke(Derived& r, T& member) {
			r.invokeTrackObject(member);
			r.atomic(member);
		}
	};

	/// Type A2: for enums
	struct ReflectEnumeration {
		template<typename T>
		static void invoke(Derived& r, T& member) {
			r.invokeTrackObject(member);
			r.enumeration(member);
		}
	};

	/// Type A3: for arrays
	struct ReflectArray {
		template<typename T>
		static void invoke(Derived& r, T& member) {
			r.invokeTrackObject(member);
			std::size_t size = sizeof(member) /
					(static_cast<const char *>(static_cast<const void *>(&member[1]))
					- static_cast<const char *>(static_cast<const void *>(&member[0])) );

			typedef typename std::remove_extent<T>::type ItemType;
			serialization::Array<ItemType> array(member, size);
			//r.collection(array);
			ReflectCollection::invoke(r, array);
		}
	};

	/// Type B1/B2: for complex types
	struct ReflectComplex {
		template<typename T>
		static void invoke(Derived& r, T& member) {
			r.invokeTrackObject(member);
			typedef typename std::remove_cv<T>::type Type;
			static_assert(!IsCollection<Type>::value,
			              "Trying to call ReflectComplex::invoke<T>() with IsCollection<T> == true - something is wrong.");
			if(IsTransparentSerializableHelper<Type,Derived>::value)
				r.invoke(member);
			else
				r.object(member);
		}
	};

	/// Type B?c: for collection types
	struct ReflectCollection {
		template<typename T>
		static void invoke(Derived& r, T& member) {
			r.invokeTrackObject(member);
			typedef typename std::remove_cv<T>::type Type;
			typedef IsTransparentSerializableHelper<Type,Derived> isTransparent;
			static_assert(!isTransparent::value,
			              "Trying to call ReflectCollection::invoke<T>() with IsTransparentSerializable<T> == true - something is wrong.");
			r.collection(member);
		}
	};

	/// Type C: for members that are pointers
	struct ReflectPointer {
		template<typename T>
		static void invoke(Derived& r, T& member) {
			// clear the inline flag
			r.pointer(member);
		}
	};

	// The following ReflectPointerXXX structs are used by our pointer() method

	/// Type C1: for members that are pointers to normal classes
	struct ReflectPointerNormal {
		template<typename T>
		static void invoke(Derived& r, T* &pointer) {
			r.pointerNormal(pointer, typeId<T>());
		}
	};


	/// Type C2: for members that are pointers to polymorphic classes derived from mira::Object
	struct ReflectPointerPolymorphic {
		template<typename T>
		static void invoke(Derived& r, T* &pointer) {
			int typeId = -1;

			if(pointer!=NULL) {
				// we have an object, so we can obtain its class information
				const Class& c = pointer->getClass();
				typeId = c.getTypeId();
			}

			// call pointerPolymorphic()
			r.pointerPolymorphic(pointer, typeId);
		}
	};

	/**
	 * Type C3: for members that are pointers to abstract classes not derived
	 * from mira::Object.
	 */
	struct ReflectPointerAbstract {
		template<typename T>
		static void invoke(Derived& r, T* &pointer) {
			r.pointerAbstract(pointer, typeId<T>());
		}
	};


	/**
	 * Detect the members type (A1,A2,A3,B1,B2,C) and choose the appropriate
	 * struct ReflectAtomic, etc. that handles that type.
	 */
	template<typename T>
	struct ChooseReflect {

		typedef typename std::remove_cv<T>::type Type;
		typedef typename
			boost::mpl::eval_if<std::is_pointer<Type>,
				boost::mpl::identity<ReflectPointer>, // T is a pointer (Type C)
			//else
				boost::mpl::eval_if<IsAtomicSerializable<Type>,
					boost::mpl::identity<ReflectAtomic>,       // atomic (Type A1)
				// else // T is not atomic
					boost::mpl::eval_if<std::is_class<Type>,
						boost::mpl::eval_if<IsCollection<Type>,
							boost::mpl::identity<ReflectCollection>,  // T is a collection (Type B?c)
						// else
							boost::mpl::identity<ReflectComplex>>,    // T is a complex class (Type B1/B2)
					//else // T is no pointer, no class --> must be atomic or enum
						boost::mpl::eval_if<std::is_enum<Type>,
							boost::mpl::identity<ReflectEnumeration>,     // enum (Type A2)
						// else
							boost::mpl::eval_if<std::is_array<Type>,
								boost::mpl::identity<ReflectArray>,       // array (Type A3)
							// ELSE unknown
								boost::mpl::identity<ReflectUnknown>
						>
					>
				>
			>
		>::type type;
	};

public:

	/**
	 * Is called to invoke this Reflector on the member with the specified
	 * meta information. This method delegates the call to the most
	 * derived invokeMemberOverwrite() method that may be overwritten in
	 * subclasses.
	 * @note Usually, the RecursiveMemberReflector should be invoked using
	 *       this invokeMember() method instead of the invoke() method of
	 *       the ReflectorInterface base class. The invoke() method is called
	 *       internally by the RecursiveMemberReflector.
	 */
	template<typename T>
	void invokeMember(T& member, const ReflectMemberMeta& meta)
	{
		this->This()->invokeMemberOverwrite(member, meta);
	}

	/**
	 * The actual invokeMember implementation that is called from invokeMember().
	 * This method may be overwritten in derived classes to add additional
	 * functionality, however, you should always call this implementation of
	 * the base class.
	 */
	template<typename T>
	void invokeMemberOverwrite(T& member, const ReflectMemberMeta& meta)
	{
		// store previous meta (it is restored at the end of this method
		// and therefore implements a stack, where mCurrentMemberMeta is
		// the top element in the stack)
		const ReflectMemberMeta* prevMeta = mCurrentMemberMeta;

		// do the same to create a stack of the full human readable object
		// id (however, only do it, if our final reflector supports human
		// readable id's, otherwise skip this step since it hurts the
		// performance
		std::string prevFullID;
		if(this->usesHumanReadableIDs()) {
			prevFullID = mCurrentMemberFullID;

			// "push" new full id
			if(mCurrentMemberFullID.empty()) {
				if(meta.id==NULL)
					MIRA_THROW(XLogical, "Top-level member cannot be 'inline'. "
						"It must have an ID != NULL!");
				mCurrentMemberFullID=meta.id;
			} else if(meta.id!=NULL)
				mCurrentMemberFullID = mCurrentMemberFullID + "." + meta.id;
		}

		// "push" current meta (but only if member is not transparent, which is
		// indicated by an id of NULL)
		if(meta.id != NULL)
			mCurrentMemberMeta = &meta;

		// Choose the type of the member and select the
		// correct ReflectXYZ struct for that type at compile type.
		// Then call the invoke method of that struct.
		// (The different structures and their invoke methods are defined
		// above)
		try
		{
			ChooseReflect<T>::type::invoke(*this->This(), member);
		}
		catch(...)
		{
			// "pop" our id
			if(this->usesHumanReadableIDs())
				mCurrentMemberFullID=prevFullID;
			// "pop" our meta data stack.
			mCurrentMemberMeta = prevMeta;
			// and throw
			throw;
		}

		// "pop" our id
		if(this->usesHumanReadableIDs())
			mCurrentMemberFullID=prevFullID;
		// "pop" our meta data stack.
		mCurrentMemberMeta = prevMeta;
	}

	/**
	 * Is called to reflect objects of pointers. They will use the
	 * same meta as the pointer
	 */
	template<typename T>
	void invokePointerObject(T& member)
	{
		ChooseReflect<T>::type::invoke(*this->This(), member);
	}


	/**
	 * Delegates to invokeMember() and rethrows any occurring
	 * XMemberNotFound exception as XIO exception.
	 */
	template<typename T>
	void invokeMemberWithoutDefault(T& member, const ReflectMemberMeta& meta)
	{
		try {
			this->invokeMember(member,meta);
		} catch (XMemberNotFound& ex) {
			throw XMemberNotFound_NoDefault(ex);
			// transform exception to avoid catching it
			// in upper levels, where it might be shadowed
			// if a default value is given there (see #685)
		}
	}

	/**
	 * Delegates to invokeMember() and handles any occurring
	 * XMemberNotFound exception by setting the member to the given
	 * default value.
	 */
	template<typename T, typename U>
	void invokeMemberWithDefault(T& member, const ReflectMemberMeta& meta,
	                             const U& defaultValue)
	{
		try {
			this->invokeMember(member,meta);
		} catch (XMemberNotFound_NoDefault&) {
			throw; // must not handle this
		} catch (XMemberNotFound&) {
			if(!Derived::isReadOnly::value) {
				MIRA_LOG(NOTICE) << "No value given for '" << meta.name << "', "
					"using the default value instead.";

				// #############################################################
				// If you get a compiler error here, your default value does not
				// match to your member type and cannot be casted. Make sure
				// that the data type of your default value can be casted to
				// the data type of your member!
				// #############################################################
				member=defaultValue;
			}
		}
	}

	/**
	 * Delegates to invokeMember() and handles any occurring
	 * XMemberNotFound exception by ignoring the exception and hence
	 * the missing member, as indicated by IgnoreMissing as default value.
	 * The value of the member remains unchanged.
	 */
	template<typename T>
	void invokeMemberWithDefault(T& member, const ReflectMemberMeta& meta,
	                             const serialization::IgnoreMissing& defaultValue)
	{
		try {
			this->invokeMember(member,meta);
		} catch (XMemberNotFound_NoDefault&) {
			throw; // must not handle this
		} catch (XMemberNotFound&) {
			// IGNORE
		}
	}


	//@}

private:

	const ReflectMemberMeta* mCurrentMemberMeta;
	std::string mCurrentMemberFullID;

};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * The RecursiveMemberReflector extents the RecursiveMemberReflectorBase class
 * and implements the member() and property() methods of the
 * ReflectorInterface class by calling the appropriate methods of
 * RecursiveMemberReflectorBase.
 *
 * <b>Implements from derived classes:</b>
 * member(), property()
 *
 * <b>Methods that should be implemented:</b>
 * beginObject(), endObject(), atomic(), pointerNormal(), pointerPolymorphic(),
 * pointerAbstract()
 *
 * <b>Methods that can be overwritten:</b>
 * invokeOverwrite(), enumeration(), array(), pointer()
 */
template <typename Derived>
class RecursiveMemberReflector : public RecursiveMemberReflectorBase<Derived>
{
public:

	// implement our supported visiting methods of the ReflectorInterface
	// note: the documentation of the following methods is inherited from
	// ReflectorInterface by Doxygen

	template<typename T>
	void member(const char* name, T& member, const char* comment,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, name, comment) );
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, name, comment) );
	}

	template<typename T>
	void member(const char* name, const std::string& id, T& member,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, id.c_str(),
			                                 comment) );
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, id.c_str(),
			                                 comment) );
	}

	template<typename T>
	void member(const char* name, const T& member, Setter<T> setter,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithoutDefault(a, ReflectMemberMeta(name, name, comment) );
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void member(const char* name, Getter<T> getter, Setter<T> setter,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithoutDefault(a, ReflectMemberMeta(name, name, comment) );
		this->This()->popObjectTrackingStore();
	}

	template<typename T, typename U>
	void member(const char* name, T& member, const char* comment,
	            const U& defaultValue, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMemberWithDefault(member, ReflectMemberMeta(name, name, comment),
			                              defaultValue);
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMemberWithDefault(member, ReflectMemberMeta(name, name, comment),
			                              defaultValue);
	}

	template<typename T, typename U>
	void member(const char* name, const T& member, Setter<T> setter,
	            const char* comment, const U& defaultValue,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithDefault(a, ReflectMemberMeta(name, name, comment),
		                              defaultValue );
		this->This()->popObjectTrackingStore();
	}

	template<typename T, typename U>
	void member(const char* name, Getter<T> getter, Setter<T> setter,
	            const char* comment, const U& defaultValue,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithDefault(a, ReflectMemberMeta(name, name, comment),
		                              defaultValue );
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void property(const char* name, T& member, const char* comment,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, name, comment) );
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, name, comment) );
	}

	template<typename T>
	void property(const char* name, const std::string& id, T& member,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, id.c_str(), comment) );
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMemberWithoutDefault(member, ReflectMemberMeta(name, id.c_str(), comment) );
	}

	template<typename T>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithoutDefault(a, ReflectMemberMeta(name, name, comment) );
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithoutDefault(a, ReflectMemberMeta(name, name, comment) );
		this->This()->popObjectTrackingStore();
	}

	template<typename T, typename U>
	void property(const char* name, T& member, const char* comment,
	              const U& defaultValue, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMemberWithDefault(member, ReflectMemberMeta(name, name, comment),
				                          defaultValue);
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMemberWithDefault(member, ReflectMemberMeta(name, name, comment),
				                          defaultValue);
	}

	template<typename T, typename U>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithDefault(a, ReflectMemberMeta(name, name, comment),
		                              defaultValue );
		this->This()->popObjectTrackingStore();
	}

	template<typename T, typename U>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMemberWithDefault(a, ReflectMemberMeta(name, name, comment),
		                              defaultValue );
		this->This()->popObjectTrackingStore();
	}

	/**
	 * Delegates the serialization directly to the specified member, without
	 * creating a separate compound for the current object.
	 * This method usually is used together with IsTransparentSerializable.
	 * Note: only one call of delegate is allowed within one reflect method.
	 */
	template<typename T>
	void delegate(T& member, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMember(member, ReflectMemberMeta(NULL, NULL, "") );
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMember(member, ReflectMemberMeta(NULL, NULL, "") );
	}

	/**
	 * Delegates the serialization directly to the specified member, without
	 * creating a separate compound for the current object.
	 * This method usually is used together with IsTransparentSerializable.
	 * Note: only one call of delegate is allowed within one reflect method.
	 */
	template<typename T>
	void delegate(const T& member, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMember(a, ReflectMemberMeta(NULL, NULL, "") );
		this->This()->popObjectTrackingStore();
	}

	/**
	 * Delegates the serialization directly to the specified member, without
	 * creating a separate compound for the current object.
	 * This method usually is used together with IsTransparentSerializable.
	 * Note: only one call of delegate is allowed within one reflect method.
	 */
	template<typename T>
	void delegate(Getter<T> getter, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMember(a, ReflectMemberMeta(NULL, NULL, "") );
		this->This()->popObjectTrackingStore();
	}

};

///////////////////////////////////////////////////////////////////////////////

}

#endif
