/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSASignature.h
 *    A class for signing and verifing a message signature based
 *    on a RSA encryption.
 *
 * @author Christian Martin
 * @date   2011/01/10
 */

#ifndef _MIRA_RSASIGNATURE_H_
#define _MIRA_RSASIGNATURE_H_

#include <sstream>

#include <platform/Types.h>
#include <error/Exceptions.h>
#include <serialization/SplitReflect.h>
#include <serialization/IsTransparentSerializable.h>

#include <security/RSAKey.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A class for signing and verifying messages using a RSA key pair.
 *
 * @ingroup SecurityModule
 */
class MIRA_BASE_EXPORT RSASignature
{
public:
	/// The digest type.
	enum DigestType {
		DIGEST_MD5,
		DIGEST_SHA1,
		DIGEST_RIPEMD160
	};

public:
	/** @name Constuctors and destructor */
	//@{

	/// Default constructor.
	RSASignature();

	/// The copy constructor.
	RSASignature(const RSASignature& signatur);

	/// The destructor.
	virtual ~RSASignature();

	//@}

public:
	/** @name Reflect operation */
	//@{

	MIRA_SPLIT_REFLECT_MEMBER

	/// Reflect
	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		std::stringstream ss;
		ss << (*this);
		std::string str = ss.str();
		r.delegate(str);
	}

	template<typename Reflector>
	void reflectWrite(Reflector& r)
	{
		std::string str;
		r.delegate(str);
		std::stringstream ss(str);
		ss >> (*this);
	}

	//@}

public:
	/** @name Operators */
	//@{

	/// The '=' operator
	RSASignature& operator=(const RSASignature& signature);

	//@}

public:
	/** @name Message signature information */
	//@{

	/// Return the size of the RSA signature
	size_t getSize() const { return(mSize); }

	/// Return the signature in a binary format.
	const uint8* getSignature() const { return(mData); }

	//@}

public:
	/** @name Sign and verify a message */
	//@{

	/**
	 * Sign a message based on the private RSA key of the sender.
	 * @param[in] iPrivateKey The private RSA key of the sender.
	 * @param[in] iDigestType The digest algorithm.
	 * @param[in] iMsg        The message itself.
	 * @param[in] iMsgLen     The size of the message.
	 * @return The RSA signature of the message.
	 */
	static RSASignature signMessage(const RSAKey& iPrivateKey,
	                                DigestType iDigestType,
	                                const char* iMsg,
	                                size_t iMsgLen);

	/**
	 * Verify a message signature based on the public RSA of the sender.
	 * @param[in] iPublicKey  The public RSA key of the sender.
	 * @param[in] iDigestType The digest algorithm.
	 * @param[in] iMsg        The message itself.
	 * @param[in] iMsgLen     The size of the message.
	 * @param[in] iSignature  The signature of the message.
	 * @return True, if the message was verified successfully.
	 */
	static bool verifyMessage(const RSAKey& iPublicKey,
	                          DigestType iDigestType,
	                          const char* iMsg,
	                          size_t iMsgLen,
	                          const RSASignature& iSignature);

	//@}

private:
	friend MIRA_BASE_EXPORT std::ostream& operator<<(std::ostream& stream,
	                                                 const RSASignature& signature);
	friend MIRA_BASE_EXPORT std::istream& operator>>(std::istream& stream, 
	                                           	     RSASignature& signature);

private:
	size_t mSize;
	uint8* mData;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * The '<<' operator for RSASignature
 *
 * Format:  <tt>HexData</tt>
 *
 * @ingroup SecurityModule
 */
MIRA_BASE_EXPORT std::ostream& operator<<(std::ostream& stream,
                                          const RSASignature& signature);

/**
 * The '>>' operator for RSASignature
 *
 * @ingroup SecurityModule
 */
MIRA_BASE_EXPORT std::istream& operator>>(std::istream& stream,
                                          RSASignature& signature);

///////////////////////////////////////////////////////////////////////////////

template<typename SerializerTag>
class IsTransparentSerializable<RSASignature,SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

} // namespaces

#endif
