/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DistanceLUT.h
 *    This header provides a class for fast looking up the upper and lower
 *    bounds of the euclidean distances between a point and cells arranged in
 *    a regular grid.
 *
 * @author Erik Einhorn, Jens Kessler (Documentation)
 * @date   2009/09/28
 */

#ifndef _MIRA_DISTANCELUT_H_
#define _MIRA_DISTANCELUT_H_

#ifndef Q_MOC_RUN
#include <boost/thread/mutex.hpp>
#endif

#include <utils/Singleton.h>
#include <geometry/Point.h>
#include <geometry/Rect.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////


/**
 * @ingroup GeometryModule
 *
 * This class creates a look up table (LUT) to calculate the minimum and
 * maximum Euclidean distance between a point and cells that are arranged in
 * a regular grid. The main goal of this structure is to fasten operations like
 * occupancy mapping, where the distance between many occupancy grid maps cells
 * and the origin of a range sensor need to be computed.
 *
 * The LUT works as follows:
 * It maintains an array of cells, that are arranged around a central point p.
 * Each cell c_i has a size of 1x1 (in non-dimensional units). Moreover, for
 * each cell the min and max distance to the central point is stored.
 *
 * \code
 *     Region cells
 * |---|---|---|---|---|
 * |   |   |   |   |   |
 * |---|---|---|---|---o<--- max distance
 * |   |   |   |   |c_i|
 * |---|---|-(1,1)-o<--+---- min distance
 * |   |   | p |   |   |
 * |---|-(0,0)-|---|---|
 * |   |   |   |   |   |
 * |---|---|---|---|---|
 * |   |   |   |   |   |
 * |---|---|---|---|---|
 * <------------------->
 *        width
 * \endcode
 *
 * The central point p is located within the central cell of the array. To
 * increase the precision, the central cell is subdivided into bins:
 * \code
 *   central cell
 *     |-|-|-| ^
 *     |-|-|-| | subdivisions
 *     |-|-|-| v
 * \endcode
 *
 * Both the size (width) of the cell array as well as the precision
 * (subdivisions) can be specified in the createLUT() method, which
 * pre-computes the LUT.
 *
 * For a distance look-up the getRegion() method needs to be called. It returns
 * a portion of the LUT containing all cells within a given region. Moreover,
 * the location of the central point within the central cell must be specified.
 * Since the dimension of each cell is 1x1, the point must be located within
 * the interval [0,0]-[1,1]. Based on the location of the point the corresponding
 * subdivision bin is selected and the region of the appropriate LUT is returned.
 *
 * The region can be iterated using an iterator that is provided by the region.
 * Each cell within the region finally contains the min and max distance to the
 * specified central point.
 *
 * @see @ref GeometryPage
 */
class MIRA_BASE_EXPORT DistanceLUT : public LazySingleton<DistanceLUT>
{
public:
	/**
	 * Creates a DistanceLUT with a default width of 1000 and 5 subdivisions.
	 * These default settings can be changed via createLUT().
	 */
	DistanceLUT();

	/// The destructor.
	~DistanceLUT();

public:
	/** A struct to store the minimal and maximal distance of the cell towards
	 *  the center point
	 */
	struct CellDist {
		float minDist, maxDist;
	};

	typedef const CellDist* iterator; ///< the class's iterator type

#if BOOST_VERSION < 105300
	// If the shared_ptr type is not an array it calls delete instead of delete[] => memory leak
	typedef boost::shared_ptr<CellDist> CellDistArrayPtr;
#else
	typedef boost::shared_ptr<CellDist[]> CellDistArrayPtr;
#endif

	/**
	 * Provides a view on a portion of the Distance LUT and can be obtained
	 * by calling DistanceLUT::getRegion(). The region can be traversed using
	 * an iterator that provides access to each cell within the region.
	 * Finally, each cell stores the distance to a given central point, that
	 * was specified in the call to DistanceLUT::getRegion().
	 */
	class Region
	{
	protected:
		friend class DistanceLUT;

		/**
		 * The constructor of the region
		 * @param[in] data: the iterator of the LUT
		 * @param[in] stride: the length of one LUT line
		 * @param[in] region: the region boundaries in cells
		 */
		Region(CellDistArrayPtr array, CellDist* data, int stride, const Rect2i& region) :
			mArrayPtr(array), mData(data), mStride(stride), mRegion(region) {}

	public:

		/// Returns a Rect containing the boundaries of this region.
		const Rect2i& getRect() const { return mRegion; }

		/// Returns the minimum x-boundary
		int getXmin() const { return mRegion.minCorner.x(); }

		/// Returns the maximum x-boundary
		int getXmax() const { return mRegion.maxCorner.x(); }

		/// Returns the minimum y-boundary
		int getYmin() const { return mRegion.minCorner.y(); }

		/// Returns the maximum y-boundary
		int getYmax() const { return mRegion.maxCorner.y(); }

		/// Returns the region's width
		int getWidth() const { return mRegion.width(); }

		/// Returns the region's height
		int getHeight() const { return mRegion.height(); }

	public:

		/**
		 * Returns an iterator to the given cell.
		 * @param[in] x: the x coordinate of the region point
		 * @param[in] y: the y coordinate of the region point
		 */
		iterator getIterator(int x, int y) const {
			return mData + y*mStride + x;
		}

		/**
		 * Returns line iterator starting in the specified line
		 * @param[in] y: the number of the specified line
		 */
		iterator begin(int y) const {
			return mData + y*mStride + getXmin();
		}

		/**
		 * Returns end iterator for the specified line
		 * @param[in] y: the number of the specified line
		 */
		iterator end(int y) const {
			return mData + y*mStride + getXmax()+1;
		}

	private:
		/// shared ptr to underlying array data, in order to keep up the ref count
		CellDistArrayPtr mArrayPtr;
		/// pointer to the LUT
		CellDist* mData;
		/// the length of one LUT line
		int mStride;
		/// the region boundaries in cells
		const Rect2i mRegion;
	};

	/**
	 * Can be called to ensure, that the specified region in the LUT is
	 * available and valid. If the region is larger than the LUT that has been
	 * generated so far, the LUT will be reallocated and recomputed. This
	 * method is called within getRegion() automatically, but may be called
	 * beforehand to ensure, that the time consuming process of recomputing
	 * the LUT is completed.
	 *
	 * @param[in] region : the region where the LUT should be calculated
	 */
	void ensureRegion(const Rect2i& region);

	/**
	 * This function returns a portion of the LUT with cells containing the
	 * min and max distance to the given point. The point d lies between [0,0]
	 * and [1,1]. The distances of the resulting LUT are returned relative to
	 * that point.
	 * @param[in] d : the point within central cell. The range must be located
	 *                between [0,0]...[1,1].
	 * @param[in] region : the region where the LUT should be calculated
	 */
	Region getRegion(const Point2f& d, const Rect2i& region);

private:

	/**
	 * Constructs the LUT and sets private members. The LUT is NOT valid, but
	 * only all needed memory is reserved.
	 * @param[in] width: the width and height of a quadratic LUT, must be >0
	 * @param[in] subdivisions: the number of subdivisions per cell, must be >0
	 */
	void createLUT(int width, int subdivisions);

	/// Frees the memory allocated by the LUT.
	void freeLUT();

	/**Computes the actual values of the LUT (min and max distances from the
	 * center). Note that the parameters should correspond to @ref createLUT.
	 * @param[in] width: the width and height of a quadratic LUT
	 * @param[in] subdivisions: the number of subdivisions per cell
	 */
	void computeLUT(int width, int subdivisions);

private:
	/// Width of the LUT: is always an odd number >=1
	int mWidth;
	/// number of subdivision for more position accuracy (is always an odd number >=1 )
	int mSubdivisions;
	/// number of layers (=subdivisions*subdivisions)
	int mLayers;
	/// number of cells per layer/subdivision
	int mSize;
	/// the central LUT
	std::vector<CellDistArrayPtr> mLUT;

	boost::mutex mMutex;
};

///////////////////////////////////////////////////////////////////////////////
}

#endif
