/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Package.C
 *    Description.
 *
 * @author Ronny Stricker
 * @date   2011/08/31
 */
#include "core/Package.h"
#include "core/PackageGroup.h"
#include "core/Database.h"
#include "core/Repository.h"

#include <serialization/Serialization.h>
#include <utils/ToString.h>

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/split.hpp>

#include <fstream>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Package::Package() : mDevelState(EXPERIMENTAL)
{
	mType = SOURCE;
}

Package::Package( std::string const& name ) : PackageGroup( name ),
		mDevelState(EXPERIMENTAL)
{
	mType = SOURCE;
}

Package::~Package()
{
	foreach( Package* package, mDependencies ) {
		delete package;
	}
}

void Package::merge( PackageGroup* other )
{
	assert( other->getClass().getIdentifier() == Package::CLASS().getIdentifier() );
	Package* package = dynamic_cast<Package*>( other );
	foreach( std::string const& repo, package->mRepos ) {
		mRepos.insert( repo );
	}
	mCurrentRepo = package->mCurrentRepo;
	mLocalPath = package->mLocalPath;
}

Package* Package::createCopy() const
{
	Package* tReturn = dynamic_cast<Package*>( PackageGroup::createCopy() );
	assert( tReturn );

	*tReturn = *this;

	return tReturn;
}

Package* Package::createDependency() const
{
	Package* tReturn = dynamic_cast<Package*>( PackageGroup::createDependency() );
	assert( tReturn );

	*tReturn = *this;

	return tReturn;
}

bool Package::operator<(const Package& other) const
{
	if ( mName != other.mName )
		return mName < other.mName;
	else {
		if ( mVersion != other.mVersion )
			return mVersion < other.mVersion;
		else {
			if ( mType != other.mType ) {
				if ( mDevelState < other.mDevelState )
					return mType < other.mType;
			}
		}
	}
	return false;
}

Package& Package::operator=(Package const& other)
{
	mRepos = other.mRepos;
	mCurrentRepo = other.mCurrentRepo;
	mRepoSubPath = other.mRepoSubPath;
	mVersion = other.mVersion;
	mDevelState = other.mDevelState;

	foreach( Package* dep, mDependencies )
		delete dep;
	mDependencies.clear();
	foreach( Package* dep, other.mDependencies )
		mDependencies.push_back( dep->createCopy() );

	mTags = other.mTags;
	mAuthor = other.mAuthor;
	mChangeLog = other.mChangeLog;

	return *this;
}

bool Package::isIdentical( PackageGroup const* other, Type ignoredFlags ) const
{
	Package const* package = dynamic_cast<Package const*>( other );
	if ( !package )
		return false;
	return ( canBeGroupedWith( other )
		&& mRepoSubPath == package->mRepoSubPath
		&& mVersion == package->mVersion
		&& mDevelState == package->mDevelState
		&& (mType | ignoredFlags) == (package->mType | ignoredFlags )
		&& mDescription == package->mDescription );
}

bool sortByPriority(Database* db, string const& a, string const& b)
{
	RepositoryPtr repoA = db->getRepoFromUrl( a );
	RepositoryPtr repoB = db->getRepoFromUrl( b );
	if ( repoA && repoB ) {
		return repoA->priority > repoB->priority;
	}
	if ( repoA )
		return false;
	return true;
}

vector<string> Package::sortedRepos( Database* db ) const
{
	vector<std::string> tReturn;
	foreach( std::string const& tUrl, mRepos ) {
		tReturn.push_back( tUrl );
	}
	sort( tReturn.begin(), tReturn.end(), boost::bind(&sortByPriority, db, _1, _2) );
	return tReturn;
}

bool Package::isPrimary() const
{
	return mType == BINARY || mType == SOURCE;
}

void Package::selectStdRepo( Database* db )
{
	if ( mCurrentRepo.empty() && mRepos.size()>0 )
		mCurrentRepo = sortedRepos( db )[0];
	PackageGroup::selectStdRepo( db );
}

///////////////////////////////////////////////////////////////////////////////

Package::Version::Version() : mMajor(0), mMinor(0), mPatch(0), mBuildId(0)
{
}

Package::Version::Version( uint32 iMajor, uint32 iMinor, uint32 iPatch) :
	mMajor( iMajor ), mMinor( iMinor ), mPatch( iPatch ), mBuildId(0)
{
}

string Package::Version::str() const
{
	if (mBuildId > 0)
		return toString<uint32>( mMajor ) + "."
				+ toString<uint32>( mMinor ) + "."
				+ toString<uint32>( mPatch ) + "-"
				+ toString<uint32>( mBuildId );
	else
		return toString<uint32>( mMajor ) + "."
				+ toString<uint32>( mMinor ) + "."
				+ toString<uint32>( mPatch );
}

void Package::Version::fromString( string const& str )
{
	mMajor = 0;
	mMinor = 0;
	mPatch = 0;
	mBuildId = 0;
	string strCopy = boost::algorithm::trim_copy(str);
	if (strCopy.size() > 0) {
		vector<string> vec;
		boost::split(vec, strCopy, boost::is_any_of(".-"));
		try {
			if (vec.size() > 0)
				mMajor = mira::fromString<uint32>(vec[0]);
			if (vec.size() > 1)
				mMinor = mira::fromString<uint32>(vec[1]);
			if (vec.size() > 2)
				mPatch = mira::fromString<uint32>(vec[2]);
			if (vec.size() > 3)
				mBuildId = mira::fromString<uint32>(vec[3]);
		} catch(...) {}
	}
}

bool Package::Version::isBinaryCompatible( Version const& other ) const
{
	return (mMajor == other.mMajor) && (mMinor == other.mMinor);
}

bool Package::Version::isIndifferent() const
{
	return mMajor == 0 && mMinor == 0 && mPatch == 0;
}

bool Package::Version::operator==( Version const& other ) const
{
	return mMajor == other.mMajor
			&& mMinor == other.mMinor
			&& mPatch == other.mPatch
			&& mBuildId == other.mBuildId;
}

bool Package::Version::operator!=( Version const& other ) const
		{
	return !( *this == other );
}

bool Package::Version::operator<(Version const& other) const
{
	if ( mMajor != other.mMajor )
		return mMajor < other.mMajor;
	else {
		if ( mMinor != other.mMinor )
			return mMinor < other.mMinor;
		else {
			if ( mPatch != other.mPatch)
				return mPatch < other.mPatch;
			else
				return mBuildId < other.mBuildId;
		}
	}
}

bool Package::Version::operator>(Version const& other) const
{
	return other<*this;
}

bool Package::Version::operator>=(Version const& other) const
{
	return !( *this < other );
}

bool Package::Version::operator<=(Version const& other) const
{
	return !( *this > other );
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::Package,mira::PackageGroup);
