/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Repository.h
 *
 * @author Ronny Stricker
 * @date   2011/08/30
 */

#include <QtNetwork/QNetworkReply>

#include <utils/Path.h>
#include <utils/EnumToFlags.h>

#include <factory/Factory.h>
#include <serialization/XMLSerializer.h>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/vector>
#include <serialization/adapters/boost/shared_ptr.hpp>

#include <core/PromptProvider.h>

#ifndef _MIRA_REPOSITORY_H_
#define _MIRA_REPOSITORY_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class Package;

/**
 * @brief Abstract base class for different repsository types.
 * A repository is responsible for indexing, installing and
 * uninstalling web packages. Therefore, it should be able
 * to keep track of local packages to enable uninstalling.
 */
class Repository :
	public QObject,
	virtual public Object
 {
	MIRA_OBJECT( Repository )
	Q_OBJECT
public:
	/** @name Constructors, destructor and reflect */
	//@{

	/// Default constructor.
	Repository() : enabled(true), url("generic"), priority( 0 ), prompt( NULL ) {}

	//Repository( std::string const& iName, Url const& iUrl )
	//	: enabled(true), name( iName ), priority( 0 ), prompt(nullptr)
	//{
	//	setUrl( iUrl );
	//}

	/// The destructor.
	virtual ~Repository();

	/// The reflect method.
	template <class Reflector>
	void reflect ( Reflector& r ) {
		serialization::VersionType version = r.version(1, this);

		r.member("name", name,
		         "The name of the repository.");
		r.member("url", url, setter(&Repository::setUrl,this),
		         "The URL of the repository.");
		r.member("indexFile", indexFile,
		         "The URL of compressed index file for repository used for quick reindex");
		r.member("priority", priority,
		         "Priority value (higher values represent higher priority).");
		r.member("mountDirs", mountDirs,
		         "Found and parsed mountDirs in the repository (using branch.xml files).");

		r.member("description", description, "A short description of the repository.");
		r.member("enabled", enabled, "Enabled/disabled flag.");
	}

	//@}

public:
	/**
	 * @brief Set url of repository.
	 * Removes tailing slashes.
	 * Throws if url does not contain ://, so that the associated protocol
	 * type can not be determined.
	 */
	virtual void setUrl( Url const& iUrl );

	/**
	 * @brief Static factory function for different repository types.
	 * If iType==autoDetect, the function tries to guess the repository
	 * type be analyzing the given Url.
	 * @param[in] iName name of the repository
	 * @param[in] iDescription Description of the repository.
	 * @param[in] iUrl Url of the repository
	 * @param[in] iIndexFile index file for quick indexing of repository
	 * @param[in] priority the higher the value, the higher the priority.
	 * @param[in] iType can be "autoDetect" or any other valid MetaObject Tag
	 *                  with identifier RepoType (see ClassFactory MetaObject
	 *                  for further details).
	 */
	static boost::shared_ptr<Repository> create( std::string const& iName,
	                                             std::string const& iDescription,
	                                             Url const& iUrl,
	                                             Url const& iIndexFile,
	                                             uint32 priority, std::string const& iType = "autoDetect" );

	/// Create a new repository by reading the information from an URL.
	static boost::shared_ptr<Repository> createFromURL(const std::string& iURL);


	/**
	 * @brief Examine repository for existing packages and branch xml files.
	 * packageList and mountDirs are filled with this call.
	 * The function clears branch infos and tries to do a
	 * quick reindex using the index file. If no further reindex actions are
	 * required or desired the functions returns true.
	 * A thorough reindex will be triggered if the quick reindex fails.
	 * A process listener function can be used. It will be informed of
	 * (filename, current package count, total package count)
	 */
	virtual void examine( std::vector<Package*>& oPackages, bool thorough,
	                      boost::function<void (std::string const&,int,int)> = NULL );

	/**
	 * @brief Repository specific implementation to do a deep reindex.
	 */
	virtual void deepExamine( std::vector<Package*>& oPackages,
	                           boost::function<void (std::string const&,int,int)> = NULL ) = 0;

	/**
	 * @brief Parse a single mountdir.xml file.
	 * Url is only used to fill mountDirs map. The fileContent have
	 * to be given as the second argument.
	 */
	virtual void parseMountDirXml( Url const& url, std::string const& fileContent );

	/**
	 * @brief Return content of the file with the given url.
	 */
	virtual std::string catFile( Url const& url ) = 0;

	/**
	 * @brief Return true if the repository can handle the local package and if
	 * the url of the package is covered by the repository.
	 */
	virtual bool isResponsibleFor( mira::Path const& localPath ) = 0;

	/**
	 * @brief Return the relative url of the given local package.
	 * The url is relative to the root url of the repository.
	 */
	virtual mira::Path getRelativePathFor( mira::Path const& localPath ) = 0;

	/**
	 * @brief Return the desired local path of the package using information
	 * from the mountdir files.
	 * If no information can be found in the files, the functions returns a
	 * empty path.
	 * @param subUrl the suburl of the package (related to the repository root)
	 * @param[out] matchedUrl optional Url of associated mountdir
	 * @return relative local path obtained from branch.xml
	 */
	virtual Url getMountDirForPackage( Url const& subUrl, std::string* matchedUrl = NULL );

	/**
	 * @brief Perform installation of the given package
	 * @param[in] package to be installed
	 * @param[in] destPath local destination path
	 * @throws XIO if installation fails.
	 */
	virtual void install( Package const& package, mira::Path const& destPath ) = 0;

	/**
	 * @brief Uninstall the given local package.
	 * The functions removes all the files installed during installation().
	 * In addition it tries to remove the libraries and the manifest files
	 * from the corresponding mira path.
	 * @throws XIO if uninstall fails.
	 */
	virtual void uninstall( Package const& package ) = 0;

	/**
	 * @brief Remove the library, manifest and binary files generated during
	 * compilation of the given package.
	 * The function searches for files associated with the package (according
	 * to its local path) in the mira lib and bin directory and removes them.
	 * Note: this function does not work on windows...
	 * @param package The package for which the associated files should be removed.
	 */
	void removeGeneratedLibraryFiles( Package const& package );

	/**
	 * @brief remove any prefix from the url.
	 * prefix are fore example the http:// or ftp:// etc...
	 */
	static Url removePrefix( Url const& path );

	/**
	 * @brief Return file at given location in the given byteArray.
	 *
	 * This method is able to handle FTP, HTTP and HTTPS requests. If the
	 * request URL is password protected, this methods asks for the username
	 * and password.
	 */
	void getFile( Url const& url, QByteArray& byteArray );

public slots:
	void networkReplyFinished();
	void networkReplyError(QNetworkReply::NetworkError code);
	void networkReplySSLErrors(const QList<QSslError>& errors);

protected:

	/**
	 * @brief return true if the url is equal to "generic"
	 */
	bool isGeneric() const;

	/**
	 * @brief Create and prepare checkout directory.
	 * Recursively create given directory if it does not exist. And
	 * add CMakeLists.txt files from the root CMakeLists.txt to
	 * the given directory if they are missing.
	 */
	void prepareCheckoutDirectory(mira::Path dir, bool generateCMakeLists );

	/**
	 * @brief generate default CMakeLists.txt
	 * CMakeLists adds all existing sub directories.
	 */
	void generateCMakeListsTxt(mira::Path dir);

public:
	/// Is this repository enabled?
	bool enabled;

	/// The name of the repository.
	std::string name;

	/// A short description of the repository.
	std::string description;

	/// The URL of the repository.
	Url url;

	/// The URL of compressed index file for repository used for quick reindex
	Url indexFile;

	// List of packages
	//std::list<Url> packageList;

	/// Found and parsed mountDirs in the repository (using mountdir.xml files).
	std::map<std::string,std::string> mountDirs;

	/// Priority value (higher values represent higher priority).
	uint32 priority;

	/// Credentials used for login.
	Credential credential;

	/// Active prompt provided (needed to display error messages and to retrieve user input).
	PromptProvider* prompt;
};

/// A pointer to a repository.
typedef boost::shared_ptr<Repository> RepositoryPtr;

///////////////////////////////////////////////////////////////////////////////

}

#endif /* _MIRA_REPOSITORY_H_ */
