/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LogView.C
 *    Implementation of LogView.h.
 *
 * @author Tim Langner
 * @date   2011/06/28
 */

#include <views/LogView.h>

#include <QEvent>

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QTextBlock>
#include <QPushButton>

#include <QApplication>

#include <error/LogCustomizableFilter.h>
#include <error/LogSimpleFormatter.h>

#include <widgets/FlowLayout.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class LogRecordEvent : public QEvent
{
public:
	static const QEvent::Type EventType =  static_cast<QEvent::Type>(QEvent::User+2000);

	LogRecordEvent(const LogRecord& r) :
		QEvent(EventType),
		record(r)
	{}

	LogRecord record;
};

///////////////////////////////////////////////////////////////////////////////

void LogViewSink::consume(const LogRecord& record)
{
	// Consume the log message and forward to the GUI by posting a Qt event
	if (mView && mView->ui)
		QApplication::postEvent(mView->ui, new LogRecordEvent(record));
}

//////////////////////////////////////////////////////////////////////////////

LogView::LogView() :
	severity(TRACE), ui(NULL)
{
	LogViewSink s(this);
	sink = boost::dynamic_pointer_cast<LogViewSink>(MIRA_LOGGER.registerSink(s));
}

LogView::~LogView()
{
	if (sink)
		MIRA_LOGGER.unregisterSink(sink);
}

QWidget* LogView::createPartControl()
{
	ui = new UI(this);
	return ui;
}

//////////////////////////////////////////////////////////////////////////////

LogView::UI::UI(QWidget* parent) :
	QWidget(parent)
{
	view = (LogView*)(parent);
	QVBoxLayout* vLayout = new QVBoxLayout(this);
	vLayout->setContentsMargins(0,3,0,0);
	vLayout->setSpacing(0);
	//vLayout->setContentsMargins(0, -1, 0, 0);
	logWidget = new QTreeWidget(this);
	logWidget->setMinimumHeight(20);
	logWidget->setRootIsDecorated(false);
	logWidget->setEnabled(true);
	logWidget->setSortingEnabled(false);
	logWidget->setColumnCount(10);
	QStringList headers;
	headers << "Level" << "Time" << "Message" << "Line" << "File" << "NS"
	        << "Class" << "Function" << "ThreadID" << "Details";
	logWidget->setHeaderLabels(headers);
	logWidget->setColumnWidth(1,200);
	logWidget->setColumnWidth(2,500);
	logWidget->setColumnWidth(3,50);
	logWidget->setColumnWidth(5,50);
	logWidget->setColumnWidth(8,75);

	FlowLayout* flowLayout = new FlowLayout();
	QHBoxLayout* hLayout = new QHBoxLayout();
	hLayout->setContentsMargins(0,2,0,0);
	hLayout->setSpacing(2);

	QLabel* lbl = new QLabel("Namespace", this);
	hLayout->addWidget(lbl);
	nsEdit = new LineEditClear(this);
	nsEdit->setText(view->nsName.c_str());
	nsEdit->setMinimumWidth(25);
	connect(nsEdit, SIGNAL(editingFinished()), this, SLOT(changeFilter()));
	hLayout->addWidget(nsEdit);
	hLayout->addSpacing(3);

	lbl = new QLabel("Class", this);
	hLayout->addWidget(lbl);
	classEdit = new LineEditClear(this);
	classEdit->setText(view->className.c_str());
	classEdit->setMinimumWidth(25);
	connect(classEdit, SIGNAL(editingFinished()), this, SLOT(changeFilter()));
	hLayout->addWidget(classEdit);
	hLayout->addSpacing(3);

	lbl = new QLabel("Thread", this);
	hLayout->addWidget(lbl);
	threadEdit = new LineEditClear(this);
	threadEdit->setText(view->threadID.c_str());
	threadEdit->setMinimumWidth(25);
	connect(threadEdit, SIGNAL(editingFinished()), this, SLOT(changeFilter()));
	hLayout->addWidget(threadEdit);
	hLayout->addSpacing(3);

	flowLayout->addItem(hLayout,1);

	hLayout = new QHBoxLayout();
	hLayout->setSpacing(2);

	hLayout->addStretch();

	lbl = new QLabel("Severity", this);
	hLayout->addWidget(lbl);
	severityBox = new QComboBox(this);
	QStringList levels;
	levels << "CRITICAL" << "ERROR" << "WARNING"
	       << "NOTICE" << "DEBUG" << "TRACE";
	severityBox->addItems(levels);
	severityBox->setCurrentIndex((int)view->severity);
	connect(severityBox, SIGNAL(currentIndexChanged(int)),
	        this, SLOT(severityChanged(int)));
	hLayout->addWidget(severityBox);
	hLayout->addSpacing(3);

	lbl = new QLabel("Set level globally", this);
	QFont font = severityBox->font();
	lbl->setFont(QFont(font.family(), font.pointSize()));
	hLayout->addWidget(lbl);
	globalBox = new QCheckBox(this);
	globalBox->setChecked(false);
	globalBox->setToolTip("Set selected severity as application-wide log level");
	hLayout->addWidget(globalBox);
	connect(globalBox, SIGNAL(toggled(bool)),
		        this, SLOT(globalChanged(bool)));

	hLayout->addSpacing(3);
	clearButton = new QPushButton(this);
	clearButton->setFixedSize(32,32);
	clearButton->setIcon(QIcon(":/icons/ClearList.png"));
	clearButton->setToolTip("Clear history");
	hLayout->addWidget(clearButton);
	connect(clearButton, SIGNAL(clicked()), this, SLOT(clearClicked()));

	pauseButton = new QPushButton(this);
	pauseButton->setFixedSize(32,32);
	pauseButton->setIcon(QIcon(":/icons/PauseList.png"));
	pauseButton->setToolTip("Pause logging");
	pauseButton->setCheckable(true);
	hLayout->addWidget(pauseButton);


	flowLayout->addItem(hLayout);

	vLayout->addLayout(flowLayout);
	vLayout->addWidget(logWidget);
	changeFilter();
}

void LogView::UI::customEvent(QEvent* event)
{
	if (pauseButton && pauseButton->isChecked())
		return;

	if (event->type() == LogRecordEvent::EventType)
	{
		if (logWidget->topLevelItemCount() > 40)
			logWidget->takeTopLevelItem(0);

		LogRecordEvent* l = dynamic_cast<LogRecordEvent*>(event);
		QTreeWidgetItem* item = new QTreeWidgetItem(logWidget);
		item->setText(0, severityLevelStr[l->record.level].c_str());
		item->setText(1, toString(l->record.time.toLocal()).c_str());
		item->setText(2, l->record.message.c_str());
		item->setText(3, toString(l->record.line).c_str());
		item->setText(4, l->record.file.c_str());
		LogRecordFunctionInfo info = MIRA_LOGGER.parseFunction(l->record);
		item->setText(5, info.nameSpace.c_str());
		item->setText(6, info.className.c_str());
		item->setText(7, info.functionName.c_str());
		item->setText(8, toString(l->record.threadID).c_str());
		item->setText(9, l->record.details.c_str());

		QColor c = Qt::white;
		switch(l->record.level)
		{
			case CRITICAL :
			case ERROR    : c = QColor(0xCC,0x55,0x55, 0xFF); break;
			case WARNING  :	c = QColor(0xCC,0xCC,0x00, 0xFF); break;
			case NOTICE   :	c = QColor(0xFF,0xFF,0xFF, 0xFF); break;
			case DEBUG    : c = QColor(0xCC,0xCC,0xCC, 0xFF); break;
			case TRACE    : c = QColor(0x55,0xCC,0x55, 0xFF); break;
		}
		for(int i=0; i<logWidget->columnCount(); ++i)
			item->setBackgroundColor(i, c);
		logWidget->addTopLevelItem(item);
		logWidget->scrollToBottom();
	}
}

void LogView::UI::changeFilter()
{
	LogCustomizableFilter filter;

	std::string t(nsEdit->text().toLocal8Bit().data());
	view->nsName = t;
	if (!t.empty())
		filter << LogCustomizableFilter::Namespace<LogCustomizableFilter::EQUAL>(t);

	t = classEdit->text().toLocal8Bit().data();
	view->className = t;
	if (!t.empty())
		filter << LogCustomizableFilter::Class<LogCustomizableFilter::EQUAL>(t);

	t = threadEdit->text().toLocal8Bit().data();
	view->threadID = t;
	if (!t.empty())
		filter << LogCustomizableFilter::Thread<LogCustomizableFilter::EQUAL>(fromString<ThreadID>(t));

	if (severityBox->currentIndex() >= 0)
	{
		view->severity = (SeverityLevel)severityBox->currentIndex();
		filter << LogCustomizableFilter::Severity<LogCustomizableFilter::LESS_EQUAL>(view->severity);
	}
	view->sink->setFilter(filter);
}

void LogView::UI::severityChanged(int index)
{
	changeFilter();

	if (globalBox->isChecked())
		MIRA_LOGGER.setSeverityLevel((SeverityLevel)index);
}

void LogView::UI::globalChanged(bool global)
{
	if (global)
		MIRA_LOGGER.setSeverityLevel((SeverityLevel)severityBox->currentIndex());
}

void LogView::UI::clearClicked()
{
	logWidget->clear();
}

void pauseModeChanged(int state)
{
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::LogView, mira::ViewPart);
