/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LibrariesListView.C
 *    Implementation of LibrariesListView.h
 *
 * @author Christof Schröter
 * @date   2020/10/05
 */

#include <views/LibrariesListView.h>

#include <fw/Framework.h>

#include <QHeaderView>
#include <QMenu>

namespace mira {

const Duration cRPCTimeout = Duration::seconds(3);

///////////////////////////////////////////////////////////////////////////////

LibrariesListView::LibrariesListView() : mira::EditorPart()
{
	mServiceBox = NULL;
	mLibrariesView = NULL;
	mUpdateTimer = 0;

	mWaitForLocal = true;
	mShowLoadedOnly = false;
}

QWidget* LibrariesListView::createPartControl()
{
	QWidget* topWidget = new QWidget(this);
	QVBoxLayout* topLayout = new QVBoxLayout;
	topWidget->setLayout(topLayout);

	mServiceBox = new QComboBox(topWidget);
	topLayout->addWidget(mServiceBox);

	mLibrariesView = new QTreeWidget(topWidget);
	mLibrariesView->setColumnCount(4);
	mLibrariesView->setRootIsDecorated(false);
	mLibrariesView->setAlternatingRowColors(true);

	QStringList headers;
	headers << trUtf8("Library") << trUtf8("Version") << trUtf8("Manifest") << trUtf8("Loaded");
	mLibrariesView->setHeaderLabels(headers);

	mLibrariesView->header()->setResizeMode(0, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setResizeMode(1, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setResizeMode(2, QHeaderView::ResizeToContents);
	mLibrariesView->header()->setResizeMode(3, QHeaderView::ResizeToContents);

	mFilter = new TreeViewFilter(mLibrariesView, topWidget);
	mFilter->setFilterColumns(0, 1);
	mFilter->setFocus();

	topLayout->addWidget(mFilter);
	topLayout->addWidget(mLibrariesView);

	updateServicesList();
	updateList();

	connect(mServiceBox, SIGNAL(activated(int)),
	        this, SLOT(updateList()));

	QMenu* menu = getViewMenu();
	QAction* action = menu->addAction("Show loaded libraries only");
	action->setCheckable(true);
	action->setChecked(mShowLoadedOnly);
	connect(action, SIGNAL(toggled(bool)),
	        this, SLOT(onShowLoadedOnly(bool)));

	mUpdateTimer = startTimer(5000);

	return topWidget;
}

void LibrariesListView::timerEvent(QTimerEvent* event)
{
	if(event->timerId() == mUpdateTimer) {
		updateServicesList();
		updateList();
	}
}

void LibrariesListView::updateList()
{
	const std::string& service = mServices[mServiceBox->currentIndex()];
	updateLibrariesList(service);
}

void LibrariesListView::onShowLoadedOnly(bool loaded)
{
	mShowLoadedOnly = loaded;
	updateList();
}

void LibrariesListView::updateServicesList()
{
	const std::string localFramework = MIRA_FW.getGlobalID();
	const QString localEntry = QString::fromStdString("local  : " + localFramework);
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	std::list<std::string> services = rpcman.queryServicesForInterface("ILibraryRegistry");

	QString current = mServiceBox->currentText();

	mServiceBox->clear();
	mServices.clear();
	foreach(const std::string& s, services) {
		mServices.push_back(s);
		if (s == localFramework) {
			mServiceBox->addItem(localEntry);
			if (mWaitForLocal) { // prefer to show local framework when it is discovered
				current = localEntry;
				mWaitForLocal = false;
			}
		}
		else
			mServiceBox->addItem(QString::fromStdString("remote : " + s));
	}

	// keep selection, if still valid
//	mServiceBox->setCurrentText(current); // works in Qt5, but not Qt4
	int index = mServiceBox->findText(current);
	if (index > 0)
		mServiceBox->setCurrentIndex(index);
}

void LibrariesListView::updateLibrariesList(const std::string& service)
{
	LibraryRegistry::Register reg;
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	try {
		std::string method = mShowLoadedOnly ? "getLoadedLibraries" : "getLibraries";
		RPCFuture<LibraryRegistry::Register> f
			= rpcman.call<LibraryRegistry::Register>(service, method);
		if (!f.timedWait(cRPCTimeout))
			return;

		reg = f.get();
	}
	catch(...)
	{
		return;
	}

	for (int i = mLibrariesView->topLevelItemCount(); i < (int)reg.size(); ++i)
		mLibrariesView->addTopLevelItem(new QTreeWidgetItem());

	while (mLibrariesView->topLevelItemCount() > (int)reg.size())
		delete mLibrariesView->topLevelItem(0);

	int row = 0;
	foreach(const auto& l, reg)
	{
		QTreeWidgetItem* item = mLibrariesView->topLevelItem(row);
		item->setText(0, QString::fromStdString(l.first));
		item->setText(1, QString("%1.%2.%3").arg(l.second.mVersion.mMajor)
		                                    .arg(l.second.mVersion.mMinor)
		                                    .arg(l.second.mVersion.mPatch));
		item->setCheckState(2, l.second.mHasManifest ? Qt::Checked : Qt::Unchecked);		
		item->setCheckState(3, l.second.mIsLoaded ? Qt::Checked : Qt::Unchecked);
		item->setFlags(item->flags() & ~Qt::ItemIsUserCheckable);
		++row;
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::LibrariesListView, mira::EditorPart);
