/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ErrorServiceView.C
 *    .
 *
 * @author Tim Langner
 * @date   2011/02/02
 */

#include <thread/ThreadMonitor.h>
#include <serialization/Serialization.h>
#include <fw/Framework.h>
#include <fw/ErrorService.h>

#include <views/ErrorServiceView.h>

#include <QVBoxLayout>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::ErrorServiceView, mira::ViewPart);

///////////////////////////////////////////////////////////////////////////////

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

ErrorServiceView::ErrorServiceView() : ui(NULL)
{
}

QWidget* ErrorServiceView::createPartControl()
{
	ui = new UI(this);
	return ui;
}

///////////////////////////////////////////////////////////////////////////////


ErrorServiceView::UI::UI(QWidget* parent) : QWidget(parent)
{
	setupUi();
}

void ErrorServiceView::UI::setupUi()
{
	QVBoxLayout* verticalLayout = new QVBoxLayout(this);
	table = new QTableWidget(this);
	verticalLayout->addWidget(table);

	table->setColumnCount(6);
	table->setColumnWidth(0, 120);
	table->setColumnWidth(1, 200);
	table->setColumnWidth(2, 200);
	table->setColumnWidth(3, 120);
	table->setColumnWidth(4, 50);
	table->setColumnWidth(5, 150);
	table->setSelectionMode(QAbstractItemView::SingleSelection);
	table->setSelectionBehavior(QAbstractItemView::SelectRows);

	// Do initial update
	update();

	// Start update timer
	connect(&updateTimer, SIGNAL(timeout()), this, SLOT(update()));
	updateTimer.start(10000);
}

void ErrorServiceView::UI::update()
{
	ErrorService::ErrorVector v = MIRA_FW.getErrorService()->getErrors();

	using namespace boost::posix_time;
	boost::shared_ptr<time_facet> timeFacet;
	timeFacet.reset(new time_facet(
		//boost::local_time::local_time_facet::default_time_format,
		"%Y-%m-%d %H:%M:%S",
		time_facet::period_formatter_type(),
		time_facet::special_values_formatter_type(),
		time_facet::date_gen_formatter_type(), 1));

	table->clear();

	QStringList l;
	l << tr("Category") << tr("Authority") << tr("Text") << tr("Message")
	  << tr("Count") << tr("Last occurrence");
	table->setHorizontalHeaderLabels(l);

	table->setRowCount(v.size());
	for(uint32 i=0; i<v.size(); ++i)
	{
		table->setItem(i, 0, new QTableWidgetItem(v[i].category.c_str()));
		table->setItem(i, 1, new QTableWidgetItem(v[i].authority.c_str()));
		table->setItem(i, 2, new QTableWidgetItem(QString::fromUtf8(v[i].trText.c_str())));
		table->setItem(i, 3, new QTableWidgetItem(QString::fromUtf8(v[i].message.c_str())));
		table->setItem(i, 4, new QTableWidgetItem(QString("%1").arg(v[i].count)));

		Time timestamp = v[i].timestamp.toLocal();
		std::ostringstream ss;
		ss.imbue(std::locale(ss.getloc(), timeFacet.get()));
		ss << timestamp;

		table->setItem(i, 5, new QTableWidgetItem(QString::fromUtf8(ss.str().c_str())));
	}
}

///////////////////////////////////////////////////////////////////////////////

}
