/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RigidModel.C
 *    Implementation of RigidModel.h.
 *
 * @author Tim Langner
 * @date   2011/05/09
 */
#include <model/RigidModel.h>

#include <set>
#include <boost/geometry/algorithms/intersects.hpp>
#include <boost/geometry/geometries/polygon.hpp>

#include <xml/XMLDomReflect.h>

#include <model/URDF.h>
#include <model/RMDF.h>

namespace mira { namespace model {

///////////////////////////////////////////////////////////////////////////////

void RigidModel::reflect(XMLDeserializer& r)
{
	// check if we have a RMDF format tag
	XMLDom specialFormat;
	r.member("RMDF", specialFormat, "The optional rmdf description of this model",
	         serialization::IgnoreMissing());
	if (specialFormat.root().begin() != specialFormat.root().end())
		loadModelFromRMDF(specialFormat.root().begin(), *this);
	else
	{
		// no RMDF tag check for URDF format tag
		specialFormat = XMLDom();
		r.member("URDF", specialFormat, "The optional urdf description of this model",
		         serialization::IgnoreMissing());
		if (specialFormat.root().begin() != specialFormat.root().end())
			loadModelFromURDF(specialFormat.root().begin(), *this);
		else
		{
			// no special format tag use default reflection
			r.member("Name", name, "The name of this model", "");
			r.member("Links", links, "The links of this model", LinkList());
			r.member("Joints", joints, "The joints of this model", JointMap());
			r.member("Materials", materials, "The materials of this model", MaterialMap());
		}
	}
}

void RigidModel::clear()
{
	joints.clear();
	links.clear();
	materials.clear();
}

std::string RigidModel::getRootLink() const
{
	// if model only contains a single stray link that belongs to no joint, this is our root link
	if (joints.empty() && links.size() == 1)
		return (*links.begin())->name;

	// collect all parents and all children
	std::set<std::string> parents;
	std::set<std::string> children;
	foreach(const auto& p, joints)
	{
		parents.insert(p.second->parent);
		children.insert(p.second->child);
	}

	// find link that is parent but no child
	std::string rootLink;
	foreach(const auto& p, links)
	{
		if(parents.count(p->name)>0 && children.count(p->name)==0)
		{
			if(!rootLink.empty())
				MIRA_THROW(XLogical, "There is more than one root link in the model");
			rootLink = p->name;
		}
	}

	if(rootLink.empty())
		MIRA_THROW(XLogical, "There is NO root link in the model");

	return rootLink;
}

RigidModel::CollisionGeometries RigidModel::getCollisionGeometries(std::string targetFrame,
                                                    const Time& timestamp,
                                                    const std::string& filter) const
{
	if(targetFrame.empty())
		targetFrame = getRootLink();

	RigidModel::CollisionGeometries geometries;
	boost::shared_ptr<FrameworkTransformer> tf = MIRA_FW.getTransformer();
	foreach(const auto& link, links)
	{
		foreach(const auto& collision, link->collisions)
			if (collision->geometry && collision->name == filter)
			{
				FrameworkTransformerNode* target = tf->getNode(link->name);
				if(target==NULL)
					MIRA_THROW(XInvalidRead, "Frame '" << link->name << "' does not exist");
				FrameworkTransformerNode* source = tf->getNode(targetFrame);
				if(source==NULL)
					MIRA_THROW(XInvalidRead, "Frame '" << targetFrame << "' does not exist");
				RigidTransform3f t =  tf->getTransform<RigidTransform3f>(target, source,
				                                                         timestamp,
				                                                         NearestNeighborInterpolator());
				t *= collision->origin;

				geometries.push_back(std::make_pair(collision->geometry,t));
			}
	}

	return std::move(geometries);
}

std::list<Box3f> RigidModel::getCollisionBoundingBoxes(std::string targetFrame,
                                                       const Time& timestamp,
                                                       const std::string& filter) const
{
	std::list<Box3f> boxes;
	RigidModel::CollisionGeometries geometries =
			getCollisionGeometries(targetFrame,timestamp,filter);

	foreach(const auto& p, geometries)
		boxes.push_back(p.first->getBoundingBox(p.second));

	return boxes;
}

Footprint RigidModel::getFootprint(std::string targetFrame, const Time& timestamp,
                                   const std::string& filter) const
{
	if(targetFrame.empty())
		targetFrame = getRootLink();

	Footprint footprint;
	boost::shared_ptr<FrameworkTransformer> tf = MIRA_FW.getTransformer();
	// iterate over all links (frames)
	foreach(const auto& link, links)
	{
		// if we have a collision entry and that entry has a geometry
		foreach(const auto& collision, link->collisions)
			if (collision->geometry && collision->name == filter)
			{
				// get the frame nodes
				FrameworkTransformerNode* target = tf->getNode(link->name);
				if(target==NULL)
					MIRA_THROW(XInvalidRead, "Frame '" << link->name << "' does not exist");
				FrameworkTransformerNode* source = tf->getNode(targetFrame);
				if(source==NULL)
					MIRA_THROW(XInvalidRead, "Frame '" << targetFrame << "' does not exist");
				// get the transform from target to source node
				RigidTransform3f t = tf->getTransform<RigidTransform3f>(target, source,
				                                                        timestamp,
				                                                        NearestNeighborInterpolator());
				// calculate the transform of the collision objects origin to the target frame
				t *= collision->origin;

				// get the footprint in the correct frame
				Polygon2f linkFootprint = collision->geometry->getFootprint(t);

				// add footprint to overall footprint if not empty
				if(!linkFootprint.empty())
					footprint.push_back(linkFootprint);
			}
	}

	// merge polygons
	bool merged;
	do
	{
		merged = false;
		Footprint mergedFootprint;
		for(std::size_t i=0; i<footprint.size(); ++i)
		{
			const Polygon2f& pi = footprint[i];
			for(std::size_t j=i+1; j<footprint.size(); ++j)
			{
				const Polygon2f& pj = footprint[j];

				// if both polygons overlap, then merge them
				if(boost::geometry::intersects(pi,pj)) {

					std::vector<boost::geometry::model::polygon<Point2f>> v;
					boost::geometry::union_(pi,pj,v);

					if(!v.empty()) {
						mergedFootprint.push_back(v[0].outer());

						// add remaining polygons
						for(std::size_t k=i+1; k<footprint.size(); ++k)
							if(k!=j)
								mergedFootprint.push_back(footprint[k]);

						merged = true;
						// we have merged, so break all inner loops and start next iteration
						goto next_iter;
					}
				}
			}
			// if we reach here pi was not merged with any other polygon, so
			// insert it without merging
			mergedFootprint.push_back(footprint[i]);
		}

	next_iter:
		std::swap(footprint, mergedFootprint);

	} while(merged);

	return footprint;
}

void RigidModel::resolveAndPublishLinks(const std::string& ns, const Time& timestamp)
{
	try
	{
		foreach (LinkPtr l, links)
		{
			if (l)
				l->name = MIRA_FW.getNameRegistry().resolve(l->name, ns);
		}
		boost::shared_ptr<FrameworkTransformer> tf = MIRA_FW.getTransformer();
		foreach (auto& j, joints)
		{
			if (j.second)
			{
				j.second->parent = MIRA_FW.getNameRegistry().resolve(j.second->parent, ns);
				j.second->child = MIRA_FW.getNameRegistry().resolve(j.second->child, ns);
				tf->addTypedLink(j.second->child, j.second->parent,
				                 j.second->type == Joint::FIXED ? FrameworkTransformerNode::FIXED
				                                                : FrameworkTransformerNode::MOVING);
				tf->publishTransform(j.second->child, j.second->origin, timestamp);
			}
		}
	}
	catch(Exception& ex)
	{
		MIRA_RETHROW(ex, "while resolving and publishing links of rigid model");
	}
}

void RigidModel::publishJoint(const std::string& name, float value,
                              const Time& timestamp) const
{
	JointPtr joint;
	auto it = joints.find(name);
	if (it == joints.end())
		MIRA_THROW(XInvalidConfig, "Joint '" << name << "' does not exist in model");
	joint = it->second;
	// Fixed joints can not be modified
	if (joint->type == Joint::FIXED)
		MIRA_THROW(XInvalidConfig, "Can not modify fixed joint '" << name << "'");
	// Can't handle floating joints here since they have 6DoF
	if (joint->type == Joint::FLOATING)
		MIRA_THROW(XInvalidConfig, "Can not modify floating joint '"
		           << name << "' with a single float value since it has 6DoF");
	RigidTransform3f t = joint->origin;
	if (joint->type == Joint::CONTINUOUS)
		t.r *= Eigen::Quaternionf(Eigen::AngleAxisf(value, joint->axis));
	else
	{
		float v = value;
		if (v < joint->lowerLimit)
			v = joint->lowerLimit;
		if (v > joint->upperLimit)
			v = joint->upperLimit;
		if (joint->type == Joint::REVOLUTE)
			t.r *= Eigen::Quaternionf(Eigen::AngleAxisf(v, joint->axis));
		if (joint->type == Joint::PRISMATIC)
			t.t += v * joint->axis;
	}
	MIRA_FW.getTransformer()->publishTransform(joint->child, t, timestamp);
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::model::RigidModel, mira::Object);
