/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Footprint.h
 *    Representation of a footprint of a rigid model (part).
 *
 * @author Erik Einhorn
 * @date   2011/11/28
 */

#ifndef _MIRA_FOOTPRINT_H_
#define _MIRA_FOOTPRINT_H_

#include <serialization/adapters/std/vector>

#include <geometry/Line.h>
#include <geometry/Polygon.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Represents the footprint of a rigid model that is the 2D projection of
 * the 3D model onto the xy-plane. The footprint is represented by a vector of
 * polygons.
 */
class Footprint : public std::vector<Polygon2f>
{
	typedef std::vector<Polygon2f> Base;
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
	}

public:

	/// Returns the radius of the inscribed circle (incircle) of the footprint
	float getInnerRadius() const {

		float rMin = std::numeric_limits<float>::infinity();
		foreach(const Polygon2f& polygon,*this)
		{
			for(std::size_t i=1; i<polygon.size(); ++i)
			{
				Line2f l(polygon[i-1], polygon[i]);
				float d = boost::geometry::distance(Point2f(0,0), l);
				if(d<rMin)
					rMin = d;
			}
		}
		if(rMin==std::numeric_limits<float>::infinity())
			return 0.0f;
		return rMin;
	}

	/// Returns the radius of the escribed circle (excircle) of the footprint
	float getOuterRadius() const {

		float rSq = 0.0f;
		foreach(const Polygon2f& polygon,*this)
		{
			foreach(const Point2f& p, polygon)
			{
				float d = p.x()*p.x() + p.y()*p.y();
				if(d>rSq)
					rSq = d;
			}
		}
		return std::sqrt(rSq);
	}

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
