/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseVectorVisualization.C
 *    Visualization of vector of poses.
 *
 * @author Tim Langner
 * @date   2011/06/15
 */

#include <serialization/Serialization.h>
#include <serialization/GetterSetter.h>

#include <math/EigenFormat.h>
#include <visualization/Visualization3DBasic.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include "private/PoseCovVisualization.h"

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename Pose>
class PoseVectorVisualizationBase : public Visualization3D
{
public:
	typedef typename PoseVectorTrait<Pose>::VectorType PoseVector;

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	PoseVectorVisualizationBase() :
		mNode(NULL),
		mNumberOfVisiblePoseVectors(0),
		mAxisLength(0.1f),
		mAxisRadius(0.01f),
		mShowArrows(true),
		mShowCovariance(CovVisTrait<Pose>::poseHasCovariance),
		mSigma(0.5f)
	{
		mDataChannel.setDataChangedCallback(
			boost::bind(&PoseVectorVisualizationBase::dataChangedCallback, this, _1));
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3D);
		channelProperty(r, "PoseVector", mDataChannel, "The channel to be visualized");

		r.property("Length", mAxisLength, "The length of the axes in m", 0.1f);
		r.property("Radius", mAxisRadius, "The radius of the axes in m", 0.01f);
		r.property("ShowArrows", mShowArrows, "Should arrows be shown", true);

		r.property("Color", mAxisColor, setter<Ogre::ColourValue>(&PoseVectorVisualizationBase::setColor, this),
				"Uniform color for the axes arrows", Ogre::ColourValue::Red);

		if (CovVisTrait<Pose>::poseHasCovariance) {
			r.property("Show Covariance", mShowCovariance,
			           setter<bool>(&PoseVectorVisualizationBase::setShowCovariance, this),
			           "Show/Hide the covariance", true);
			r.property("Sigma", mSigma, "The sigma of the shown surface", 0.5f);
		}
	}

	//@}

public:
	/** @name Public implementation of Visualization3D */
	//@{

	virtual void setupScene(IVisualization3DSite* site)
	{
		mNode = site->getSceneManager()->getRootSceneNode()->createChildSceneNode();
	}

	virtual Ogre::SceneNode* getNode()
	{
		return mNode;
	}

	void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		for (std::size_t i = 0; i < mPoses.size(); ++i) {
			if (i < mNumberOfVisiblePoseVectors) {
				mPoses[i]->axes.setVisible(enabled);
				mPoses[i]->cov.setVisible(enabled && mShowCovariance);
			}
			else {
				mPoses[i]->axes.setVisible(false);
				mPoses[i]->cov.setVisible(false);
			}
		}
	}

	virtual DataConnection getDataConnection()
	{
		return DataConnection(mDataChannel);
	}

	//@}

protected:
	/** @name Protected implementation of Visualization3D */
	//@{

	virtual void dataChanged(ChannelRead<PoseVector> data)
	{
		while(mPoses.size()<data->size())
		{
			Ogre::SceneManager* sceneManager = this->getSite()->getSceneManager();
			boost::shared_ptr<PoseObject> p;
			p.reset(new PoseObject(sceneManager, mNode));
			p->axes.showYAxis(Pose::Dim>2);
			p->axes.showZAxis(Pose::Dim>2);
			mPoses.push_back(p);
		}

		// setup visible axes
		for(std::size_t i=0; i<data->size(); ++i)
		{
			mPoses[i]->pose = data->value()[i];
			mPoses[i]->axes.showArrows(mShowArrows);
			mPoses[i]->axes.setVisible(Visualization3D::isEnabled());
			mPoses[i]->cov.setVisible(Visualization3D::isEnabled() && mShowCovariance);
		}

		updatePoses();
		mNumberOfVisiblePoseVectors = data->size();

		// hide unused poses
		for(std::size_t i=data->size(); i<mPoses.size(); ++i)
		{
			mPoses[i]->axes.setVisible(false);
			mPoses[i]->cov.setVisible(false);
		}
	}

	virtual void update(Duration dt)
	{
		if (!mDataChannel.isValid())
			return;

		updatePoses();
	}

	//@}

private:
	void dataChangedCallback(ChannelRead<PoseVector> data)
	{
		mDataFrameID = data->frameID;
		mDataTimestamp = data->timestamp;
		dataChanged(data);
	}

	void updatePoses()
	{
		mParentTransform = getAuthority().template getTransform<Pose>(
		    mDataFrameID, getSite()->getCameraFrame());

		// setup visible axes
		for(std::size_t i=0; i<mPoses.size(); ++i)
		{
			Pose pose = mParentTransform;
			pose *=  mPoses[i]->pose;

			mPoses[i]->axes.setLength(mAxisLength);
			mPoses[i]->axes.setRadius(mAxisRadius);
			mPoses[i]->axes.setTransform(pose.template cast<float>());
			mPoses[i]->axes.setColor(mAxisColor);

			mPoses[i]->cov.setSigma(mSigma);
			mPoses[i]->cov.update(pose.template cast<float>());
		}
	}

	void setShowCovariance(bool show)
	{
		mShowCovariance = show;
		setEnabled(isEnabled());
	}

	void setColor(const Ogre::ColourValue& color)
	{
		mAxisColor = color;

		for(std::size_t i=0; i < mPoses.size(); ++i)
			mPoses[i]->axes.setColor(color);
	}

private:
	struct PoseObject {
		PoseObject(Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent) :
			axes(sceneManager, parent) {
			cov.setupScene(sceneManager,&axes);
		}

		EIGEN_MAKE_ALIGNED_OPERATOR_NEW
		AxesObject axes;
		CovVisualization cov;
		Pose pose;
	};

	Ogre::SceneNode* mNode;
	std::size_t mNumberOfVisiblePoseVectors;
	std::vector<boost::shared_ptr<PoseObject>> mPoses;
	float mAxisLength;
	float mAxisRadius;
	bool mShowArrows;
	Ogre::ColourValue mAxisColor;

	bool mShowCovariance;
	float mSigma;

	ChannelProperty<PoseVector> mDataChannel;

	std::string mDataFrameID;
	Time mDataTimestamp;
	Pose mParentTransform;
};

///////////////////////////////////////////////////////////////////////////////

class Pose3VectorVisualization : public PoseVectorVisualizationBase<Pose3>
{
	MIRA_META_OBJECT(Pose3VectorVisualization,
		("Name", "PoseVector3D")
		("Description", "Visualization of a vector of 3D points")
		("Category", "Pose"))
};

class Pose2VectorVisualization : public PoseVectorVisualizationBase<Pose2>
{
	MIRA_META_OBJECT(Pose2VectorVisualization,
		("Name", "PoseVector2D")
		("Description", "Visualization of a vector of 2D points")
		("Category", "Pose"))
};

class RigidTransform2dVectorVisualization : public PoseVectorVisualizationBase<RigidTransform2d>
{
	MIRA_META_OBJECT(RigidTransform2dVectorVisualization,
		("Name", "PoseVector2D (double)")
		("Description", "Visualization of a vector of 2D double points")
		("Category", "Pose"))
};

class Pose3CovVectorVisualization : public PoseVectorVisualizationBase<PoseCov3>
{
	MIRA_META_OBJECT(Pose3CovVectorVisualization,
		("Name", "PoseCovVector3D")
		("Description", "Visualization of a vector of 3D covariance points")
		("Category", "Pose"))
};

class Pose2CovVectorVisualization : public PoseVectorVisualizationBase<PoseCov2>
{
	MIRA_META_OBJECT(Pose2CovVectorVisualization,
		("Name", "PoseCovVector2D")
		("Description", "Visualization of a vector of 2D covariance points")
		("Category", "Pose"))
};

class RigidTransformCov2dVectorVisualization : public PoseVectorVisualizationBase<RigidTransformCov2d>
{
	MIRA_META_OBJECT(RigidTransformCov2dVectorVisualization,
		("Name", "PoseCovVector2D (double)")
		("Description", "Visualization of a vector of 2D double covariance points")
		("Category", "Pose"))
};

//////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Pose3VectorVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose2VectorVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::RigidTransform2dVectorVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose3CovVectorVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose2CovVectorVisualization,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::RigidTransformCov2dVectorVisualization,
                         mira::Visualization3D);
