#!/bin/bash

###############################################################################
#
# A bash script, which creates a new release of a package. The scripts does
# the following steps:
#   - put a release date to MIRA_CHANGELOG in the package file
#   - ask the user for verifying the modified package file
#   - commit the package file
#   - create a tag for the new release
#   - add a mountdir.xml for the new tag
#
# The script supports svn and git.
#
# Usage:  releasePackage.sh PathToPackage PackageName
#
#     PathToPackage:  The path to package directory
#     PackageName:    The name of the package (without .package extension!)
#
# Author: Christian Martin
#
###############################################################################

confirm()
{
	echo -n "$1 ? (n) "
	read ans
	case "$ans" in
	y|Y|yes|YES|Yes) return 0 ;;
	*) return 1 ;;
	esac
}

###############################################################################
# Extract and check the command line arguments

if [ $# != 2 ]; then
	echo "Usage: $0 PathToPackage PackageName"
	exit
fi

SRCDIR=$1
PKGNAME=$2

if [ ! -e $SRCDIR ]; then
	echo "The directory $SRCDIR does not exist. Abort!"
	exit
fi

if [ ! -f $SRCDIR/$PKGNAME.package ]; then
	echo "The file $SRCDIR/$PKGNAME.package does not exist. Abort!"
	exit
fi

###############################################################################
# First determine used version control system. We support svn and git here.

VCS=""
if [ -d $SRCDIR/.svn ] ; then
	VCS="svn"
elif [ -d $SRCDIR/.git ] ; then
	VCS="git"
else
	echo "Missing or unknown version control system in $SRCDIR"
	exit
fi

echo "Using $VCS for $SRCDIR"
cd $SRCDIR

###############################################################################
# Update to current head

if [ $VCS == "svn" ] ; then
	# Perform a SVN update
	echo "Doing 'svn update' for directory $SRCDIR..."
	svn update
fi

if [ $VCS == "git" ] ; then
	# Perform a GIT pull
	echo "Doing 'git pull' for directory $SRCDIR..."
	git pull
fi

# Extract the version number from the package file
PKG_VER=`cat $PKGNAME.package | grep MIRA_VERSION | \
	sed -r "s/MIRA_VERSION\([ \t]*([0-9]*)[ \t]*([0-9]*)[ \t]*([0-9]*)\)/\1.\2.\3/g"`

echo $PKG_VER

###############################################################################
# First, extract all relevant information

if [ $VCS == "svn" ] ; then

	SVN_URL=`svn --xml info . | xml_grep --text_only 'url'`
	SVN_ROOT=`svn --xml info . | xml_grep --text_only 'root'`

	# remove $SVN_ROOT from the beginning of $SVN_URL
	SVN_REL=${SVN_URL#${SVN_ROOT}/}

	# get the sub-dir in the repository and check, if it starts with "branches"
	SVN_PKG_REL_PATH=`dirname ${SVN_REL}`
	if [[ ! $SVN_PKG_REL_PATH == branches/* ]] ; then
		echo "ERROR: Don't know how to handle sub-dir '$SVN_PKG_REL_PATH'."
		exit
	fi

	# remove "branches" from beginning of SVN_PKG_REL_PATH
	MOUNT_DIR=${SVN_PKG_REL_PATH#branches/}

	# get the directory name of the package
	SVN_PKG_DIR_NAME=`basename ${SVN_REL}`

	# create name of the tag
	SVN_TAG_DIR_NAME=$SVN_ROOT/tags/$MOUNT_DIR/$SVN_PKG_DIR_NAME-$PKG_VER/$SVN_PKG_DIR_NAME
	MOUNT_DIR_DEST=$SVN_ROOT/tags/$MOUNT_DIR/$SVN_PKG_DIR_NAME-$PKG_VER
fi

if [ $VCS == "git" ] ; then
	#GIT_REMOTE_URL=`git remote get-url origin`
	GIT_REMOTE_URL=`git config --get remote.origin.url`
fi

###############################################################################
# Show the information to the user and let the user confirm them

echo ""
echo "Please verify the following information:"
echo "  Name of package  : $PKGNAME"
echo "  Version          : $PKG_VER"
if [ $VCS == "svn" ] ; then
	echo "  SVN root URL     : $SVN_ROOT"
	echo "  Path to package  : $SVN_PKG_REL_PATH"
	echo "  Package dir name : $SVN_PKG_DIR_NAME"
	echo ""
	echo "  MountDir         : $MOUNT_DIR/$SVN_PKG_DIR_NAME"
	echo "  TagDir           : $SVN_TAG_DIR_NAME"
	echo ""
fi
if [ $VCS == "git" ] ; then
	echo "  Git remote URL   : $GIT_REMOTE_URL"
	echo ""
fi

if ! confirm "Are this information correct" ; then
	echo "Abort"
	exit
fi

###############################################################################
# Modify the changelog file for the new release

# Put the release date into the package file
DATESTR=`date +%Y-%m-%d`
cat $PKGNAME.changelog | \
	sed "0,/Version $PKG_VER.*/s//Version $PKG_VER (Release date: $DATESTR)/" \
	> $PKGNAME.changelog.new

# Lets verify the changelog file by the user
echo ""
read -p "Please check the updated changelog file (Enter)..."
less $PKGNAME.changelog.new
echo ""
if ! confirm "Is the changelog file correct" ; then
	rm $PKGNAME.changelog.new
	echo "Abort"
	exit
fi

# Lets verify the package file by the user
echo ""
read -p "Please check the package file (Enter)..."
less $PKGNAME.package
echo ""
if ! confirm "Is the package file correct" ; then
	echo "Abort"
	exit
fi

# Update the changelog file
mv $PKGNAME.changelog.new $PKGNAME.changelog

###############################################################################
# Create and add the mountdir.xml

if [ ! -f mountdir.xml ]; then

	# If git is used, it is not so easy to determine the correct mount dir.
	if [ $VCS == "git" ] ; then
		echo ""
		echo "File mountdir.xml is missing. Abort."
		exit
	fi

	echo "<root>" > mountdir.xml
	echo "  <path>$MOUNT_DIR/$SVN_PKG_DIR_NAME</path>" >> mountdir.xml
	echo "</root>" >> mountdir.xml

	echo ""
	echo "Adding mountdir.xml..."
	svn add mountdir.xml

else
	echo ""
	echo "Skipped mountdir.xml, since it already exists."
fi

###############################################################################
# Show "svn stat" and let the user confirm the status

echo ""
echo "Please verify the current status:"
echo ""

if [ $VCS == "svn" ] ; then
	svn stat .
fi
if [ $VCS == "git" ] ; then
	git status .
fi

echo ""
if ! confirm "Continue" ; then
	echo "Abort"
	exit
fi

###############################################################################
# Commit the package file to the SVN

MSG_PREFIX="Auto-commit by releasePackage.sh: "

echo ""
echo "Commiting new release..."
if [ $VCS == "svn" ] ; then
	svn commit -m "$MSG_PREFIX Released $PKGNAME-$PKG_VER" .
fi
if [ $VCS == "git" ] ; then
	git commit -a -m "$MSG_PREFIX Released $PKGNAME-$PKG_VER"
	git push origin
fi

###############################################################################
# Tag the new version

echo ""
echo "Creating tag for new release..."
if [ $VCS == "svn" ] ; then
	svn copy -m "$MSG_PREFIX Created tag for $PKGNAME-$PKG_VER" -r HEAD --parents $SVN_URL $SVN_TAG_DIR_NAME
fi
if [ $VCS == "git" ] ; then
	# create tag
	git tag -a $PKG_VER -m "$MSG_PREFIX Created tag for $PKGNAME-$PKG_VER"
	# and push to remote server
	git push origin $PKG_VER
fi

