/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditorCommonDelegates.C
 *    Implementation of private/PropertyEditorCommonDelegates.h
 *
 * @author Erik Einhorn
 * @date   2010/12/14
 */

#include <math/Saturate.h>
#include <utils/Path.h>

#include <widgets/PropertyEditor.h>
#include <widgets/private/PropertyEditorCommonDelegates.h>
#include <widgets/LineEditTextFormat.h>
#include <widgets/QtUtils.h>

#include <QObject>
#include <QLineEdit>
#include <QWheelEvent>

#include <QApplication>
#include <QPainter>

#include <QFileDialog>


namespace mira {

///////////////////////////////////////////////////////////////////////////////

PropertyDelegate_Slider::PropertyDelegate_Slider(PropertyNode* p,
                                                 QWidget* parent) :
	QSlider(parent), property(p)
{
	setOrientation(Qt::Horizontal);
	setMinimum(p->getHint<int>("minimum",std::numeric_limits<int>::min()));
	setMaximum(p->getHint<int>("maximum",std::numeric_limits<int>::max()));
	setSingleStep(p->getHint<int>("step",1));
	QVariant val(p->getAsString().c_str());
	setValue(val.toInt());
	connect(this, SIGNAL(valueChanged(int)), this, SLOT(slotSetValue(int)));
}

void PropertyDelegate_Slider::slotSetValue(int value)
{
	QVariant val(value);
	property->setFromString(val.toString().toStdString());
}

//////////////////////////////////////////////////////////////////////////////

SpinBoxSmartScroll::SpinBoxSmartScroll() :
	mMouseMovedSinceLastScroll(true), mScrolledDigit(0), mToAdd(0.0)
{}

void SpinBoxSmartScroll::handleEventFilter(QEvent* event)
{
	if (event->type() == QEvent::MouseMove){
		mMouseMovedSinceLastScroll = true;
	}
}

double SpinBoxSmartScroll::handleWheelEvent(QWheelEvent* event,
                                           QLineEdit* lineEdit,
                                           QChar decimalPoint,
                                           double defaultStep)
{
	if (event->orientation() == Qt::Vertical) {
		if (mMouseMovedSinceLastScroll){
			mMouseMovedSinceLastScroll = false;
			int cursorPosX = event->pos().x() - 4;
			// ^^ const int QLineEditPrivate::horizontalMargin(2);
			QFontMetrics fm = lineEdit->fontMetrics();
			QString content = lineEdit->text() + "M"; // add spacing at the end
			// to be able to step 0.1 when displayed "2", otherwise defaultStep
			int cursorPosition = content.size();
			bool cursorOverText = false;
			for(int i = 1; i <= content.size(); ++i){
				QString contentPart = content.left(i);
				int contentPartWidth = fm.width(contentPart);
				if (cursorPosX < contentPartWidth){
					cursorPosition = i - 1;
					cursorOverText = true;
					break;
				}
			}

			if (cursorOverText){
				mScrolledDigit = 0;
				int exponent = 0;
				const QString text  = lineEdit->text();

				// handle decimal point
				int pointPosition = text.indexOf(decimalPoint);
				if (pointPosition < 0)
					pointPosition = text.size();

				// handle numbers like 2.444e10
				auto s = text.split('e');
				if (s.size() == 2)
					exponent = s.at(1).toInt();

				mScrolledDigit = pointPosition-cursorPosition;

				int scrollExponent = mScrolledDigit;
				if(scrollExponent>=0)
					scrollExponent--; // remove effect of decimal point
				if(mScrolledDigit==0)
					mScrolledDigit--; // when we are on the decimal point, we scroll the first digit behind it

				mToAdd = pow(10.0, (double)(scrollExponent+exponent));
			} else {
				// cursor is right to the numbers, fall back to old value
				mToAdd = defaultStep;
				mScrolledDigit=0;
			}
		}

		float degrees = event->delta() / 8.0f; // Qt-Documentation:
		float steps = degrees / 15.0f;          // Most mouse types work in steps of 15 degrees
		//mScrollError += steps;
		int wholeSteps = (int)floor(steps); // this should add support for scrollwheels with steps < 15 degrees
		steps -= wholeSteps;
		event->accept();
		return mToAdd * wholeSteps;
	}
	return 0;
}

void SpinBoxSmartScroll::highlightScrolledDigit(QLineEdit* lineEdit, QChar decimalPoint)
{
	if(mScrolledDigit!=0) {
		const QString text  = lineEdit->text();
		int pointPosition = text.indexOf(decimalPoint);
		if (pointPosition < 0)
			pointPosition = text.size();

		lineEdit->setCursor(Qt::ArrowCursor);
		QList<QTextLayout::FormatRange> formats;
		QTextCharFormat f;
		f.setFontWeight(QFont::Black);
		f.setFontUnderline(true);
		QTextLayout::FormatRange fr;
		fr.start = pointPosition - mScrolledDigit;
		fr.length = 1;
		fr.format = f;
		formats.append(fr);
		setLineEditTextFormat(lineEdit, formats);
	} else {
		lineEdit->setCursor(Qt::IBeamCursor);
		clearLineEditTextFormat(lineEdit);
	}
}

int SpinBoxSmartScroll::scrolledDigit() const
{
	return mScrolledDigit;
}

///////////////////////////////////////////////////////////////////////////////

PropertyDelegate_SpinBox::PropertyDelegate_SpinBox(PropertyNode* p,
                                                   QWidget* parent) :
	QSpinBox(parent), property(p)
{
	setFrame(false);
	setMinimum(p->getHint<int>("minimum",std::numeric_limits<int>::min()));
	setMaximum(p->getHint<int>("maximum",std::numeric_limits<int>::max()));
	setSingleStep(p->getHint<int>("step",1));
	QVariant val(p->getAsString().c_str());
	setValue(val.toInt());
	connect(this, SIGNAL(valueChanged(int)), this, SLOT(slotSetValue(int)));
	lineEdit()->setMouseTracking(true);
	lineEdit()->installEventFilter(this); // mouse move events on QDoubleSpinBox work only on step buttons
}

bool PropertyDelegate_SpinBox::eventFilter(QObject* obj, QEvent* event)
{
	this->handleEventFilter(event);
	return false;
}

void PropertyDelegate_SpinBox::wheelEvent(QWheelEvent* event)
{
	int toAdd = this->handleWheelEvent(event, lineEdit(), QChar(' '), singleStep());
	int newValue = saturate(lineEdit()->text().toInt() + toAdd, minimum(), maximum());
	slotSetValue(newValue);
	lineEdit()->setText(QString::number(newValue));
	highlightScrolledDigit(lineEdit(), QChar(' '));
}

void PropertyDelegate_SpinBox::slotSetValue(int value)
{
	QVariant val(value);
	property->setFromString(val.toString().toStdString());
}


//////////////////////////////////////////////////////////////////////////////

PropertyDelegate_ComboBox::PropertyDelegate_ComboBox(PropertyNode* p,
                                                     QWidget* parent) :
	QComboBox(parent), property(p)
{
	QString valIdx = p->getAsString().c_str();
	int idx = findData(valIdx);
	setCurrentIndex(idx);
	connect(this, SIGNAL(activated(int)), this, SLOT(slotSetValue(int)));
}

void PropertyDelegate_ComboBox::slotSetValue(int value)
{
	QVariant val = itemData(value);
	property->setFromString(val.toString().toStdString());
}

///////////////////////////////////////////////////////////////////////////////

// templated/overloaded methods for translation between int/std::string/QString,
// enables templating getEnumerationString/createEnumerationEditor to use
// with int and std::string properties (specific key type ensures correct ordering)

// the default implementation assumes T = int or compatible type

template <typename T>
const T& out(const T& t)
{
	return t;
}

std::string out(const QString& s)
{
	return s.toStdString();
}

template <typename T>
T toT(const QString& s)
{
	bool ok;
	T t = s.toInt(&ok);

	if (!ok)
		MIRA_THROW(XInvalidParameter,
		           "The enumeration value '" << out(s)
		           << "' is not valid for an integral-type property");

	return t;
}

template <typename T>
T toT(int i)
{
	return i;
}

template <typename T>
T toT(const std::string& s)
{
	return std::stoi(s);
}

QString toQString(int i)
{
	return QString::number(i);
}

template <>
QString toT<QString>(const QString& s)
{
	return s;
}

template <>
QString toT<QString>(int i)
{
	return toString(i).c_str();
}

template <>
QString toT<QString>(const std::string& s)
{
	return s.c_str();
}

const QString& toQString(const QString& s)
{
	return s;
}

template<typename T>
std::string getEnumerationString(const PropertyNode* property,
                                 const std::string& enumTexts)
{
	QString s(enumTexts.c_str());
	QStringList texts = s.split(";");
	int assignCount = s.count('=');
	std::map<T, QString> values;
	if (assignCount > 0)
	{
		if (assignCount < texts.size())
			MIRA_THROW(XInvalidParameter, "The enumeration must either contain only ; separated values"
			           " or ; separated index=value pairs");
		foreach(const QString& t, texts)
		{
			QStringList pair = t.split('=');
			values[toT<T>(pair[0])] = pair[1];
		}
	}
	else
	{
		for (int i=0; i<texts.size(); ++i)
			values[toT<T>(i)] = texts[i];
	}
	T idx = toT<T>(property->getAsString());
	if(values.count(idx) == 0)
		MIRA_THROW(XInvalidParameter, "The enumeration value '" << out(idx)
		           << "' is out of bounds of the specified enumeration"
		           " texts: '" << enumTexts << "'");
	return values[idx].toStdString();
}

template <typename T>
QWidget* createEnumerationEditor(PropertyNode* property, QWidget* parent,
                                 const std::string& enumTexts)
{
	QComboBox* box = new PropertyDelegate_ComboBox(property,parent);
	QString s(enumTexts.c_str());
	QStringList texts = s.split(";");
	int assignCount = s.count('=');
	std::map<T, QString> values;
	if (assignCount > 0)
	{
		if (assignCount < texts.size())
			MIRA_THROW(XInvalidParameter, "The enumeration must either contain only ; separated values"
			           " or ; separated index=value pairs");
		foreach(const QString& t, texts)
		{
			QStringList pair = t.split('=');
			values[toT<T>(pair[0])] = pair[1];
		}
	}
	else
	{
		for (int i=0; i<texts.size(); ++i)
			values[toT<T>(i)] = texts[i];
	}
	foreach(auto v, values)
		box->addItem(v.second, toQString(v.first));

	QString valIdx = property->getAsString().c_str();
	int idx = box->findData(valIdx);
	if (idx < 0)
		MIRA_THROW(XInvalidParameter, "Index " << valIdx.toStdString() << " is out of bounds in enumeration " << enumTexts);
	box->setCurrentIndex(idx);
	return box;
}

class PropertyDelegate_Integers : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_Integers)
public:

	virtual SupportedTypes supportedTypes() const {
		return
			makeSupportedType<uint8>(SupportedType::TEXT | SupportedType::EDITOR) +
			makeSupportedType<uint16>(SupportedType::TEXT | SupportedType::EDITOR) +
			makeSupportedType<uint32>(SupportedType::TEXT | SupportedType::EDITOR) +
			makeSupportedType<uint64>(SupportedType::TEXT) +
			makeSupportedType<int8>(SupportedType::TEXT | SupportedType::EDITOR) +
			makeSupportedType<int16>(SupportedType::TEXT | SupportedType::EDITOR) +
			makeSupportedType<int32>(SupportedType::TEXT | SupportedType::EDITOR) +
			makeSupportedType<int64>(SupportedType::TEXT);
	}

	virtual std::string getText(const PropertyNode* property) {
		// special treatment if we have an enumeration
		if(!property->getHint<std::string>("enumeration").empty()) {
			std::string enumTexts = property->getHint<std::string>("enumeration");
			return getEnumerationString<int>(property, enumTexts);
		}

		return property->getAsString();
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent) {

		// special treatment if we have an enumeration
		if(!property->getHint<std::string>("enumeration").empty()) {
			std::string enumTexts = property->getHint<std::string>("enumeration");
			return createEnumerationEditor<int>(property, parent, enumTexts);
		}

		if(property->getHint<std::string>("type")=="slider")
			return new PropertyDelegate_Slider(property, parent);
		else
			return new PropertyDelegate_SpinBox(property, parent);
	}
};

///////////////////////////////////////////////////////////////////////////////

PropertyDelegate_DoubleSlider::PropertyDelegate_DoubleSlider(PropertyNode* p,
                                                 QWidget* parent) :
	QSlider(parent), property(p)
{
	setOrientation(Qt::Horizontal);
	mMinimum=(p->getHint<double>("minimum",0.0));
	mMaximum=(p->getHint<double>("maximum",1.0));
	mStep=(p->getHint<double>("step",0.1));

	// floating point values are mapped to integral steps of slider
	setMinimum(0);
	setMaximum((int)(round((mMaximum-mMinimum)/mStep)+0.5));
	setSingleStep(1);

	QVariant val(p->getAsString().c_str());
	double value = val.toDouble();
	int step = (int)(round((value-mMinimum)/mStep)+0.5);
	step = saturate<int>(step, 0, maximum());
	setValue(step);
	connect(this, SIGNAL(valueChanged(int)), this, SLOT(slotSetValue(int)));
}

void PropertyDelegate_DoubleSlider::slotSetValue(int step)
{
	double value = saturate<double>(step*mStep + mMinimum, mMinimum, mMaximum);
	QVariant val(value);
	property->setFromString(val.toString().toStdString());
}

///////////////////////////////////////////////////////////////////////////////

PropertyDelegate_DoubleSpinBox::PropertyDelegate_DoubleSpinBox(PropertyNode* p,
                                                               QWidget* parent) :
	QDoubleSpinBox(parent), SpinBoxSmartScroll(), property(p), locale("C")
{
	// set "C" locale
	setLocale(locale);
	// prevent Qt to insert a group separator into the string
	locale.setNumberOptions(QLocale::OmitGroupSeparator);

	setFrame(false);
	setMinimum(p->getHint<double>("minimum",-std::numeric_limits<double>::max()));
	setMaximum(p->getHint<double>("maximum",std::numeric_limits<double>::max()));
	precision = p->getHint<int>("precision",6);
	setDecimals(precision);
	setSingleStep(p->getHint<double>("step",0.1));
	QVariant val(p->getAsString().c_str());
	setValue(val.toDouble());
	connect(this, SIGNAL(valueChanged(double)), this, SLOT(slotSetValue(double)));
	lineEdit()->setMouseTracking(true);
	lineEdit()->installEventFilter(this); // mouse move events on QDoubleSpinBox work only on step buttons
}

bool PropertyDelegate_DoubleSpinBox::eventFilter(QObject* obj, QEvent* event)
{
	this->handleEventFilter(event);
	return false;
}

void PropertyDelegate_DoubleSpinBox::wheelEvent(QWheelEvent* event)
{
	double toAdd = this->handleWheelEvent(event, lineEdit(), locale.decimalPoint(), singleStep());
	double newValue = saturate(valueFromText(lineEdit()->text()) + toAdd, minimum(), maximum());
	slotSetValue(newValue);
	lineEdit()->setText(textFromValue(newValue));
	highlightScrolledDigit(lineEdit(), locale.decimalPoint());
}

QString PropertyDelegate_DoubleSpinBox::textFromValue(double value) const
{
	return locale.toString(value, 'g', precision);
}

double PropertyDelegate_DoubleSpinBox::valueFromText(const QString& text) const
{
	return locale.toDouble(text);
}

void PropertyDelegate_DoubleSpinBox::slotSetValue(double value)
{
	QVariant val(value);
	property->setFromString(val.toString().toStdString());
}

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_Floats : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_Floats)
public:

	virtual SupportedTypes supportedTypes() const {
		return
			makeSupportedType<float>(SupportedType::TEXT  | SupportedType::EDITOR) +
			makeSupportedType<double>(SupportedType::TEXT | SupportedType::EDITOR);
	}

	virtual std::string getText(const PropertyNode* property) {
		return property->getAsString();
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent) {
		if(property->getHint<std::string>("type")=="slider")
			return new PropertyDelegate_DoubleSlider(property, parent);
		else
			return new PropertyDelegate_DoubleSpinBox(property, parent);
	}
};

///////////////////////////////////////////////////////////////////////////////

PropertyDelegate_LineEdit::PropertyDelegate_LineEdit(PropertyNode* p,
                                                     QWidget* parent) :
	QLineEdit(p->getAsString().c_str(), parent), property(p) {
	connect(this, SIGNAL(editingFinished ()), this, SLOT(slotSetValue()));
}

void PropertyDelegate_LineEdit::slotSetValue() {
	property->setFromString(text().toStdString());
}

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_String : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_String)
public:

	virtual SupportedTypes supportedTypes() const {
		return makeSupportedType<std::string>(SupportedType::TEXT |
		                                      SupportedType::EDITOR);
	}

	virtual std::string getText(const PropertyNode* property) {
		// special treatment if we have an enumeration
		if(!property->getHint<std::string>("enumeration").empty()) {
			std::string enumTexts = property->getHint<std::string>("enumeration");
			return getEnumerationString<QString>(property, enumTexts);
		}

		return property->getAsString();
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent) {
		// special treatment if we have an enumeration
		if(!property->getHint<std::string>("enumeration").empty()) {
			std::string enumTexts = property->getHint<std::string>("enumeration");
			return createEnumerationEditor<QString>(property, parent, enumTexts);
		}

		PropertyDelegate_LineEdit* e =
				new PropertyDelegate_LineEdit(property, parent);
		e->setFrame(false);
		return e;
	}
};

///////////////////////////////////////////////////////////////////////////////

// Return an icon containing a check box indicator
static QIcon drawCheckBox(bool value)
{
	QStyleOptionButton opt;
	opt.state |= value ? QStyle::State_On : QStyle::State_Off;
	opt.state |= QStyle::State_Enabled;
	const QStyle *style = QApplication::style();

	// Figure out size of an indicator and make sure it is not scaled down in
	// a list view item by making the pixmap as big as a list view icon and
	// centering the indicator in it.
	// (if it is smaller, it can't be helped)
	const int indicatorWidth = style->pixelMetric(QStyle::PM_IndicatorWidth, &opt);
	const int indicatorHeight = style->pixelMetric(QStyle::PM_IndicatorHeight, &opt);
	const int listViewIconSize = indicatorWidth;
	const int pixmapWidth = indicatorWidth;
	const int pixmapHeight = qMax(indicatorHeight, listViewIconSize);

	opt.rect = QRect(0, 0, indicatorWidth, indicatorHeight);
	QPixmap pixmap = QPixmap(pixmapWidth, pixmapHeight);
	pixmap.fill(Qt::transparent);
	{
		const int xoff = (pixmapWidth  > indicatorWidth)  ? (pixmapWidth  - indicatorWidth)  / 2 : 0;
		const int yoff = (pixmapHeight > indicatorHeight) ? (pixmapHeight - indicatorHeight) / 2 : 0;
		QPainter painter(&pixmap);
		painter.translate(xoff, yoff);
		style->drawPrimitive(QStyle::PE_IndicatorCheckBox, &opt, &painter);
	}
	return QIcon(pixmap);
}

PropertyDelegate_CheckBox::PropertyDelegate_CheckBox(PropertyNode* p,
                                                     QWidget* parent) :
	QCheckBox(parent)
{
	property = p->toTyped<bool>();
	assert(property!=NULL);
	setChecked(property->get());
	setText(property->getAsString().c_str());

	// hm, which one? clicked or toggled?
	connect(this, SIGNAL(clicked(bool)), this, SLOT(slotSetValue(bool)));
	//connect(this, SIGNAL(toggled(bool)), this, SLOT(slotSetValue(bool)));
}

void PropertyDelegate_CheckBox::slotSetValue(bool value)
{
	property->set(value);
	setText(property->getAsString().c_str());
}

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_Bool : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_Bool)
public:

	virtual SupportedTypes supportedTypes() const {
		return makeSupportedType<bool>(SupportedType::TEXT |
		                               SupportedType::ICON |
		                               SupportedType::EDITOR);
	}

	virtual std::string getText(const PropertyNode* property) {
		return property->getAsString();
	}

	virtual QIcon getIcon(const PropertyNode* property) {
		const TypedPropertyNode<bool>* p = property->toTyped<bool>();
		assert(p!=NULL);
		return drawCheckBox(p->get());
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent) {
		PropertyDelegate_CheckBox* e =
				new PropertyDelegate_CheckBox(property, parent);
		return e;
	}
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_Path : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_Path)
public:

	virtual SupportedTypes supportedTypes() const {
		return makeSupportedType<Path>(SupportedType::TEXT |
		                               SupportedType::EDITOR |
		                               SupportedType::DIALOG);
	}

	virtual std::string getText(const PropertyNode* property) {
		return property->getAsString();
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent) {
		PropertyDelegate_LineEdit* e =
				new PropertyDelegate_LineEdit(property, parent);
		e->setFrame(false);
		return e;
	}

	virtual bool execDialog(PropertyNode* property, QWidget* parent) {
		QString dir;
		try { // start open dialog in the path of the currently selected file/folder
			dir = Path(getText(property)).parent_path().c_str();
		}
		catch(...) {}
				
		QString path;
		if(property->hasHint("file")) {
			std::string filter = property->getHint<std::string>("file");
			if (property->hasHint("save") && property->getHint<bool>("save")) // if 'save' == true
				path = QtUtils::getSaveFileName(parent, QString(), dir, QString(filter.c_str()));
			else
				path = QtUtils::getOpenFileName(parent, QString(), dir, QString(filter.c_str()));
		} else {
			path = QtUtils::getExistingDirectory(parent, QString(), dir);
		}

		if(path.isNull() || path.isEmpty())
			return false; // canceled

		property->setFromString(path.toStdString());
		return true;
	}
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::PropertyDelegate_Integers,mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::PropertyDelegate_Floats, mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::PropertyDelegate_String, mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::PropertyDelegate_Bool, mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::PropertyDelegate_Path, mira::PropertyEditor::Delegate);
