/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MoveFrameTool.C
 *    Implementation of MoveFrameTool.h.
 *
 * @author Tim Langner
 * @date   2014/01/14
 */

#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreSceneManager.h>

#include <QMouseEvent>

#include <transform/Pose.h>

#include <visualization/3d/SceneQueryTool.h>
#include <visualization/3d/RectObject.h>

#include <widgets/OgreUtils.h>

namespace mira { namespace mapping {

///////////////////////////////////////////////////////////////////////////////

/**
 * Tool plugin for miracenter to move a frame around on the xy plane of a 3d view.
 */
class MoveFrameTool : public SceneQueryTool
{
	MIRA_META_OBJECT(MoveFrameTool,
		("Name", "MoveFrame")
		("Category", "Interact")
		("Description", "Move a frame around the xy plane")
		("Default", "false")
		("Order", "300"))

public:
	MoveFrameTool() :
		mRectSize(0.2f, 0.2f) {}

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SceneQueryTool);
		ChannelProperty<Pose2>::channelProperty(r, "Frame", mFrame, "The frame to move", this);
		r.member("FramePoses", mFramePoses,
		         "Persistent storage of all the frames set by this MoveFrameTool", std::map<std::string, Pose2>());
	}

	virtual void init(IVisualizationSite* site)
	{
		VisualizationTool3D::init(site);
		restoreFramePoses();
	}

	/** @name Implementation of VisualizationTool */
	//@{

	virtual QIcon getIcon()
	{
		QPixmap pixmap(":/icons/ToolGoal.png");
		return QIcon(pixmap);
	}

	//@}

protected:

	void activate()
	{
		mRectObject->setVisible(true);
	}

	/// called when the user has deactivated the tool by selecting another one.
	void deactivate()
	{
		mRectObject->setVisible(false);
	}

	virtual void onMousePressed(QMouseEvent* e)
	{
		mPose = Pose2();
		if (getSite())
		{
			try
			{
				mPose=getTransform<Pose2>(mFrame.getID(), getSite()->getCameraFrame());
			}
			catch (...)
			{}
		}

		auto i = getIntersection(e->posF().x(), e->posF().y());
		if (i.first.empty())
			return;
		mRectObject->setColor(Ogre::ColourValue::Green);
		mMousePressedPos = i.second;
		if (e->buttons() & Qt::LeftButton)
		{
			OgreUtils::setPosition(mRectNode, Point2f(i.second.x, i.second.y));
			mPose.x() = i.second.x;
			mPose.y() = i.second.y;
			publishPose();
		}
		else if (e->buttons() & Qt::RightButton)
		{
			mStart = mPose;
		}
	}

	virtual void onMouseReleased(QMouseEvent* e)
	{
		mRectObject->setColor(Ogre::ColourValue::Black);
	}

	virtual void onMouseMoved(QMouseEvent* e)
	{
		auto i = getIntersection(e->posF().x(), e->posF().y());
		if (i.first.empty())
			return;

		if (e->buttons() & Qt::LeftButton)
		{
			OgreUtils::setPosition(mRectNode, Point2f(i.second.x, i.second.y));
			mPose.x() = i.second.x;
			mPose.y() = i.second.y;
			publishPose();
		}
		else if (e->buttons() & Qt::RightButton)
		{
			Ogre::Vector3 diff = i.second-mMousePressedPos;
			mPose.phi() = mStart.phi() + std::atan2(diff.y, diff.x);
			OgreUtils::setOrientation(mRectNode, mPose.r);
			publishPose();
		}
	}

	virtual void setupScene(IVisualization3DSite* site)
	{
		SceneQueryTool::setupScene(site);
		mRectNode = site->getSceneManager()->getRootSceneNode()->createChildSceneNode();
		mRectObject.reset(new RectObject(site->getSceneManager(), mRectNode));
		mRectObject->setRect(Rect2f(Point2f(-0.1f, -0.1f), mRectSize));
		mRectObject->setVisible(false);
	}

	std::pair<std::string, Ogre::Vector3> getIntersection(float x, float y,
	                                                      const std::string& mask = "")
	{
		return mask.empty() ? SceneQueryTool::getIntersection(x, y, "ImageObject")
	                        : SceneQueryTool::getIntersection(x, y, mask, true);
	}

	void publishPose()
	{
		publishTransformIndirect<Pose2>(mFrame.getID(), mFrame.getID(),
										getSite()->getCameraFrame(), mPose, Time::now());
		mFramePoses[mFrame.getID()] = mPose;
	}

	void restoreFramePoses()
	{
		foreach (auto p, mFramePoses)
			publishTransform<Pose2>(p.first, p.second, Time::now());
	}

protected:

	Ogre::SceneNode* mRectNode;
	boost::shared_ptr<RectObject> mRectObject;
	Size2f mRectSize;

	ChannelProperty<Pose2> mFrame;
	std::map<std::string, Pose2> mFramePoses;

	Pose2 mStart;
	Pose2 mPose;
	Ogre::Vector3 mMousePressedPos;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::mapping::MoveFrameTool, mira::VisualizationTool3D)

