/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DynamicRenderable.C
 *    Base class providing dynamically growing hardware buffer
 *    Adapted from http://www.ogre3d.org/tikiwiki/DynamicGrowingBuffers&structure=Cookbook
 *
 *    Original file: DynamicRenderable.cpp
 */

#include <visualization/3d/DynamicRenderable.h>

#include <OGRE/OgreCamera.h>
#include <OGRE/OgreHardwareBufferManager.h>

#include <error/Logging.h>
#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

DynamicRenderable::DynamicRenderable()
{
}

DynamicRenderable::~DynamicRenderable()
{
	delete mRenderOp.vertexData;
	delete mRenderOp.indexData;
}

void DynamicRenderable::initialize(
		Ogre::RenderOperation::OperationType operationType,
		const Ogre::VertexDeclaration& vertexDecl, bool useIndices)
{
	// Initialize render operation
	mRenderOp.operationType = operationType;
	mRenderOp.useIndexes = useIndices;
	mRenderOp.vertexData = new Ogre::VertexData;
	if (mRenderOp.useIndexes)
		mRenderOp.indexData = new Ogre::IndexData;

	// Reset buffer capacities
	mVertexBufferCapacity = 0;
	mIndexBufferCapacity = 0;

	// set the vertex declaration
	*mRenderOp.vertexData->vertexDeclaration = vertexDecl;
}

void DynamicRenderable::setOperationType(Ogre::RenderOperation::OperationType operationType)
{
	mRenderOp.operationType = operationType;
	// rest of our buffers stays valid!
}

void DynamicRenderable::enableIndexBuffer(bool enable)
{
	if(enable==true && mRenderOp.indexData==NULL)
		MIRA_THROW(XInvalidParameter, "You cannot enable the index buffer, if 'useIndices' was not specified in the initialize() call");
	mRenderOp.useIndexes = enable;
}

void DynamicRenderable::prepareBuffers(size_t vertexCount, size_t indexCount, Flags flags, Ogre::HardwareBuffer::Usage usage)
{
	// Prepare vertex buffer
	size_t newVertCapacity = mVertexBufferCapacity;
	if ((vertexCount > mVertexBufferCapacity) || (!mVertexBufferCapacity))
	{
		// vertexCount exceeds current capacity!
		// It is necessary to reallocate the buffer.

		// Check if this is the first call
		if (!newVertCapacity)
			newVertCapacity = 1;

		// Make capacity the next power of two
		while (newVertCapacity < vertexCount)
			newVertCapacity <<= 1;
	}
	else
		if (vertexCount < mVertexBufferCapacity >> 1)
		{
			// Make capacity the previous power of two
			while (vertexCount < newVertCapacity >> 1)
				newVertCapacity >>= 1;
		}
	if (newVertCapacity != mVertexBufferCapacity)
	{
		mVertexBufferCapacity = newVertCapacity;
		// Create new vertex buffer
		Ogre::HardwareVertexBufferSharedPtr vbuf =
				Ogre::HardwareBufferManager::getSingleton().createVertexBuffer(
						mRenderOp.vertexData->vertexDeclaration->getVertexSize(
								0), mVertexBufferCapacity, usage);

		// Bind buffer
		mRenderOp.vertexData->vertexBufferBinding->setBinding(0, vbuf);
	}
	// Update vertex count in the render operation
	mRenderOp.vertexData->vertexCount = vertexCount;

	if (mRenderOp.useIndexes)
	{
		if(!(flags & INDEX_BUFFER_32BIT))
			OgreAssert(indexCount <= std::numeric_limits<unsigned short>::max(), "indexCount exceeds 16 bit");

		size_t newIndexCapacity = mIndexBufferCapacity;
		// Prepare index buffer
		if ((indexCount > newIndexCapacity) || (!newIndexCapacity))
		{
			// indexCount exceeds current capacity!
			// It is necessary to reallocate the buffer.

			// Check if this is the first call
			if (!newIndexCapacity)
				newIndexCapacity = 1;

			// Make capacity the next power of two
			while (newIndexCapacity < indexCount)
				newIndexCapacity <<= 1;

		}
		else
			if (indexCount < newIndexCapacity >> 1)
			{
				// Make capacity the previous power of two
				while (indexCount < newIndexCapacity >> 1)
					newIndexCapacity >>= 1;
			}

		if (newIndexCapacity != mIndexBufferCapacity)
		{
			mIndexBufferCapacity = newIndexCapacity;
			// Create new index buffer
			mRenderOp.indexData->indexBuffer
					= Ogre::HardwareBufferManager::getSingleton().createIndexBuffer(
							(flags & INDEX_BUFFER_32BIT) ? Ogre::HardwareIndexBuffer::IT_32BIT : Ogre::HardwareIndexBuffer::IT_16BIT,
							mIndexBufferCapacity,
							usage);
		}

		// Update index count in the render operation
		mRenderOp.indexData->indexCount = indexCount;
	}
}

void DynamicRenderable::updateVertexCount(size_t vertexCount)
{
	// index = 0 as used in binding (in prepareBuffers)
	if (vertexCount > mRenderOp.vertexData->vertexBufferBinding->getBuffer(0)->getNumVertices()) {
		// throwing an exception here sound better, but somehow this is catched and ignored in miracenter, so no use
		MIRA_LOG(ERROR) << "DynamicRenderable::updateVertexCount(): "
		                   "Trying to set vertex count exceeding buffer size, ignored";
		return;
	}
	mRenderOp.vertexData->vertexCount = vertexCount;
}

Ogre::HardwareVertexBufferSharedPtr DynamicRenderable::getVertexBuffer(unsigned short index)
{
	return mRenderOp.vertexData->vertexBufferBinding->getBuffer(index);
}

Ogre::HardwareIndexBufferSharedPtr DynamicRenderable::getIndexBuffer()
{
	return mRenderOp.indexData->indexBuffer;
}


Ogre::Real DynamicRenderable::getBoundingRadius(void) const
{
	return Ogre::Math::Sqrt(std::max(mBox.getMaximum().squaredLength(),
			mBox.getMinimum().squaredLength()));
}

Ogre::Real DynamicRenderable::getSquaredViewDepth(const Ogre::Camera* cam) const
{
	Ogre::Vector3 vMin, vMax, vMid, vDist;
	vMin = mBox.getMinimum();
	vMax = mBox.getMaximum();
	vMid = ((vMax - vMin) * 0.5) + vMin;
	vDist = cam->getDerivedPosition() - vMid;

	return vDist.squaredLength();
}

///////////////////////////////////////////////////////////////////////////////

}
