/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkGraphTest.C
 *    Test case for the FrameworkGraph.
 *
 * @author Tim van der Grinten
 * @date   2017/10/18
 */


#include <boost/test/unit_test.hpp>
#include <fw/FrameworkGraph.h>

using namespace mira;

BOOST_AUTO_TEST_CASE(MultiFrameworkGraphTest)
{
	const char* argv[] = {"MultiFrameworkGraphTest",
	                      "${find MultiFrameworkGraphTest.xml}",
	                      "--no-colors"};
	Framework fw(3,(char**)argv);
	fw.load();
	fw.start();

	MIRA_SLEEP(2500); // give the framework time to boot up

	FrameworkGraph<> mGraph;
	mGraph.discover();


	// check frameworks
	std::cout << "Frameworks:" << std::endl;
	foreach(auto f, mGraph.getFrameworks())
	{
		std::cout << "  " << f->name << std::endl;
	}
	BOOST_CHECK_EQUAL(mGraph.getFrameworks().size(), 2);


	// check namespaces
	std::cout << "Namespaces:" << std::endl;
	foreach(auto n, mGraph.getNamespaces())
	{
		std::cout << "  " << n->ns << std::endl;
	}
	BOOST_CHECK_EQUAL(mGraph.getNamespaces().size(), 6);


	// check authorities
	std::cout << "Authorities:" << std::endl;
	foreach(auto a, mGraph.getAuthorities())
	{
		std::cout << "  " << a->desc.id << std::endl;
	}
	BOOST_CHECK_EQUAL(mGraph.getAuthorities().size(), 6);

	auto s = std::find_if(mGraph.getAuthorities().begin(), mGraph.getAuthorities().end(),
	                      [](const FrameworkGraph<>::AuthorityNodePtr& p){ return p->desc.id == "Synchronizer"; });
	BOOST_REQUIRE(s != mGraph.getAuthorities().end());
	BOOST_CHECK_EQUAL((*s)->getFramework()->name, std::string("local"));

	auto i1 = std::find_if(mGraph.getAuthorities().begin(), mGraph.getAuthorities().end(),
	                       [](const FrameworkGraph<>::AuthorityNodePtr& p){ return p->desc.id == "IntPublisher1"; });
	BOOST_REQUIRE(i1 != mGraph.getAuthorities().end());
	BOOST_CHECK_EQUAL((*i1)->getFramework()->name, std::string("127.0.0.1:1234"));

	auto i2 = std::find_if(mGraph.getAuthorities().begin(), mGraph.getAuthorities().end(),
	                       [](const FrameworkGraph<>::AuthorityNodePtr& p){ return p->desc.id == "IntPublisher2"; });
	BOOST_REQUIRE(i2 != mGraph.getAuthorities().end());
	BOOST_CHECK_EQUAL((*i2)->getFramework()->name, std::string("127.0.0.1:1234"));

	auto i3 = std::find_if(mGraph.getAuthorities().begin(), mGraph.getAuthorities().end(),
	                       [](const FrameworkGraph<>::AuthorityNodePtr& p){ return p->desc.id == "IntPublisher3"; });
	BOOST_REQUIRE(i3 != mGraph.getAuthorities().end());
	BOOST_CHECK_EQUAL((*i3)->getFramework()->name, std::string("127.0.0.1:1234"));


	// check channels
	std::cout << "Channels:" << std::endl;
	foreach(auto c, mGraph.getChannels())
	{
		std::cout << "  " << c->id << std::endl;
	}
	BOOST_CHECK_EQUAL(mGraph.getChannels().size(), 3);
}
