/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file NamespaceLoader.C
 *    Implementation of a NamespaceLoader.
 *
 * @author Erik Einhorn
 * @date   2010/09/16
 */

#include <fw/ResourceName.h>

#include <loader/Loader.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin used to load namespaces.
 */
class NamespaceLoader : public ConfigurationLoaderPlugin
{
	MIRA_OBJECT(NamespaceLoader)
public:
	virtual std::vector<std::string> getTags() const {
		return boost::assign::list_of("namespace");
	}

	virtual void startDocument(ConfigurationLoader* loader);
	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader);
};

void NamespaceLoader::startDocument(ConfigurationLoader* loader)
{
	loader->getContext()["namespace"] = "/";
}

void NamespaceLoader::parseNode(const XMLDom::const_iterator& node,
                                ConfigurationLoader* loader)
{
	if ( loader->getContext().count("unit_id") )
		MIRA_THROW(XInvalidConfig, "Error in <namespace>. "
		           "Namespaces cannot be declared inside <unit> tags ('"
		           << loader->getContext()["unit_id"] << "'");

	auto iName = node.find_attribute("name");
	if (iName == node.attribute_end())
		MIRA_THROW(XInvalidConfig, "Error in <namespace>. "
		           "No name attribute specified. "
		           "Usage <namespace name=\"MyNamespace\">...</namespace>.");

	std::string  name = iName.value();
	ResourceName ns  = loader->getContext()["namespace"];

	// enter the namespace
	ResourceName innerNs = ns / ResourceName(name);
	loader->getContext()["namespace"] = innerNs;

	// parse the content
	loader->parse(node);

	// leave the inner namespace and go back to previous namespace
	loader->getContext()["namespace"] = ns;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::NamespaceLoader, mira::ConfigurationLoaderPlugin)
