/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteServer.h
 *    Server that accepts connections from remote frameworks.
 *
 * @author Tim Langner
 * @date   2010/11/18
 */

#ifndef _MIRA_REMOTESERVER_H_
#define _MIRA_REMOTESERVER_H_

#include <set>

#include <communication/IOService.h>
#include <platform/Types.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// forward declaration
class RemoteIncomingConnection;

/**
 * Server class handling incoming connections from remote frameworks
 */
class RemoteServer
{
public:

	/// Constructs a server that listens on the given port
	RemoteServer(uint16 port);

	/// Destructor. Stops server.
	~RemoteServer();

	/**
	 * Connection handler. Gets called when a client connects.
	 * @param[in] error The error flag
	 */
	void handleAccept(const boost::system::error_code& error);

	void startListening();

	/**
	 * Runs the server with the given number of threads.
	 * @note When using TCPAsyncConnection as connection class, all clients
	 *       will get the same underlying io_service object. This means all
	 *       clients share the same thread(s). Keep in mind to start with more
	 *       threads if you expect many connections at a time.
	 * @param[in] threadCount The number of threads to launch
	 * @param[in] runInThisThread If true this method will block until the server stops.
	 */
	void run(std::size_t threadCount=0, bool runInThisThread=true);

	/**
	 * Stop the server.
	 */
	void stop();

	/**
	 * Handler that gets called whenever a client disconnects.
	 * @param[in] connection The pointer of the disconnected connection.
	 */
	void onDisconnect(RemoteIncomingConnection* connection);

	/**
	 * Returns the port of the server.
	 */
	uint16 getPort() const
	{
		return mAcceptor.local_endpoint().port();
	}

protected:
	IOService mService;
	boost::asio::ip::tcp::acceptor mAcceptor;
	std::set<RemoteIncomingConnection*> mClients;
	RemoteIncomingConnection* mPendingConnection;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
