/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteConnection.h
 *    Connection classes representing connections between frameworks.
 *
 * @author Tim Langner
 * @date   2010/11/16
 */

#ifndef _MIRA_REMOTECONNECTION_H_
#define _MIRA_REMOTECONNECTION_H_

#ifndef Q_MOC_RUN
#include <boost/array.hpp>
#include <boost/make_shared.hpp>
#endif

#include <utils/UUID.h>
#include <communication/IOService.h>
#include <error/Exceptions.h>

#include <rpc/RPCManager.h>

#include <fw/RemoteAuthority.h>
#include <fw/AuthorityDescription.h>
#include <fw/Channel.h>
#include <fw/FrameworkMessage.h>
#include <fw/ServiceLevel.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// forward declaration
class MIRA_FRAMEWORK_EXPORT Authority;
class MIRA_FRAMEWORK_EXPORT MicroUnit;

/**
 * Informations about a known remote framework
 */
struct KnownFramework
{
	KnownFramework() :
		address(""),
		keep(true),
		lastConnectionTry(Time::unixEpoch()),
		forcePTP(false),
		binaryFormatVersion(BinaryBufferSerializer::getSerializerFormatVersion()) {}

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Address", address, "");
		r.roproperty("Address", address, "The adress IP:port");

		r.member("KeepConnected", keep, "", true);
		r.roproperty("KeepConnected", keep, "Whether to reconnect after a lost connection");

		r.member("ForcePTP", forcePTP, "", false);
		r.roproperty("ForcePTP", forcePTP, "Whether to force PTP time sync for this connection");

		r.member("BinaryFormatVersion", binaryFormatVersion, "",
		         BinaryBufferSerializer::getSerializerFormatVersion());
		r.roproperty("BinaryFormatVersion", binaryFormatVersion,
		             "Binary format version used by the remote framework");

		r.member("LastConnectionTry", lastConnectionTry, "",
		         serialization::IgnoreMissing());		
		r.roproperty("LastConnectionTry", lastConnectionTry,
		             "Last time tried to establish connection (UTC)");
	}

	void validate() const;
	boost::tuple<std::string, uint16> getHostPort() const;

	/// address in the form of host:port
	std::string address;

	/**
	 * if true the information is stored in list of frameworks
	 * that we try to reconnect to after disconnect
	 */ 
	bool keep;

	/**
	 * The last time we tried to connect to that address. Used for
	 * waiting at least some seconds before trying to connect again.
	 */
	Time lastConnectionTry;

	/// force PTP time sync
	bool forcePTP;

	/**
	 * The binary format used by the framework
	 * (to enable connecting to legacy framework).
	 */	
	uint8 binaryFormatVersion;
};

template <typename SerializerTag>
class IsTransparentSerializable<KnownFramework, SerializerTag> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

class TimeOffsetCompensation
{
public:
	TimeOffsetCompensation(const TimeOffsetCompensation& other)
		: mCompensationInterval(other.mCompensationInterval),
		  mTargetOffset(other.mTargetOffset),
		  mStartOffset(other.mStartOffset),
		  mStartTime(other.mStartTime) {}

	explicit TimeOffsetCompensation(const Duration& compensationInterval = Duration::seconds(10))
		: mCompensationInterval(compensationInterval.totalMicroseconds()),
		  mTargetOffset(Duration::invalid()),
		  mStartOffset(Duration::invalid()),
		  mStartTime(Time::invalid()) {}

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("CompensationInterval",
		           getter(&TimeOffsetCompensation::getCompensationInterval, this),
		           setter(&TimeOffsetCompensation::setCompensationInterval, this),
		           "Interval to reach target offset");
		r.roproperty("TargetOffset", mTargetOffset, "Target offset");
		r.roproperty("StartOffset", mStartOffset, "Start offset");
		r.roproperty("StartTime", mStartTime, "Time when start offset was valid");
		r.roproperty("CurrentOffset",
		             getter(&TimeOffsetCompensation::queryOffset, this),
		             "Current offset (interpolated between start and target offset)");
	}

public:
	bool isInitialized()const { return mTargetOffset.isValid(); }

public:
	Duration getCompensationInterval() const;
	void setCompensationInterval(const Duration& interval);

	void setTargetOffset(const Duration& target, const Time& ts = Time::now());

public:
	Duration queryOffset(const Time& ts) const;
	Duration queryOffset() const { return queryOffset(Time::now()); } // for use in getter

public:

	// assignment operator uses copy constructor (through by-value argument)
	TimeOffsetCompensation& operator=(TimeOffsetCompensation other) {
		using std::swap;
		swap(mCompensationInterval, other.mCompensationInterval);
		swap(mTargetOffset, other.mTargetOffset);
		swap(mStartOffset, other.mStartOffset);
		swap(mStartTime, other.mStartTime);
		return *this;
	}

	// "Interface adaptors" to support assign from and query as Duration
	TimeOffsetCompensation& operator=(const Duration& target) {
		setTargetOffset(target);
		return *this;
	}

	operator Duration() const { return queryOffset(); }

protected:

	Duration offset(const Time& ts) const;

protected:

	int mCompensationInterval; // total microseconds
	Duration mTargetOffset;
	Duration mStartOffset;
	Time mStartTime;

	mutable boost::mutex mMutex;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class of connections between frameworks
 */
class RemoteConnection : public Object
{
	MIRA_OBJECT(RemoteConnection)
public:

	typedef std::map<std::string, Typename> ChannelTypeMap;
	typedef std::set<std::string> StringSet;
	typedef std::map<std::string, uint8> ChannelSendMap;
	typedef std::set<std::string> MetaSet;
	typedef std::list<AuthorityDescription> AuthorityDescriptions;

	///////////////////////////////////////////////////////////////////////////

	/**
	 * RPCHandler for getting notified when an rpc call on server side is
	 * finished and the response is ready.
	 * Will send the response back to the calling framework using the
	 * connection pointer.
	 */
	class RPCRemoteFinishHandler : public RPCManager::RemoteFinishHandler
	{
	public:
		/// Constructor taking the connection pointer
		RPCRemoteFinishHandler(RemoteConnection* iConnection) :
			mConnection(iConnection)
		{}

		/**
		 * Implementation of RPCManager::RemoteFinishHandler
		 * Will send answer back to calling framework using the internal
		 * connection pointer (if set)
		 */
		virtual void onRPCfinished(Buffer<uint8>&& answer);

		/// Set the connection pointer. Used to reset the connection on disconnect
		void setConnection(RemoteConnection* iConnection);

	protected:

		boost::mutex mConnectionMutex;
		RemoteConnection* mConnection;
	};

	///////////////////////////////////////////////////////////////////////////

	/**
	 * RPCHandler for sending a rpc call to the server side.
	 * Will send the request to the framework that will process the rpc call
	 * using the connection pointer.
	 */
	class RPCRemoteRequestHandler : public RPCManager::RemoteRequestHandler
	{
	public:
		/// Constructor taking the connection pointer
		RPCRemoteRequestHandler(RemoteConnection* iConnection) :
			mConnection(iConnection)
		{}

		/**
		 * Implementation of RPCManager::RemoteRequestHandler
		 * Will send request to the framework that will process the call
		 * using the internal connection pointer (if set)
		 */
		virtual void onRPCrequested(Buffer<uint8>&& request);

		/// Set the connection pointer. Used to reset the connection on disconnect
		void setConnection(RemoteConnection* iConnection);

	protected:

		boost::mutex mConnectionMutex;
		RemoteConnection* mConnection;
	};

	///////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a remote connection that uses its own io service.
	 * run()/runThreads() must be called on the service by the subclass!
	 */
	RemoteConnection();

	/**
	 *  Constructs a remote connection that uses a given io service.
	 *  run()/runThreads() must be called on the service by the owner!
	 */
	RemoteConnection(boost::asio::io_service& service);

	/// Destructor
	virtual ~RemoteConnection();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.roproperty("Connection", address, "The remote framework info");
		r.property("TimeSynch", masterSlaveOffset, "PTP time synchronization");
	}

	/// Returns the network socket of this connection
	boost::asio::ip::tcp::socket& getSocket()
	{
		return mSocket;
	}

	/// Returns true if this is a local connection between two frameworks on the same machine
	bool isLocal() const;

	/// Starts the connection. Can be implemented in derived classes.
	virtual void start();

	/// Called when writing to the socket failed. Can be implemented in derived classes.
	virtual void onWriteError(boost::system::system_error& e) {}

	/// Close the socket
	void stop();

	virtual void onStop() {}

	/**
	 * Time synchronization between frameworks.
	 *
	 * For local connected frameworks the method synchronizeFrameworks will be
	 * called. For frameworks on remote hosts sendPTP will be called.
	 */
	void syncTime();

	/**
	 * Sends a PTP command used for time synchronization between frameworks
	 */
	void sendPTP();

	/**
	 * Create a timer to frequently call syncTime.
	 * Time sync must be completed at least once before data is transmitted
	 * (if ping timeout is enabled in the RemoteModule, time sync is automatically
	 *  done in the ping thread and the timer should not be created).
	 */
	void startTimeSync();

	/**
	 * Sends a ping command. This method ensures, that the ping is not send
	 * more than the configured interval in the RemoteModule.
	 */
	void ping();

	/**
	 * Check if the connection incoming ping's are still alive.
	 */
	bool hasPingTimeout() const;

	/**
	 * Notifies the connected framework that we have at least one publisher for
	 * each of the channels in the channels map.
	 */
	virtual void publishChannels(const ChannelTypeMap& channels) = 0;

	/**
	 * Notifies the connected framework that we no longer have a publisher for
	 * the given channel.
	 */
	void unpublishChannel(const std::string& channel);

	/**
	 * Notify the connected remote frameworks that we have a
	 * subscriber for the given channel (assuming it has a publisher for it).
	 */
	virtual void subscribeChannel(const std::string& channelID, const ServiceLevel& serviceLevel) = 0;

	/**
	 * Notifies the connected framework that the authorities in the authorities
	 * map exist in our framework
	 */
	virtual void publishAuthorities(const AuthorityDescriptions& authorities) = 0;

	/**
	 * Notifies the connected framework that the authorities in the authorities
	 * map do not longer exist in our framework
	 */
	virtual void unpublishAuthorities(const AuthorityDescriptions& authorities) = 0;

	/**
	 * Send a request to the connected framework to transfer ownership of a unit
	 * to this framework.
	 */
	void migrateUnit(const std::string& id);

	/**
	 * Check if a authority with given full id exists in the connected
	 * framework
	 */
	bool hasAuthority(const std::string& id) const;

	/**
	 * Notifies the connected framework that the services in the services
	 * set exist in our framework
	 */
	virtual void publishServices(const StringSet& services) = 0;

	/**
	 * Notifies the connected framework that the services in the services
	 * set do not longer exist in our framework
	 */
	virtual void unpublishServices(const StringSet& services) = 0;

	/**
	 * Send data in the buffers to the connected framework
	 * @param[in] buffers The data to send
	 */
	template <typename BufferSequence>
	void write(const BufferSequence& buffers)
	{
		boost::mutex::scoped_lock lock(mWriteMutex);
		if (mStopped)
			return;
		try
		{
			std::size_t bytes = boost::asio::write(mSocket, buffers, boost::asio::transfer_all());
			updateOutgoingStats(bytes);
		}
		catch (boost::system::system_error& e)
		{
			// let the derived class decide what to do with the error
			onWriteError(e);
		}
	}

	/**
	 * Writes a message to the other framework.
	 * This method takes the message type and a variable number of optional
	 * parameters that are sent within the message.
	 */
	void writeMessage(FrameworkMessageType msg) {
		FrameworkMessageHeader header(0, msg);
		boost::array<boost::asio::const_buffer, 1> buffers =
		{{
			header.asBuffer(),
		}};
		write(buffers);
	}

	///@cond INTERNAL

	/**
	 * Generates the writeMessage(FrameworkMessageType msg, param1 ... paramN)
	 * methods for multiple parameter counts using boost preprocessor.
	 */
	#define GEN_WRITEMESSAGE_WRITE(z,n,_) os << p##n;
	#define GEN_WRITEMESSAGE_PARAM_DECL(z,n,_) , const P##n & p##n
	#define GEN_WRITEMESSAGE(z,n,_)                                            \
	template<BOOST_PP_ENUM_PARAMS_Z(z,BOOST_PP_INC(n),typename P)>             \
	void writeMessage(FrameworkMessageType msg                                 \
	  BOOST_PP_REPEAT_ ## z(BOOST_PP_INC(n),GEN_WRITEMESSAGE_PARAM_DECL,nil)) {\
		BinaryBufferOstream::buffer_type buffer;                               \
		BinaryBufferOstream os(&buffer);                                       \
		BOOST_PP_REPEAT_ ## z(BOOST_PP_INC(n),GEN_WRITEMESSAGE_WRITE,nil)      \
		FrameworkMessageHeader header(buffer.size(), msg);                     \
		boost::array<boost::asio::const_buffer, 2> buffers =                   \
		{{                                                                     \
			header.asBuffer(),                                                 \
			boost::asio::buffer(buffer.data(), buffer.size())                  \
		}};                                                                    \
		write(buffers);                                                        \
	}
	BOOST_PP_REPEAT(BOOST_PP_INC(8),GEN_WRITEMESSAGE, nil);
	#undef GEN_WRITEMESSAGE
	#undef GEN_WRITEMESSAGE_WRITE
	#undef GEN_WRITEMESSAGE_PARAM_DECL

	///@endcond

	/**
	 * Writes a message to the other framework.
	 * Instead of writing parameters into a buffer for generating the message,
	 * this method expects a buffer as parameter directly, e.g. created by serializing data.
	 */
	template <typename BufferType>
	void writeMessageFromBuffer(FrameworkMessageType msg, const BufferType& buffer) {
		FrameworkMessageHeader header(buffer.size(), msg);
		boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer(buffer.data(), buffer.size())
		}};
		write(buffers);
	}

	/**
	 * Writes a message to the other framework.
	 * Instead of writing the parameter into a buffer for generating the message,
	 * (as writeMessage does), this method uses the parameter memory location directly.
	 */
	template <typename DataType>
	void writeMessageFromData(FrameworkMessageType msg, const DataType& data) {
		FrameworkMessageHeader header(sizeof(DataType), msg);
		boost::array<boost::asio::const_buffer, 2> buffers =
		{{
			header.asBuffer(),
			boost::asio::buffer((void*)&data, sizeof(DataType))
		}};
		write(buffers);
	}

	/**
	 * Queue an outgoing RPC request or RPC response
	 * to be transmitted in a separate thread.
	 * msg must be either RPC_RESPONSE_MSG or RPC_REQUEST_MSG
	 */
	void queueRPCMessage(FrameworkMessageType msg, Buffer<uint8>&& answer);

	/**
	 * Channel callback method that gets registered on each channel the
	 * connected framework subscribes.
	 * When data in the channel changes it is sent to the connected framework.
	 */
	void valueChanged(ChannelRead<void> value, ServiceLevel& serviceLevel);

	/**
	 * Parses an incoming message (stored in mMessage) and calls the
	 * respective receivedXXX method.
	 */
	void parseMessage();

	KnownFramework address;       ///< The address of the connected framework
	UUID remoteID;                ///< The UUID of the connected framework
	std::string frameworkID;      ///< The ID/Name of the connected framework
	uint32 remoteVersion;         ///< The protocol version of the connected framework
	Authority* authority;         ///< Our authority used for subscribing to data
	ChannelSendMap subscriptions; ///< List of channels the connected framework is subscribed to
	MetaSet sentMetaInformation;  ///< Set of type meta information already sent
	StringSet publishedServices;  ///< List of services of the connected framework

	bool synchronized;            ///< Is the connection fully established (e.g. PTP synchronized)
	TimeOffsetCompensation masterSlaveOffset; ///< The time offset between us and the connected framework

private:

	void init(); // called by constructors to initialize members

protected:

	void receivedPTPFollowUp(uint64 timestamp);
	void receivedPTPDelayResponse(uint64 timestamp);
	void receivedPTPDelayRequest(uint64 timestamp);
	void receivedPTPFinish();
	void receivedSubscribeChannelRequest();
	void receivedUnsubscribeChannelRequest(const std::string& channelID);
	void receivedPublishChannelMsg();
	void receivedUnpublishChannelMsg();
	void receivedPublishAuthorityMsg();
	void receivedUnpublishAuthorityMsg();
	virtual void receivedPublishServiceMsg() = 0;
	void receivedUnpublishServiceMsg();
	void receivedWriteChannelMsg();
	virtual void receivedRPCRequestMsg() = 0;
	virtual void receivedRPCResponseMsg() = 0;
	void receivedRequestMigrationMsg();
	void receivedMigrationMsg();
	void receivedMigrationSinkSuccessMsg();
	void receivedMigrationSinkFailureMsg();
	void receivedMigrationFinishedMsg();
	void receivedTypeMetaMsg();
	void receivedChannelMetaMsg();
	void receivedPingMsg();

	// This method must not be pure virtual: When a remote incoming connection
	// times outs and will be destroyed, some still pending valueChanged()
	// callbacks will be executed, which will call sendData. At this time, the
	// RemoteIncomingConnection is already destroyed (or will be destroyed) and
	// then sendData will be a pure-virtual function.
	virtual void sendData(ChannelRead<void> value, ServiceLevel& serviceLevel) {}

	virtual int addBinaryFormatVersion(Buffer<uint8>& data) = 0;
	void synchronizeFrameworks();
	void updateOutgoingStats(std::size_t size);

	/**
	 * Returns true, if the message (header) is valid, i.e. if the specified
	 * message length does not exceed a max size and the message type is
	 * known. This is used to prevent Denial of Service attacks.
	 */
	bool checkMessageHeader() const;

	void sendConnectDenied(const std::string& msg);

	void sendRPCMessagesThread();
	void processPingThread();
	void checkPingTimeoutThread();

	IOService mService;
	boost::asio::ip::tcp::socket mSocket;

	boost::condition_variable mRPCMessagesCondition;

	boost::mutex mWriteMutex;
	boost::mutex mStopMutex;

	TimerPtr mSyncTimeTimer;

	boost::thread mSendRPCMessagesThread;
	boost::thread mProcessPingThread;
	boost::thread mCheckPingTimeoutThread;

	Time mHeaderReceived;
	FrameworkMessageHeader mHeader;
	Buffer<uint8> mMessage;

	enum AuthState {
		AUTHSTATE_NONE,
		AUTHSTATE_CONNECTING,
		AUTHSTATE_AUTHENTICATING,
		AUTHSTATE_ACCEPTED,
		AUTHSTATE_DENIED,
	};
	AuthState mAuthState;
	std::string mAuthSignMsg; // random message that is used for strong authentication

	Time mPTPSyncSlave;
	Time mPTPSyncMaster;
	Time mPTPDelaySlave;
	Time mPTPDelayMaster;
	Time mLastPTP;

	Time mPingLastSend;
	Time mPingLastReceived;

	boost::shared_ptr<MicroUnit> mMigrationUnit;
	std::string mMigrationNS;
	std::string mMigrationID;
	bool mStopped;

	boost::shared_ptr<RPCRemoteFinishHandler> mRPCFinishHandler;
	boost::shared_ptr<RPCRemoteRequestHandler> mRPCRequestHandler;
	std::map<std::string, boost::shared_ptr<RemoteAuthority>> mRemoteAuthorities;

	typedef std::pair<FrameworkMessageType, Buffer<uint8>> RPCMessage;
	std::list<RPCMessage> mOutgoingRPCMessages;
	boost::mutex mRPCMessagesMutex;
};

template <uint8 BinaryFormatVersion>
class ConcreteRemoteConnection : public virtual RemoteConnection
{
public:
	/// Constructs a remote connection that uses its own io service
	ConcreteRemoteConnection() : RemoteConnection() {}

	/// Constructs a remote connection that uses a given io service
	ConcreteRemoteConnection(boost::asio::io_service& service) : RemoteConnection(service) {}

	/// Destructor
	virtual ~ConcreteRemoteConnection() {};

public:

	virtual void publishChannels(const ChannelTypeMap& channels);
	virtual void subscribeChannel(const std::string& channelID, const ServiceLevel& serviceLevel);
	virtual void publishAuthorities(const AuthorityDescriptions& authorities);
	virtual void unpublishAuthorities(const AuthorityDescriptions& authorities);
	virtual void publishServices(const StringSet& services);
	virtual void unpublishServices(const StringSet& services);
	virtual void sendData(ChannelRead<void> value, ServiceLevel& serviceLevel);
	virtual int addBinaryFormatVersion(Buffer<uint8>& data);

	virtual void receivedPublishServiceMsg();
	virtual void receivedRPCRequestMsg();
	virtual void receivedRPCResponseMsg();

private:

	typedef ConcreteBinarySerializer<BinaryBufferOstream, BinaryFormatVersion> BinarySerializer;
	boost::shared_ptr<BinarySerializer> createBinarySerializer(Buffer<uint8>* buffer)
	{
		return boost::make_shared<BinarySerializer>(buffer);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Connection class for incoming connections.
 */
class RemoteIncomingConnection : public ConcreteRemoteConnection<2>
{
	MIRA_OBJECT(RemoteIncomingConnection)
public:

	RemoteIncomingConnection();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, ConcreteRemoteConnection<2>);
		static const bool outgoing = false;
		r.roproperty("Outgoing", outgoing, "Whether it is an incoming or outgoing connection");
	}

	/// Implementation of RemoteConnection
	virtual void start();

	/// Implementation of RemoteConnection
	virtual void onStop();

	/// Register a callback function that is called when this connection closes
	void setDisconnectCallback(boost::function<void (RemoteIncomingConnection*)> fn)
	{
		mDisconnectedCallback = fn;
	}

	/// Returns a reference to the endpoint of the remote peer
	boost::asio::ip::tcp::endpoint& getEndpoint() {
		return mPeerEndpoint;
	}

protected:

	void handleReadHeader(const boost::system::error_code& error);

	void handleReadMessage(const boost::system::error_code& error);

	boost::function<void (RemoteIncomingConnection*)> mDisconnectedCallback;

protected:

	/// endpoint of the remote peer
	boost::asio::ip::tcp::endpoint mPeerEndpoint;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Connection class for outgoing connections.
 */
class RemoteOutgoingConnectionBase : public virtual RemoteConnection
{
	MIRA_OBJECT(RemoteOutgoingConnectionBase)
public:

	RemoteOutgoingConnectionBase(const KnownFramework& address);

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, RemoteConnection);
		static const bool outgoing = true;
		r.roproperty("Outgoing", outgoing, "Whether it is an incoming or outgoing connection");
	}

	/// Implementation of RemoteConnection
	virtual void start();

	/// Implementation of RemoteConnection
	virtual void onStop();

	/// Implementation of RemoteConnection
	virtual void onWriteError(boost::system::system_error& e);

protected:

	void handleConnect(const boost::system::error_code& error,
	                   boost::asio::ip::tcp::resolver::iterator iterator);

	void handleReadHeader(const boost::system::error_code& error);

	void handleReadMessage(const boost::system::error_code& error);

	std::string mHostName;
	uint16 mPort;
	bool mStopScheduled;
};

template <uint8 BinaryFormatVersion>
class ConcreteRemoteOutgoingConnection : public RemoteOutgoingConnectionBase,
                                         public ConcreteRemoteConnection<BinaryFormatVersion>
{
public:
	ConcreteRemoteOutgoingConnection(const KnownFramework& address)
		: RemoteOutgoingConnectionBase(address) {}
};

typedef ConcreteRemoteOutgoingConnection<2> RemoteOutgoingConnection;
typedef ConcreteRemoteOutgoingConnection<0> RemoteOutgoingConnectionLegacy;

//////////////////////////////////////////////////////////////////////////////

}

#endif
