/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteAuthority.h
 *    An authority class that represents a remote authority that is located
 *    in a connected framework
 *
 * @author Tim Langner, Erik Einhorn, Christof Schröter
 * @date   2011/09/08
 */

#ifndef _MIRA_REMOTEAUTHORITY_H_
#define _MIRA_REMOTEAUTHORITY_H_

#include <fw/AbstractAuthority.h>
#include <fw/FrameworkDefines.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class RemoteAuthority;
class RemoteAuthorityPropertyNode;
class RemoteAuthorityPropertyUpdater;
typedef boost::shared_ptr<class RemoteAuthorityPropertyUpdater> RemoteAuthorityPropertyUpdaterPtr;

class RemoteAuthorityRootPropertyNode : public RootPropertyNode
{
public:

	RemoteAuthorityRootPropertyNode(RemoteAuthority* iAuthority,
	                                RemoteAuthorityPropertyUpdaterPtr iUpdater)
		: authority(iAuthority), updater(iUpdater)
	{}

public:

	const PropertyNode* getRootProperty() const
	{
		if (children().empty())
			return NULL;

		return children().front();
	}

	PropertyNode* getRootProperty()
	{
		const RemoteAuthorityRootPropertyNode* This = this;
		return const_cast<PropertyNode*>(This->getRootProperty());
	}

	const RemoteAuthority* getAuthority() const { return authority; }
	RemoteAuthority* getAuthority() { return authority; }

	const RemoteAuthorityPropertyUpdaterPtr getUpdater() const { return updater; }
	RemoteAuthorityPropertyUpdaterPtr getUpdater() { return updater; }

protected:
	RemoteAuthority* authority;
	RemoteAuthorityPropertyUpdaterPtr updater;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A remote authority is located in another framework. This class acts as an
 * interface or proxy to remote authorities. All calls to this interface will
 * be redirected to the remote instance using RPCs.
 * For the RPC calls that are used internally a limited timeout is used. Hence,
 * some of the methods below will throw an exception, if the operation was not
 * successful.
 * Getting remote properties will be handled in a separate worker thread. That
 * means, that the first get on a remote property will cause an exception
 * ("not available yet"). Subsequent get calls will return the last received
 * value of the property. Each get call on a property will trigger an update,
 * which is executed by the worker thread.
 */
class RemoteAuthority : public AbstractAuthority
{
public:

	RemoteAuthority(const AuthorityDescription& desc);

	virtual ~RemoteAuthority();

public:

	/** @name Implementation of AbstractAuthority */
	//@{

	virtual StatusManager::StatusMap getStatusMap();

	virtual boost::shared_ptr<PropertyNode> getProperties();

	virtual std::set<std::string> getPublishedChannels();

	virtual std::set<std::string> getSubscribedChannels();

	virtual std::set<std::string> getServiceInterfaces();

	virtual ChannelNameMapping getPublishedChannelNames();

	virtual ChannelNameMapping getSubscribedChannelNames();
	
	virtual ChannelStatusMap getSubscribedChannelStatus();

	virtual void start();

	virtual void stop();

	virtual bool hasUnrecoverableFailure() const;

	virtual bool isStarted() const;

	virtual bool isRunning() const;

	//@}

	bool needSynchronizeProperties();
	void synchronizeProperties();

private:

	PropertyTree fetchProperties();

private:

	friend class RemoteAuthorityPropertyUpdater;

	boost::shared_ptr<RemoteAuthorityPropertyNode> mRemoteProperties;
	RemoteAuthorityRootPropertyNode mRemotePropertiesRoot;
	Time mRemotePropertiesTimestamp;
	bool mRemotePropertyValuesQueried;

#if	MIRA_MAJOR_VERSION(MIRA_PROTOCOL_VERSION)<=3
	bool mHasServiceInterfaceQueryMethod;
#endif

	boost::mutex mMutex;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
