/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MicroUnit.h
 *    Base class for all units.
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2010/09/06
 */

#ifndef _MIRA_MICROUNIT_H_
#define _MIRA_MICROUNIT_H_

#include <serialization/Serialization.h>
#include <serialization/GetterSetter.h>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// forward declaration
class MIRA_FRAMEWORK_EXPORT Unit;

/**
 * @ingroup FWModule
 * Units are basic modules of a complex application.
 * Simple units are drivers for cameras, sensors etc.
 * More complex units will be algorithms like a person tracker or a navigator.
 * This class represents a very simple unit.
 */
class MIRA_FRAMEWORK_EXPORT MicroUnit : public Object, public Authority
{
	MIRA_OBJECT(MicroUnit)
public:

	friend class Unit;

	MicroUnit(Flags flags = NORMAL) :
		Authority(flags),
		mNeedRecover(false),
		mRecoverInterval(Duration::seconds(1))
	{}

	/// Destructor checks out our authority
	virtual ~MicroUnit();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Authority);
		r.property("RecoverInterval", mRecoverInterval,
		           "The interval for trying to recover", Duration::seconds(1));
	}

	/**
	 * Gets called by a higher application level to check ourself in.
	 * Will checkin the underlying Authority.
	 * @param ns The namespace our unit lives in.
	 * @param name The name of our unit
	 */
	virtual void checkin(const std::string& ns, const std::string& name);

	/**
	 * Gets called by a higher application level to check ourself in.
	 * Will checkin the underlying Authority by using the given authority
	 * as parent.
	 * @param parent The parent authority
	 * @param ns The namespace our unit lives in.
	 * @param name The name of our unit
	 */
	virtual void checkin(Authority& parent, const std::string& ns,
	                     const std::string& name);

	/// Sets the main property node
	void setPropertyNode(boost::shared_ptr<PropertyNode> node);

	/** @name Implementation of Authority */
	//@{
	/// see Authority::getProperties()
	virtual boost::shared_ptr<PropertyNode> getProperties();

	/**
	 * Starts the unit by performing following steps.
	 * # Start the Units main thread dispatcher (if this is not a sub unit)
	 * # Enable all subscriber callbacks - Unit will receive channel callbacks
	 * # Enable all RPC callbacks - Unit will receive RPC calls if it is a service
	 * # Start all sub units
	 * # Schedule a call to resume() that gets executed in the Units main thread
	 * @note If you have registered Timers you may want to implement resume() to
	 *       start your timers there if you have stopped them in pause()
	 */
	virtual void start();

	/**
	 * Stops the unit by performing following steps. 
	 * # Stops the Units main thread dispatcher (if this is not a sub unit)
	 * # Disable all subscriber callbacks - Unit will no longer receive channel callbacks
	 * # Disable all RPC callbacks - Unit will no longer receive RPC calls if it is a service
	 * # Stop all sub units
	 * # Schedule a call to pause() that gets executed in the Units main thread
	 * @note If you have registered Timers you may want to implement pause() to
	 *       stop your timers there (e.g. for saving performance)
	 */
	virtual void stop();

	//@}

	/**
	 * Notifies this Unit that it is going to be destructed.
	 * Schedules a call to pause() and to finalize() which get executed
	 * in the Units main thread.
	 */
	virtual void destruct();

protected:

	/** @name Interface for derived classes */
	//@{

	/**
	 * Overwrite this method to publish own or subscribe to channels.
	 * Blocking in this function (e.g. waiting for some needed channels to have
	 * valid data) is allowed and will not block other units in your framework.
	 * But note: while staying in that function your call back functions on
	 * subscribed channels will not get called.
	 * Also if you use the Unit class instead the process method will not be
	 * called before leaving this method.
	 * \code
	 * virtual void initialize()
	 * {
	 *     publish<Pose>("Pose");
	 *     // onLaserChanged will not get called until the Int channel has valid data
	 *     subscribe<RangeData>("Laser", &MyUnit::onLaserChanged, this);
	 *     Channel<int> intChannel = subscribe<int>("Int");
	 *     waitForData(intChannel);
	 * }
	 * \endcode
	 */
	virtual void initialize();

	/**
	 * Overload this method in derived class. It gets called every time
	 * the unit is started/resumed (after initialization is completed)
	 * It will be executed in the Units main thread - the same thread
	 * initialize() was called in.
	 * Can be used to start timers that were stopped in pause().
	 */
	virtual void resume();

	/**
	 * Overload this method in derived class. It gets called every time
	 * the unit is stopped/paused.
	 * It will be executed in the Units main thread - the same thread
	 * initialize() was called in.
	 * Can be used to stop timers to save performance.
	 */
	virtual void pause();

	/**
	 * Overload this method in derived class. It gets called right after
	 * pause() when the unit is going to be destructed.
	 * It will be executed in the Units main thread - the same thread
	 * initialize() was called in.
	 */
	virtual void finalize() {}

	/**
	 * Can be called to signal that this unit needs recovery.
	 * In recover mode recover() gets called.
	 * This can be used to reestablish broken connections or implement any
	 * kind of error recovery.
	 * An optional reason for recovery can be specified
	 */
	virtual void needRecovery(const std::string& reason = "");

	/**
	 * Must be called to reset bootup and recover state by signaling that
	 * this unit is fully operational.
	 */
	virtual void operational();

	/**
	 * Overload this method in derived class. It gets called when
	 * recover mode is set via needRecovery().
	 * It will be executed in the Units main thread - the same thread
	 * initialize() was called in.
	 */
	virtual void recover() {};

	/// Return true, if the unit is in recovery mode.
	bool inRecoveryMode() const { return mNeedRecover; }

	//@}

protected:

	void initializeIntern();

	void recoverIntern(const Timer& timer);

protected:
	RootPropertyNode mPropertiesRoot;

private:

	void setup();

	boost::shared_ptr<PropertyNode> mProperties;

	bool mNeedRecover;
	Duration mRecoverInterval;
};

///////////////////////////////////////////////////////////////////////////////

typedef boost::shared_ptr<MicroUnit> MicroUnitPtr;

///////////////////////////////////////////////////////////////////////////////

}

#endif
