/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDomPreprocessor.h
 *    Preprocessor for XML dom documents
 *
 * @author Tim Langner
 * @date   2013/05/31
 */

#ifndef _MIRA_XMLDOMPREPROCESSOR_H_
#define _MIRA_XMLDOMPREPROCESSOR_H_

#include <list>
#include <map>
#ifndef Q_MOC_RUN
#include <boost/function.hpp>
#endif
#include <xml/XMLDom.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// Variables defined in xml documents

struct XMLVariableValue
{
	XMLVariableValue(const std::string& v = "",
	                 const std::string& a = "") : value(v), annotation(a) {}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("value", value, "Variable value");
		r.property("annotation", annotation, "Source annotation");
	}

	XMLVariableValue& operator=(const char* s) { value = s; annotation = ""; return *this;}

	bool operator==(const std::string& s) const { return value == s; }
	bool operator!=(const std::string& s) const { return value != s; }
	bool operator==(const char* s) const { return value == s; }
	bool operator!=(const char* s) const { return value != s; }

	operator std::string() const { return value; }

	friend std::ostream& operator<<(std::ostream& s, const XMLVariableValue& x);

	std::string value;
	std::string annotation;
};

std::ostream& operator<<(std::ostream& s, const XMLVariableValue& x);

typedef std::map<std::string, XMLVariableValue> XMLVariablesMap;

///////////////////////////////////////////////////////////////////////////////

/**
 * Preprocesses XML documents and resolves all special tags
 * like <include>, <if>, <warning> and so on. It also parses variable declarations
 * via the <var> tag. See help about XML config files.
 * The member variables contains a key value variables map that
 * will be altered if an <define> or <var> tag is found
 * and the variable name is not already in the map.
 * It also processes special xml variables like find, findpkg or env.
 * Additionally the user can add callbacks for user defined patterns that are called
 * whenever the pattern is found anywhere in the document.
 */
class MIRA_BASE_EXPORT XMLDomPreprocessor
{
protected:

	struct IteratorPair {
		XMLDom::iterator first, second;
		std::string name;
	};

public:
	typedef boost::function<std::string (const std::string&)> Callback;
	typedef std::list<Path> IncludePathStack;

	XMLDomPreprocessor() {}
	XMLDomPreprocessor(const XMLVariablesMap& iVariables) :
		variables(iVariables) {}

	void preprocessXML(XMLDom& ioXml);
	void registerXMLVariableCallback(const std::string& pattern, Callback&& callback);

	XMLVariablesMap variables;

protected:

	std::string resolveXMLVariables(const std::string& pattern, const std::string& var);
	std::string resolveContent(std::string content);

	void preprocessAllLowLevel(XMLDom::iterator& iNode,
	                           IncludePathStack& ioPathStack);

	void preprocessAll(XMLDom::iterator& iNode,
	                   IncludePathStack& ioPathStack);
	void processSpecialVariables(XMLDom::iterator& node);

	bool parseCondition(XMLDom::iterator& node);
	bool parseConditionsRequireAll(XMLDom::iterator& node);
	bool parseConditionsRequireAny(XMLDom::iterator& node);
	bool parseExists(XMLDom::iterator& node);
	bool parseExistsRequireAll(XMLDom::iterator& node);
	bool parseExistsRequireAny(XMLDom::iterator& node);
	XMLDom::iterator parseInclude(XMLDom::iterator& ioNode, IncludePathStack& ioPathStack);

	std::map<std::string, Callback> mCallbacks;

	bool mAddedInfoToException;
};


/**
 * Shortcut for processing a XML document without creating a XMLDomPreprocessor
 * yourself. It internally creates a XMLDomPreprocessor and calls its preprocessXML method.
 */
void MIRA_BASE_EXPORT preprocessXML(XMLDom& ioXml, XMLVariablesMap& ioVariables);

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
