/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Thread.h
 *    Includes, defines and functions for threads.
 *
 * @author Tim Langner, Christof Schröter
 * @date   2011/06/23
 */

#ifndef _MIRA_THREAD_H_
#define _MIRA_THREAD_H_

#ifndef Q_MOC_RUN
#include <boost/thread/thread.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/recursive_mutex.hpp>
#include <boost/thread/locks.hpp>
#endif

#include <platform/Platform.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup ThreadModule
 * Let the calling thread sleep for the given duration.
 * This is a thread interruption point - if interruption of the thread has
 * been requested, its execution will terminate.
 */
using boost::this_thread::sleep;

/**
 * @ingroup ThreadModule
 * Returns the CPU usage of the calling process in percent.
 */
float MIRA_BASE_EXPORT getCPUUsage();

/**
 * @ingroup ThreadModule
 * An object of this class disables interruption of the current thread
 * during its lifetime.
 */
using boost::this_thread::disable_interruption;

///////////////////////////////////////////////////////////////////////////////

}

/**
 * @ingroup ThreadModule
 * Sleeps for ms milliseconds
 * This is a thread interruption point - if interruption of the current thread
 * has been requested, its execution will terminate.
 * See https://www.boost.org/doc/libs/1_54_0/doc/html/thread/thread_management.html#thread.thread_management.tutorial.interruption
 */
#define MIRA_SLEEP(ms) mira::sleep(mira::Duration::milliseconds(ms));

/**
 * @ingroup ThreadModule
 * Disable interruption of the current thread in the scope where this is placed.
 * See https://www.boost.org/doc/libs/1_54_0/doc/html/thread/thread_management.html#thread.thread_management.this_thread.disable_interruption
 */
#define MIRA_DISABLE_THREAD_INTERRUPT disable_interruption _disable_interruption_USE_ONLY_ONCE_PER_SCOPE_;

/**
 * @ingroup ThreadModule
 * Sleeps for ms milliseconds, thread interruption is disabled for this time
 * (but not beyond).
 */
#define MIRA_SLEEP_NO_INTERRUPT(ms) \
{ \
	MIRA_DISABLE_THREAD_INTERRUPT \
	MIRA_SLEEP(ms) \
}

#endif
