/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ReflectorInterface.h
 *    Contains the base interface of all Reflectors, Serializers, etc.
 *
 * @author Erik Einhorn
 * @date   2010/09/22
 */

#ifndef _MIRA_REFLECTORINTERFACE_H_
#define _MIRA_REFLECTORINTERFACE_H_

#include <string>

#ifndef Q_MOC_RUN
#include <boost/preprocessor/repetition.hpp>
#include <boost/mpl/bool.hpp>
#include <boost/type_traits/integral_constant.hpp>
#endif

#include <platform/Environment.h>
#include <platform/Typename.h>
#include <utils/HasMember.h>

#include <rpc/RPCMacros.h>
#include <serialization/GetterSetter.h>
#include <serialization/PropertyHint.h>
#include <serialization/ReflectControlFlags.h>

//////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace mira { namespace detail {
template<class Type>
struct ____MISSING_REFLECT_METHOD_FOR_{
	static_assert(sizeof(Type)==0,
	    "Cannot find an intrusive or nonintrusive reflect method for Type."
	    "If Type is a class that you have implemented, then you forgot to "
	    "implement its reflect() method. If Type is a common class or "
	    "container from STL, BOOST, etc. you need to include an adapter header "
	    "e.g. #include <serialization/adapters/std/vector>.");
	static void invoke() {}
};
}}

namespace mira {
template <typename Derived>
class ReflectorInterface;
}

// generic reflect function for generating readable compiler errors
// (must stay outside of MIRA namespace)
template<class Derived, class Type>
void reflect( mira::ReflectorInterface<Derived>& reflector, Type& type)
{
	mira::detail::____MISSING_REFLECT_METHOD_FOR_<Type>::invoke();
}
///@endcond

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace serialization {
	typedef uint8 VersionType;
}

/**
 * @ingroup SerializationModule
 *
 * This is the public interface of all reflectors that are able to visit
 * a class' reflect() method. The reflect-method is used to realize some
 * kind of reflection that is known from other programming languages like
 * C# or Java.
 * Within the reflect method a class or actually a certain instance of a class
 * can specify it's members, properties, methods etc. by calling the
 * visitor-methods of the reflector. All available visiting methods are
 * specified in this ReflectorInterface class.
 *
 * Concrete derived Reflectors may then implement a subset of these
 * visitor-methods below to process the given information. E.g. a reflector like
 * the @ref RecursiveMemberReflectorBase that collects information about
 * the class members, will implement all "method"-visitor methods.
 *
 * If new visitors require additional visitor-methods, these methods must
 * be added to this interface with some default behavior to ensure that
 * the existing Reflectors still can be used with all reflect methods
 * (including those that use the new visitor-methods).
 *
 * You should not derive from this class directly. Use @ref AbstractReflector
 * as base class instead, since it implements many important methods.
 *
 * @see @ref SerializationPage
 */
template <typename Derived>
class ReflectorInterface
{
public:
	struct Tag {};

	/**
	 * Specifies, if the Reflector is read-only (true) or write-only (false).
	 * This type must be set to the proper boost::mpl::bool type by the
	 * derived Reflector class. (Default is true)
	 */
	typedef boost::mpl::bool_<true> isReadOnly;

	/**
	 * Specifies, if the Reflector supports human readable IDs.
	 * This type must be set to the proper boost::mpl::bool type by the
	 * derived Reflector class. (Default is true)
	 */
	typedef boost::mpl::bool_<true> useHumanReadableIDs;

	/**
	 * Specifies whether the Reflector uses so-called 'reflect barriers' to
	 * allow maintaining separate states for individual blocks of reflected
	 * data. A reflect barrier is just a virtual separation line declared
	 * by a reflected object. It tells the reflector that certain parts of
	 * the reflected data should be considered as independent and can have
	 * separate states (if the reflector maintains any states for data).
	 * As an example, the BinarySerializer reserves space for a version number
	 * for each reflected object. If there is more than one individual block,
	 * it reserves space for a version for each of the blocks.
	 * If the reflector needs reflect barriers, it defines
	 * requireReflectBarriers::type to true in its class scope, and defines
	 * methods preReflect() and postReflect() that should be called by reflected
	 * objects for separation of independent reflection blocks (where the
	 * separation is not already implied by visitors like reflector.member()!)
	 * (The default is false).
	 * Also see \ref ReflectState and \ref MIRA_REFLECT_CALL.
	 */
	typedef boost::mpl::bool_<false> requireReflectBarriers;

	/**
	 * If 'reflect barriers' are used, each separated reflection block within an
	 * object has a separate state (defined by the Reflector).
	 * When reflection of one block is interrupted to reflect a member block or
	 * call a helper subroutine, the state information refering to the parent block
	 * is pushed to a stack, then popped and restored when the flow of reflection
	 * returns to the parent, ensuring the current state held by the reflector always
	 * corresponds to the block just being reflected.
	 * This stack keeping is done internally for cases where the reflector is made
	 * aware of the transit from one block to the other by member() etc.
	 * If the change results from the reflecting method directly calling another method,
	 * it should call reflector.preReflect() before and keep the returned state locally,
	 * then (after the member/subroutine is finished), give back the state to
	 * reflector.postReflect(), thus emulating a stack in a single local variable kept
	 * on the program stack automatically with each subroutine call (more efficient than
	 * a true stack held internally).
	 * The type of the stacked state is defined by the Reflector. The default state type
	 * is an empty struct.
	 * The MIRA_REFLECT_CALL macro encapsulates the use of preReflect()/postReflect()
	 * for stack-keeping and makes it simple to use this in class reflection.
	 *
	 * Example: BinarySerializer keeps a version placeholder (buffer position) for each
	 * reflection block, and tracks access to make sure a block does not write to it
	 * more than once during serialization. The state for a block therefore consists of
	 * a size_t (position in buffer) and a bool (verson was written).
	 * BinaryDeserializer just stores the int version read from the buffer and a
	 * flag to ensure it is read at least once.
	 */
	struct ReflectState {};

public:
	/**
	 * @name Optional Class Versioning.
	 *    These visiting methods may be supported by different Reflectors like
	 *    the RecursiveMemberReflector and the RPCMethodReflector, etc.
	 *
	 * @note Specifying versions for serialized classes is optional.
	 *
	 * @note requireVersion() and version() currently are supported
	 *       by serialization reflectors only.
	 */
	//@{

	typedef serialization::VersionType VersionType;

	/**
	 * Specifies the current class version and returns the version found in
	 * the data stream. During serialization (read-only reflector) the specified
	 * version usually is stored as class version in the data stream. During
	 * deserialization (write-only reflector) it reads the version found in the
	 * data stream and returns it.
	 * Providing the type of the caller enables defining separate
	 * version numbers for all base classes of an inherited type. Each class in
	 * the inheritance hierarchy can store and retrieve a version number
	 * independently of all its bases or sub-classes (assuming reflect() calls
	 * the reflect() method of the base class using reflectBase()). The sole
	 * purpose of the 'caller' parameter is to enable automatic deduction of the
	 * template parameter T. It can be omitted when T is given explicitly.
	 *
	 * Usage:
	 * \code
	 *    template<typename Reflector>
	 *    void reflect(Reflector& r)
	 *    {
	 *        serialization::VersionType v = r.version(4, this); // provides version 4
	 *        if(v==3) { // handle reading version 3
	 *           ...
	 *        } else // all other versions
	 *          ...
	 *    }
	 * \endcode
	 *
	 * @note Specifying versions for serialized classes is optional.
	 *
	 * @note requireVersion() and version() currently are supported
	 *       by serialization reflectors only
	 */
	template <typename T>
	VersionType version(VersionType version, const T* caller = NULL)
	{
		return version;
	}

	/// Deprecated 'anonymous' (no type) version().
	MIRA_DEPRECATED("Please call as version<MyType>(v) or version(v, this)",
	VersionType version(VersionType version)) { return version; }

	/**
	 * Specifies the current class version and a minimum version for class data
	 * accepted during deserialization, returns the version found in the data stream.
	 * If that version does not meet the specified minimum version, an XIO
	 * exception is thrown at this point. Calls version() internally.
	 *
	 * Usage:
	 * \code
	 *    template<typename Reflector>
	 *    void reflect(Reflector& r)
	 *    {
	 *        r.requireVersion(4, 2, this); // provides version 4, requires version 2 or higher
	 *        if(v==3) { // handle reading version 3
	 *           ...
	 *        } else // all other versions
	 *        ...
	 *    }
	 * \endcode
	 *
	 * @note Specifying versions for serialized classes is optional.
	 *
	 * @note requireVersion() and version() currently are supported
	 *       by serialization reflectors only
	 */
	template <typename T>
	VersionType requireVersion(VersionType version,
	                           VersionType minVersion, const T* caller = NULL) { return version; }

	/// Deprecated 'anonymous' (no type) requireVersion().
	MIRA_DEPRECATED("Please call as requireVersion<MyType>(v, minV) or requireVersion(v, minV, this)",
	VersionType requireVersion(VersionType version, VersionType minVersion)) { return version; }

	/**
	 * Specifies the current class version, which is also the minimum (and therefore only)
	 * version for class data accepted during deserialization. If the version found in
	 * the data stream does not meet the specified version, an XIO
	 * exception is thrown at this point.  Calls version() internally.
	 *
	 * Usage:
	 * \code
	 *    template<typename Reflector>
	 *    void reflect(Reflector& r)
	 *    {
	 *        r.requireVersion(4, this); // provides and requires version 4
	 *        ...
	 *    }
	 * \endcode
	 */
	template <typename T>
	void requireVersion(VersionType version, const T* caller = NULL) {}

	/// Deprecated 'anonymous' (no type) requireVersion().
	MIRA_DEPRECATED("Please call as requireVersion<MyType>(v) or requireVersion(v, this)",
	VersionType requireVersion(VersionType version)) { return version; }

	//@}

	/**
	 * @name Methods for reflecting members
	 * These visiting methods are supported by the RecursiveMemberReflector.
	 */
	//@{

	/**
	 * Specifies a class member for reflection/serialization.
	 * For each member its name (as used in XML files, etc) and a comment that
	 * describes the member must be specified.
	 * @param name    The name of the member (as used in XML files)
	 * @param member  The member that should be reflected/serialized
	 * @param comment A comment that describes the member
	 * @param flags   Flags for controlling the reflection
	 */
	template<typename T>
	void member(const char* name, T& member, const char* comment,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, with an extra parameter for specifying the id explicitly
	 * if it differs from name. (Used by several adapters to STL-containers,
	 * etc.)
	 */
	template<typename T>
	void member(const char* name, const std::string& id, T& member,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, with a special setter method for reading the member.
	 * @see @ref Serialization_GettersSetters
	 */
	template<typename T>
	void member(const char* name, const T& member, Setter<T> setter,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, with a special getter and setter accessing the member.
	 * Since both, getter and setter are specified, the member does not need to be specified
	 * itself. The full access is done using the accessor methods.
	 * @see @ref Serialization_GettersSetters
	 */
	template<typename T>
	void member(const char* name, Getter<T> getter, Setter<T> setter,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}


	// with default value

	/**
	 * Same as above, where a default value can be specified that is used,
	 * if the value is not available in the data source.
	 * @see @ref Serialization_Default
	 */
	template<typename T, typename U>
	void member(const char* name, T& member, const char* comment,
	            const U& defaultValue, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, where a default value can be specified that is used,
	 * if the value is not available in the data source.
	 * @see @ref Serialization_Default
	 */
	template<typename T, typename U>
	void member(const char* name, const T& member, Setter<T> setter,
	            const char* comment, const U& defaultValue,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, where a default value can be specified that is used,
	 * if the value is not available in the data source.
	 * @see @ref Serialization_Default
	 */
	template<typename T, typename U>
	void member(const char* name, Getter<T> getter, Setter<T> setter,
	            const char* comment, const U& defaultValue,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	// Properties

	/**
	 * Same as the corresponding member-method, to indicate that the member
	 * is a property and can be altered at runtime. Using the last optional
	 * parameter certain "property hints" can be specified
	 */
	template<typename T>
	void property(const char* name, T& member, const char* comment,
			             PropertyHint&& hint = PropertyHint(),
			             ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, with an extra parameter for specifying the id explicitly
	 * if it differs from name. (Used by several adapters to STL-containers,
	 * etc.)
	 */
	template<typename T>
	void property(const char* name, const std::string& id, T& member,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as the corresponding member-method, to indicate that the member is
	 * a property and can be altered at runtime.
	 */
	template<typename T>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as the corresponding member-method, to indicate that the member is
	 * a property and can be altered at runtime.
	 */
	template<typename T>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	// default values

	/**
	 * Same as the corresponding member-method, to indicate that the member is
	 * a property and can be altered at runtime.
	 */
	template<typename T, typename U>
	void property(const char* name, T& member, const char* comment,
	              const U& defaultValue, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as the corresponding member-method, to indicate that the member is
	 * a property and can be altered at runtime.
	 */
	template<typename T, typename U>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as the corresponding member-method, to indicate that the member is
	 * a property and can be altered at runtime.
	 */
	template<typename T, typename U>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}


	/**
	 * A property that just reflects a runtime state of the object and can be
	 * displayed but not altered.
	 */
	template<typename T>
	void roproperty(const char* name, const T& member,
	                const char* comment, PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, with an extra parameter for specifying the id explicitly
	 * if it differs from name.
	 */
	template<typename T>
	void roproperty(const char* name, const std::string& id, const T& member,
	                const char* comment, PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Same as above, with a special getter method for reading the member.
	 * @see @ref Serialization_GettersSetters
	 */
	template<typename T>
	void roproperty(const char* name, Getter<T> getter, const char* comment,
	                PropertyHint&& hint = PropertyHint(),
	                ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * If the currently reflected object is an item within a collection, this
	 * allows to specify an explicit name for that item.
	 * This currently is interpreted by the PropertySerializer only.
	 */
	void itemName(const std::string& name) {}

	/**
	 * Delegates the serialization/reflection of a member directly to the
	 * member. Must be used with IsTransparentSerializable and is for most
	 * advanced users only.
	 */
	template<typename T>
	void delegate(T& member,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Delegates the serialization/reflection of a member directly to the
	 * member. Must be used with IsTransparentSerializable and is for most
	 * advanced users only.
	 */
	template<typename T>
	void delegate(const T& member, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	/**
	 * Delegates the serialization/reflection of a member directly to the
	 * member. Must be used with IsTransparentSerializable and is for most
	 * advanced users only.
	 */
	template<typename T>
	void delegate(Getter<T> getter, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

	//@}

	/** @name Methods for reflecting RPC (remote procedure call) methods */
	//@{

	/**
	 * Indicates that the class implements the specified RPC interface.
	 * If you indicate that your class implements the specified interface by
	 * calling this method in your reflect method, there is an implicit contract
	 * that your class provides all methods that are defined by that interface.
	 */
	void interface(const char* name) {}

	#define RPC_METHODS_MAX_PARAMS 8

	// method(name, function<...>, comment)
	#define RPCGEN_METHODS(z,n,_)                                                                        \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                 \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), const char* comment) {}        \
	                                                                                                     \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class*, const char* comment) {} \
	                                                                                                     \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class*, const char* comment) {} \
	                                                                                                     \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                 \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))>, const char* comment) {} \

	BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_METHODS,nil)
	#undef RPCGEN_METHODS

	// method(name, function<...>, comment, paramname, paramdescription, ... )
	#define RPCGEN_METHODS_PARAMDESC(z,n,_)                                                              \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                 \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)),                                \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil)) {}                 \
	                                                                                                     \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class*,                 \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil)) {}                 \
	                                                                                                     \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                 \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class*,           \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil)) {}                 \
	                                                                                                     \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                 \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))>,                    \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil)) {}                 \

	// generate variants with parameter descriptions only for param count > 0 
	BOOST_PP_REPEAT_FROM_TO(1,BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_METHODS_PARAMDESC,nil)
	#undef RPCGEN_METHODS_PARAMDESC

	// method(name, function<...>, comment, paramname, paramdescription, paramsamplevalue, ... )
	#define RPCGEN_METHODS_PARAMDESCSAMPLE(z,n,_)                                                        \
	template<typename R, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>                           \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)),                                \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil)) {}          \
	                                                                                                     \
	template<typename R, typename Class, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>           \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class*,                 \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil)) {}          \
	                                                                                                     \
	template<typename R, typename Class, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>           \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class*,           \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil)) {}          \
	                                                                                                     \
	template<typename R, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>                           \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))>,                    \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil)) {}          \

	// generate variants with parameter descriptions only for param count > 0 
	BOOST_PP_REPEAT_FROM_TO(1,BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_METHODS_PARAMDESCSAMPLE,nil)
	#undef RPCGEN_METHODS_PARAMDESCSAMPLE
	#undef RPCGEN_METHODS_CONSTCHARPTR_NAME_DESC_SAMPLE

	// These are essentially catch-all clauses for mismatched parameter numbers/types. We will just ignore
	// the mismatch here where we don't implement these functions anyway (i.e., subclasses NOT overriding
	// method() with actual implementation will also not complain about invalid optional parameters).
	// We could just include ONLY these catch-alls here instead of the 2 sets above, but they have the extra
	// requirement of compiler support for variadic templates. Keeping them as a separate add-on allows to
	// make them available conditionally if required (their existence just helps getting more informative
	// compiler error messages on wrong use).
	#define RPCGEN_METHODS_WRONG_ARGUMENT_NUMBER(z,n,_)                                                  \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args>               \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)),                                \
	            const char* comment, Args...) {}                                                         \
	                                                                                                     \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args> \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class* This,            \
	            const char* comment, Args...) {}                                                         \
	                                                                                                     \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args> \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class* This,      \
	            const char* comment, Args...) {}                                                         \
	                                                                                                     \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args>               \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))> fn,                 \
	            const char* comment, Args...) {}                                                         \

	BOOST_PP_REPEAT_FROM_TO(1,BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_METHODS_WRONG_ARGUMENT_NUMBER,nil)
	#undef RPCGEN_METHODS_WRONG_ARGUMENT_NUMBER

	// provide a method that causes an error at compile time if instantiated. Derived classes overriding
	// method() should call this in their implementation for wrong argument number
	#define RPCGEN_METHODS_INVALID(z,n,_)                                                                \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                 \
	void invalid_method()                                                                                \
	{    /* force this to fail whenever it is instantiated - but only then! */                           \
		static_assert(sizeof(R*)==0,                                                                     \
		              "_____________NUMBER OR TYPES OF NAMES/DESCRIPTIONS/SAMPLEVALUES FOR PARAMETERS "  \
		              "IN CALL TO Reflector::method() DO NOT MATCH THE PROVIDED FUNCTION'S SIGNATURE. "  \
		              "PLEASE PROVIDE const char* NAME AND DESCRIPTION FOR EACH RPC METHOD PARAMETER, "  \
		              "OR NAME, DESCRIPTION AND A CORRECTLY TYPED SAMPLE VALUE FOR EACH PARAMETER._____________"); \
	}

	BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_METHODS_INVALID,nil)
	#undef RPCGEN_METHODS_INVALID

#ifdef DOXYGEN
	// this is a fake method to generate a doxygen documentation for the
	// methods created above

	/**
	 * Specifies that the class that is reflected provides a service through the specified
	 * static member function or free function.
	 * The method will then be available for RPC calls.
	 * For each method its name (used to identify the method in RPC calls) and
	 * a comment that describes the method must be specified.
	 * @param name             The name of the method (used for calling it)
	 * @param method           The static member function (or free function) that implements the method
	 * @param comment          A comment that describes the method
	 * @param paramName        Optional additional argument(s) for the names of the method's parameters
	 * @param paramDescription Optional additional argument(s) for the descriptions of the method's
	 *                         parameters.
	 * @param paramSampleValue Optional additional argument(s) providing a sample value for each parameter.
	 *                         These are not default values, they are only used for documentation purpose,
	 *                         and should mainly serve to illustrate notation.
	 *
	 * The optional parameter documentation arguments must be either a sequence of pairs of const char* values
	 * or a sequence of triplets, where each third value is a value of the respective method parameter's type.
	 * (name and description must both be provided, for all parameters or for none, and if sample values are to
	 * be added, they also need to be added for all parameters).
	 *
	 * Examples:
	 *
	 * \code
	 *     r.method("myMethod1", myStaticMethod1, "static method that does something");
	 *     r.method("myMethod2", myStaticMethod2, "static method that does something else",
	 *              "x", "1st parameter", "y", "2nd parameter");
	 *     r.method("myMethod3", myStaticMethod3, "yet another static method"
	 *              "a", "parameter 1", ParamType1(...), "b", "parameter b", ParamType2(...));
	 * \endcode
	 */
	void method(const char* name, Method method, const char* comment,
	            const char* paramName, const char* paramDescription, P paramSampleValue, ...);

	/**
	 * Specifies that the class that is reflected provides a service through the specified
	 * non-static member function.
	 * The method will then be available for RPC calls.
	 * For each method its name (used to identify the method in RPC calls) and
	 * a comment that describes the method must be specified.
	 * @param name             The name of the method (used for calling it)
	 * @param method           The non-static member function (const or non-const) that implements the method
	 * @param object           A pointer to the object on which the function is called (usually "this")
	 * @param comment          A comment that describes the method
	 * @param paramName        Optional additional argument(s) for the names of the method's parameters
	 * @param paramDescription Optional additional argument(s) for the descriptions of the method's
	 *                         parameters.
	 * @param paramSampleValue Optional additional argument(s) providing a sample value for each parameter.
	 *
	 * See above for documentation of optional parameter documentation arguments.
	 *
	 * Example:
	 *
	 * \code
	 *     r.method("myMethod2", &MyClass::myMethod, this, "a non-static method");
	 * \endcode
	 */
	void method(const char* name, Method method, Class* object, const char* comment,
	            const char* paramName, const char* paramDescription, P paramSampleValue, ...);

	/**
	 * Specifies that the class that is reflected provides a service through the specified
	 * function wrapper.
	 * The method will then be available for RPC calls.
	 * For each method its name (used to identify the method in RPC calls) and
	 * a comment that describes the method must be specified.
	 * @param name             The name of the method (used for calling it)
	 * @param method           The function wrapper object that implements the method,
	 *                         e.g. created using boost::bind
	 * @param comment          A comment that describes the method
	 * @param paramName        Optional additional argument(s) for the names of the method's parameters
	 * @param paramDescription Optional additional argument(s) for the descriptions of the method's
	 *                         parameters.
	 * @param paramSampleValue Optional additional argument(s) providing a sample value for each parameter.
	 *
	 * See above for documentation of optional parameter documentation arguments.
	 *
	 * Example:
	 *
	 * \code
	 *     r.method("myMethod3", boost::bind(&MyClass::myMethod, this, 1.23, 4, _1), "a function wrapper");
	 * \endcode
	 */
	void method(const char* name, Method method, const char* comment,
	            const char* paramName, const char* paramDescription, P paramSampleValue, ...);

#endif

	//@}

	/**
	 * @name Controlling the object tracking state.
	 */
	//@{

	/// Push the current object tracking memory to an internal stack (backup).
	void pushObjectTrackingStore() {}

	/// Pop the current object tracking memory from internal stack (restore).
	void popObjectTrackingStore() {}

	//@}

	/**
	 * Method that can be called to reflect the base class easily.
	 *
	 * Usage:
	 *
	 * You can use the @ref MIRA_REFLECT_BASE macro:
	 *
	 * \code
	 * template<typename Reflector>
	 * void reflect(Reflector& r)
	 * {
	 *     MIRA_REFLECT_BASE(r, MyBaseClass);
	 * }
	 * \endcode
	 *
	 * or either call
	 *
	 * \code
	 * reflector.reflectBase((BaseClass&)*this);
	 * \endcode
	 *
	 * or use
	 *
	 * \code
	 * reflector.template reflectBase<BaseClass>(*this);
	 * \endcode
	 *
	 * The method can also be used in non-intrusive reflection:
	 *
	 * \code
	 * template<typename Reflector>
	 * void reflect(Reflector& r, MyClass& c) {
	 * {
	 *     r.template reflectBase<MyBase>(c);
	 * }
	 * \endcode
	 *
	 */
	template <typename Base>
	void reflectBase(Base& base) {}


	/**
	 * Returns true, if the concrete derived Reflector supports human
	 * readable IDs.
	 * The return value of this method is known at compile time and hence
	 * the compiler is able to optimize the code depending on that value.
	 */
	static bool usesHumanReadableIDs() {
		return Derived::useHumanReadableIDs::value;
	}

	/**
	 * If a reflector requires reflection barriers, preReflect() and postReflect()
	 * should be called before/after calling an external method within reflect(), to declare
	 * these barriers (such a 'barrier' is just a declaration to the reflector, in fact).
	 * The base implementation is empty and should not actually be called (if a reflector
	 * needs these methods, it must provide own definitions).
	 *
	 * @param context   Can be used to provide context, e.g. name of the reflected class
	 *                  or function that is called. How this is used depends on the reflector
	 *                  implementation, e.g. MetaSerializer uses it to generate comments on
	 *                  data elements implicitly defined by preReflect.
	 *
	 * The complete process of calling these methods is wrapped by \ref MIRA_REFLECT_CALL.
	 * That macro should be used to simplify handling.
	 *
	 * See \ref requireReflectBarriers and ReflectState.
	 */
	ReflectState preReflect(const char* context = "")
	{
		MIRA_THROW(XNotImplemented,
		           typeName<Derived>() << "::requireReflectBarriers == true, but " <<
		           typeName<Derived>() << "::preReflect() not implemented!");
	
		return ReflectState();
	}

	/// See \ref preReflect for documentation.
	void postReflect(const ReflectState&)
	{
		MIRA_THROW(XNotImplemented,
		           typeName<Derived>() << "::requireReflectBarriers == true, but " <<
		           typeName<Derived>() << "::postReflect() not implemented!");
	}

protected:

	static int checkForcedVersion(const std::string& variable)
	{
		try {
			std::stringstream ss;
			ss.str(resolveEnvironmentVariable(variable));
			int i;
			ss >> i;
			return i;
		}
		catch(XInvalidConfig&) {}

		return -1;
	}
};


/**
 * @ingroup SerializationModule
 *
 * Macro that should be used to specify the (reflected) version of the class.
 *
 * \code
 * template<typename Reflector>
 * void reflect(Reflector& r)
 * {
 *     MIRA_REFLECT_VERSION(r, version);
 * }
 * \endcode
 *
 * The macro calls the version() method of the reflector.
 *
 * @see @ref SerializationPage
 */
#define MIRA_REFLECT_VERSION(reflector, versionNumber) reflector.version(versionNumber, this)

/**
 * @ingroup SerializationModule
 *
 * Macro that should be used to specify the required (reflected) version of the class.
 *
 * \code
 * template<typename Reflector>
 * void reflect(Reflector& r)
 * {
 *     MIRA_REFLECT_REQUIRE_VERSION(r, version);
 * }
 * \endcode
 *
 * The macro calls the requireVersion() method of the reflector.
 *
 * @see @ref SerializationPage
 */
#define MIRA_REFLECT_REQUIRE_VERSION(reflector, versionNumber) reflector.requireVersion(versionNumber, this)

/**
 * @ingroup SerializationModule
 *
 * Macro that can be used to reflect the base class easily.
 *
 * \code
 * template<typename Reflector>
 * void reflect(Reflector& r)
 * {
 *     MIRA_REFLECT_BASE(r, MyBaseClass); // reflect our base class first
 * }
 * \endcode
 *
 * The macro calls the reflectBase() method of the reflector.
 *
 * @see @ref SerializationPage
 */
#define MIRA_REFLECT_BASE(reflector, BaseClass) reflector.template reflectBase<BaseClass>(*this)

/**
 * @ingroup SerializationModule
 *
 * Macro that can be used to reflect the base class easily in a non-intrusive reflect() method.
 *
 * \code
 * template<typename Reflector>
 * void reflect(Reflector& r, Class& object)
 * {
 *     MIRA_REFLECT_BASE(r, BaseClass, object); // reflect Class' base class first
 * }
 * \endcode
 *
 * The macro calls the reflectBase() method of the reflector.
 *
 * @see @ref SerializationPage
 */
#define MIRA_REFLECT_BASE_NONINTRUSIVE(reflector, BaseClass, object) reflector.template reflectBase<BaseClass>(object)

/**
 * @ingroup SerializationModule
 *
 * Macro that can be used to selectively call BaseClass::reflect() directly or through reflectBase(),
 * depending on version.
 *
 * Assume someone forgot to properly use reflectBase/MIRA_REFLECT_BASE when implementing the
 * reflect method for a derived class:
 *
 * \code
 * template<typename Reflector>
 * void reflect(Reflector& r, Class& object)
 * {
 *     BaseClass::reflect(r);
 * }
 * \endcode
 *
 * Simply replacing BaseClass::reflect(r) with MIRA_REFLECT_BASE will break binary compatibility
 * (it will add an extra version field for the base class).
 * A non-breaking fix (allowing to read data created by the older implementation) is to define a
 * new version and call MIRA_REFLECT_BASE for the new version, or BaseClass::reflect for the
 * old version:
 *
 * \code
 * template<typename Reflector>
 * void reflect(Reflector& r, Class& object)
 * {
 *     serialization::VersionType v = MIRA_REFLECT_VERSION(r, 1);
 *     if (v < 1) // data from older implementations will be recognized as version=0
 *         BaseClass::reflect(r);
 *     else
 *         reflector.template reflectBase<BaseClass>(*this);
 * }
 * \endcode
 *
 * @see @ref SerializationPage
 */
#define MIRA_REFLECT_BASE_IF_VERSION_ATLEAST(reflector, BaseClass, versionvar, minversion) \
	if (versionvar < minversion)                                                           \
		BaseClass::reflect(reflector);                                                     \
	else                                                                                   \
		reflector.template reflectBase<BaseClass>(*this);                                  \

/**
 * @ingroup SerializationModule
 *
 * Macro that can be used in non-intrusive reflection to selectively call
 * reflect(reflector, (BaseClass&)object) directly or through reflectBase(), depending on version.
 *
 * @see MIRA_REFLECT_BASE_IF_VERSION_ATLEAST
 *
 */
#define MIRA_REFLECT_BASE_NONINTRUSIVE_IF_VERSION_ATLEAST(reflector, BaseClass, object, versionvar, minversion) \
	if (versionvar < minversion)                                                           \
		reflect(reflector, (BaseClass&)object);                                            \
	else                                                                                   \
		reflector.template reflectBase<BaseClass>(object);                                 \

/**
 * @ingroup SerializationModule
 *
 * Macro that should be used to reflect a member if a manually specified
 * ID is used. This macro checks if the reflector supports human readable IDs.
 * If not, the usually time consuming construction of the ID is optimized out.
 *
 * @note If you do not need to specify an ID explicitly, then call the
 *       appropriate member() method of the Reflector directly.
 *
 * @see @ref SerializationPage
 */
#define MIRA_MEMBER_WITH_ID(reflector, name, id, var, ...) \
	if(reflector.usesHumanReadableIDs())                   \
		reflector.member(name, id, var, __VA_ARGS__);      \
	else                                                   \
		reflector.member(name, var, __VA_ARGS__);

#define MIRA_PROPERTY_WITH_ID(reflector, name, id, var, ...) \
	if(reflector.usesHumanReadableIDs())                     \
		reflector.property(name, id, var, __VA_ARGS__);      \
	else                                                     \
		reflector.property(name, var, __VA_ARGS__);

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
