/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file JSONRPCResponse.h
 *    Special response type for JSON RPC calls.
 *
 * @author Tim Langner
 * @date   2012/04/02
 */

#ifndef _MIRA_JSONRPCRESPONSE_H_
#define _MIRA_JSONRPCRESPONSE_H_

#include <json/JSON.h>
#include <rpc/RPCError.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Wraps a JSON RPC call response.
 * Is used for specialization of RPCFuture for JSON RPC calls.
 */
class JSONRPCResponse
{
public:
	JSONRPCResponse() {}
	JSONRPCResponse(const json::Value& iResponse) :
		response(iResponse) {}

	/**
	 * Return the result of the RPC call
	 * @throw XRPC if no result was set or if the JSON RPC call returned an error
	 * @return The result as json::Value
	 */
	json::Value getResult() const
	{
		auto it = response.get_obj().find("error");
		if(it!=response.get_obj().end()) {
			auto& obj = it->second;
			auto messageit = obj.get_obj().find("message");
			if(messageit==obj.get_obj().end()) {
				MIRA_THROW(XRPC, "RPC client response has no correct error format: message missing");
			}
			auto stackit = obj.get_obj().find("callstack");
			// if there is no callstack, throw with local callstack info
			if(stackit==obj.get_obj().end())
				MIRA_THROW(XRPC, messageit->second.get_str());
				
			auto threadit = obj.get_obj().find("thread");
			if(threadit==obj.get_obj().end())
				MIRA_THROW(XRPC, "RPC client response has no correct error format: callstack without thread id");
			
			CallStack stack;
			ThreadID thread;
			
			try {
				JSONDeserializer js(stackit->second);
				js.deserialize(stack);
			}
			catch(XIO&) {
				MIRA_THROW(XRPC, "RPC client response has no correct error format: callstack unreadable");
			}

			try {
				thread = threadit->second.get_uint64();
			}
			catch(std::runtime_error&) {
				MIRA_THROW(XRPC, "RPC client response has no correct error format: thread id unreadable");
			}
			
			XRPC ex(messageit->second.get_str());
			ex.addExternalStackInfo<XRPC>(stack, thread);

			// try to deserialize and add the original exception
			auto excit = obj.get_obj().find("exception");
			if(excit!=obj.get_obj().end()) {
				try {
					SerializableException* origEx;
					JSONDeserializer js(excit->second);
					js.deserialize(origEx);
					ex.setOrigException(origEx);
				}
				catch(...) {} // ignore errors, it could e.g. be an unknown exception type			
			}

			throw ex;
		}

		it = response.get_obj().find("result");
		if(it==response.get_obj().end()) {
			MIRA_THROW(XRPC, "RPC client response has no correct result format");
		}
		return it->second;
	}

	json::Value response;	///< The complete JSON RPC 2.0 response
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
