/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Img.h
 *   Header file containing Img classes that wrap cv::Mat
 *
 * @author Michael Volkhardt
 * @date   2010/10/18
 */

#ifndef _MIRA_IMG_H_
#define _MIRA_IMG_H_

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>

#include <platform/Types.h>
#include <geometry/Polygon.h>
#include <geometry/Size.h>
#include <serialization/BinarySerializer.h>
#include <serialization/NoGenericReflect.h>
#include <serialization/IsNotMetaSerializable.h>

#include <image/ImgPixel.h>
#include <image/ImgIterator.h>

/// IplImage was removed in OpenCV-4.x.
#if CV_MAJOR_VERSION >= 4
# define MIRA_USE_IPL_IMAGE 0
#endif
/// By default, we don't use the old OpenCV IplImage. If somebody really needs
/// this (only OpenCV < 4.x) MIRA must be compiled with -DMIRA_USE_IPL_IMAGE=1
#ifndef MIRA_USE_IPL_IMAGE
# define MIRA_USE_IPL_IMAGE 0
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief ImgBase class
 *
 * Base class implementing functionality for untyped Img<> template class and
 * typed ImgTypedBase template classes. You should not directly create
 * instances of this class. Therefore, the constructor is protected.
 */
template<typename TImg>
class ImgBase {
public:
	/**
	 * @brief Image storage format description (used e.g. for reflection)
	 */
	struct ImgFormat
	{
		ImgFormat(int depth, const std::string& type, int channels)
			: depth(depth), type(type), channels(channels)
		{}

		void reflect(JSONSerializer& r) {
			r.member("Depth", depth, "The data type's depth (bits/channel)");
			r.member("Channels", channels, "The number of channels");
			r.member("Type", type, "The data type (signed/unsigned/float)");
		}

		int depth;
		std::string type;
		int channels;
	};

protected:
	/**
	 * @brief base constructor
	 * internal data is empty.
	 */
	ImgBase() {
	}

	/**
	 * @brief constructor to initialize internal data from cv::Mat
	 * @note This method does not create any copy of the given Mat. Hence,
	 *       if the Mat is changed then the changes will take affect in
	 *       all images that share that data. To create a deep copy use clone()
	 *       method.
	 */
	ImgBase(const cv::Mat& data) :
		mData(data) {
	}

	/**
	 * @brief constructor
	 * initialize internal data with width, height, and type
	 * note: the image data is allocated but not initialized
	 */
	ImgBase(int width, int height, int type) :
		mData(height, width, type) {
	}

#if MIRA_USE_IPL_IMAGE > 0
	/**
	 * @brief constructor to initialize from old-style IplImage
	 * @param img Image data
	 * @param copyData specifies if the data from the IplImage is copied. If
	 * false just a new header is created around the IplImage data (default)
	 */
	ImgBase(const IplImage* img, bool copyData = false) {
		mData = cv::cvarrToMat(img, copyData);
	}
#endif

	/**
	 * Copy constructor, that creates a shallow copy.
	 * @note This method does not create any copy of the image data, hence
	 *       if the image data is changed then the changes will take affect in
	 *       all images that share that data. To create a deep copy use the
	 *       clone() method.
	 */
	ImgBase(const ImgBase& other) :
		mData(other.mData) {
	}

public:
	///cast operator to const cv::Mat
	operator const cv::Mat&() const {
		return mData;
	}

	///cast operator to cv::Mat
	operator cv::Mat&() {
		return mData;
	}

#if ((CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION >= 3) || CV_MAJOR_VERSION >= 3)
	///cast operator to const cv::_OutputArray
	operator const cv::_OutputArray() const {
		return mData;
	}

	///cast operator to cv::_OutputArray
	operator cv::_OutputArray() {
		return mData;
	}

	///cast operator to cv::_InputArray()
	operator cv::_InputArray() {
		return mData;
	}

	///cast operator to const cv::_InputArray()
	operator const cv::_InputArray() const {
		return mData;
	}
#endif

#if MIRA_USE_IPL_IMAGE > 0
	///cast operator to IplImage
	operator IplImage() {
		return (IplImage) mData;
	}

	///cast operator to const IplImage
	operator const IplImage() const {
		return (IplImage) mData;
	}
#endif

public:

	/**
	 * Assignment operator, that usually makes a shallow copy of "other".
	 */
	ImgBase& operator=(const ImgBase& other) {
		mData = other;
		return *this;
	}

	/**
	 * Copies the content of other into the specified region of interest (ROI)
	 * of THIS image. Note, the number of rows and columns of "other" and
	 * "this" ROI must be equal.
	 */
	void assignROI(const cv::Rect& roi, const ImgBase& other)
	{
		cv::Mat roiMat = this->mData(roi);

		if(roiMat.cols!=other.width()||roiMat.rows!=other.height())
			MIRA_THROW(XLogical, "Target roi and source image width and height must match");

		// make sure, that type, channels and depth match !
		assert(CV_MAT_TYPE(other.mData.flags)  == CV_MAT_TYPE(roiMat.flags)&&
		       CV_MAT_CN(other.mData.flags)    == CV_MAT_CN(roiMat.flags)  &&
		       CV_MAT_DEPTH(other.mData.flags) == CV_MAT_DEPTH(roiMat.flags));

		// copy roi data scanline by scanline
		std::size_t bpl = other.bytesPerLine();
		for(int y=0; y<roi.height; ++y)
			memcpy(roiMat.data+y*(size_t)roiMat.step, other.data(y),bpl);
	}

	/**
	 * Copies the content of other into the specified region of interest (ROI)
	 * of THIS image, where mask has non-zero values.
	 * Note, the number of rows and columns of "other", "mask" and "this" ROI must be equal.
	 */
	void assignMask(const cv::Rect& roi, const ImgBase& other, const cv::Mat& mask)
	{
		cv::Mat roiMat = this->mData(roi);

		if(roiMat.cols!=other.width()||roiMat.rows!=other.height())
			MIRA_THROW(XLogical, "Target roi and source image width and height must match");

		if(mask.cols!=other.width()||mask.rows!=other.height())
			MIRA_THROW(XLogical, "Mask and image width and height must match");

		// make sure, that type, channels and depth match !
		assert(CV_MAT_TYPE(other.mData.flags)  == CV_MAT_TYPE(roiMat.flags)&&
		       CV_MAT_CN(other.mData.flags)    == CV_MAT_CN(roiMat.flags)  &&
		       CV_MAT_DEPTH(other.mData.flags) == CV_MAT_DEPTH(roiMat.flags));

		other.mData.copyTo(roiMat, mask);
	}

	/**
	 * Copies the content of other into the specified region of interest (ROI)
	 * of THIS image, in area covered by polygon.
	 * The result does not depend on cw/ccw order of polygon points.
	 * Note, the number of rows and columns of "other" and "this" ROI must be equal.
	 * Polygon points are relative to roi, any polygon points outside roi are ignored!
	 * (i.e.they must be >= 0, < roi width/height)!
	 */
	void assignPolygon(const cv::Rect& roi, const ImgBase& other, const Polygon2i& poly)
	{
		if(poly.empty())
			MIRA_THROW(XLogical, "Need at least 1 polygon point");

		std::vector<cv::Point> v;
		foreach (const Point2i& p, poly)
			v.emplace_back(p.x(), p.y());

		cv::Mat mask(roi.height, roi.width, CV_8U);
		mask = cv::Scalar(0);
		const cv::Point* vv[1] = { v.data() };
		int n[1] = { int(v.size()) };
		cv::fillPoly(mask, vv, n, 1, cv::Scalar(1));

		assignMask(roi, other, mask);
	}

public:


	/**
	 * @brief Returns true if two images are equal.
	 * this methods checks if two images are equal by comparing their
	 * dimensions, flags and pixel data (byte wise)
	 */
	bool operator==(const cv::Mat& other) const {

		if (other.cols != mData.cols || other.rows != mData.rows)
			return false; // not even dimensions are equal

		if (CV_MAT_TYPE(other.flags)  != CV_MAT_TYPE(mData.flags)  ||
			CV_MAT_CN(other.flags)    != CV_MAT_CN(mData.flags)    ||
			CV_MAT_DEPTH(other.flags) != CV_MAT_DEPTH(mData.flags)) // also flags must match
			return false;

		// compare scanline by scanline, byte by byte
		for (int y = 0; y < other.rows; ++y) {
			const uint8* a = other.ptr(y);
			const uint8* b = mData.ptr(y);
			for (int x = 0; x < other.cols; ++x) {
				if (a[x] != b[x])
					return false; // we found a difference
			}
		}

		// if we reach here, both were equal
		return true;
	}

	/**
	 * Returns true if two images are different. See operator==
	 */
	bool operator!=(const cv::Mat& other) const {
		return !ImgBase::operator==(other);
	}

public:
	///Returns const access to internal data
	const cv::Mat& getMat() const {
		return mData;
	}

	/// returns true if internal Mat is empty
	bool empty() const {
		return mData.empty();
	}

	int width() const {
		return mData.cols;
	}

	int height() const {
		return mData.rows;
	}

	/// Number of bytes from one row to the next
	std::size_t step() const {
		return (size_t)mData.step;
	}

	/// Returns the number of bytes per image line
	std::size_t bytesPerLine() const {
		return mData.elemSize() * mData.cols;
	}

	/// Returns size of image data
	Size2i size() const {
		return Size2i(mData.size());
	}

	/** Returns the bit-depth of the image
	 * E.g. CV_8U, CV_8S, CV_16U, CV_16S, CV_32S, CV_32F, CV_64F
	 * @return bit-depth of image
	 */
	int depth() const {
		return mData.depth();
	}

	/** Returns the number of channels of this image.
	 * Note: typed images also have a static enum value 'Channels'
	 * @return Number of image channels
	 */
	int channels() const {
		return mData.channels();
	}

	/// returns true if internal Mat is empty
	bool isEmpty() const {
		return mData.empty();
	}

	/// Returns pointer to underlying image data
	uint8* data() { return mData.data; }

	/// Returns const pointer to underlying image data
	const uint8* data() const { return mData.data; }


	/// Returns pointer to underlying image data starting at scanline y
	uint8* data(int y) { return mData.data + y*step(); }

	/// Returns const pointer to underlying image data starting at scanline y
	const uint8* data(int y) const { return mData.data + y*step(); }

	/// Return the storage format description
	ImgFormat format() {
		static const char types[8] = {'U', 'S', 'U', 'S', 'S', 'F', 'F', '?'};
		return ImgFormat(8*mData.elemSize1(),
		                 std::string(types+CV_MAT_DEPTH(mData.flags), 1),
		                 mData.channels());
	}

public:

	/**
	 * Resizes the image to the specified dimensions (without changing its
	 * type, i.e. channels and bit depth remains constant).
	 * The method does nothing, if the specified dimensions equal the current
	 * size of the image.
	 */
	void resize(const Size2i& s) {
		if(s==size())
			return; // no need to resize
		// create new matrix with requested size but same type
		mData = cv::Mat(s.height(), s.width(), mData.type());
	}

	/// Same as above method.
	void resize(int width, int height) {
		resize(Size2i(width,height));
	}

public:

	/**
	 * Sets each pixel and each channel to zero.
	 */
	void clear()
	{
		std::size_t bpl = bytesPerLine();
		for(int y=0; y<height(); ++y)
			memset(data(y), 0, bpl);
	}


public:
	/// Deep copy of image data
	TImg clone() const {
		return TImg(mData.clone());
	}

public:
	/// Returns the total size of the matrix (rows*cols)
	std::size_t total() const {
		return mData.rows * mData.cols;
	}

public:

	MIRA_NO_GENERIC_REFLECT_MEMBER(TImg)

	/** @brief reflect method for binaryStream serialization
	 * This method implements the reflection of width, height, type,
	 * and the actual image data serialization.
	 */
	template<typename Derived>
	void reflect(BinarySerializer<Derived>& r) {
		r.version(2, this);

		// try to serialize with codec, if any
		if(r.codec(mData))
			return;

		r.member("Width", mData.cols, "The width of the image");
		r.member("Height", mData.rows, "The height of the image");
		int type = mData.type();
		r.member("Type", type, "The data type including number of channels");

		std::size_t bpl = bytesPerLine();

		// serialize the image data line by line
		const uchar* data = mData.data;
		for (int i = 0; i < mData.rows; ++i, data += mData.step)
			r.write(data, bpl);

	}

	/** @brief reflect method for binaryStream deserialization
	 * This method implements the reflection for width, height, type,
	 * and the image data deserialization.
	 */
	template<typename Derived>
	void reflect(BinaryDeserializer<Derived>& r) {
		int cols, rows, type;

		if(r.version(2, this) > 1) {
			// try to deserialize with codec, if any
			if(r.codec(mData))
				return;
		}

		r.member("Width", cols, "The width of the image");
		r.member("Height", rows, "The height of the image");
		r.member("Type", type, "The data type including number of channels");

		// create the empty image
		mData.create(rows, cols, type);

		std::size_t bpl = bytesPerLine();

		// and deserialize the image line by line
		uchar* data = mData.data;
		for (int i = 0; i < mData.rows; ++i, data += mData.step)
			r.read(data, bpl);
	}

	/** @brief reflect method for json serialization
	 * This method implements the reflection of general data
	 * like width, height and image format, but not the full image data.
	 * It enables a basic overview in a text visualization.
	 */
	void reflect(JSONSerializer& r) {
		Size2i size(mData.cols, mData.rows);
		r.member("Size", size, "The image size", REFLECT_CTRLFLAG_TEMP_TRACKING);

		ImgFormat f = format();
		r.member("Format", f, "The storage format", REFLECT_CTRLFLAG_TEMP_TRACKING);
	}

protected:
	cv::Mat mData;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Base class for typed images.
 * Class providing the base functionality of typed images. You should not
 * create instances of this class by yourself, so the constructor is protected.
 * Please, use the typed Img<type,channels> class instead.
 */
template<typename TImg, typename TPixel>
class ImgTypedBase: public ImgBase<TImg> {
	typedef ImgBase<TImg> Base;

protected:
	friend class ImgBase<TImg> ;

	ImgTypedBase(const cv::Mat& data) :
		Base(data) {
	}

	/**
	 * @brief standard constructor
	 */
	ImgTypedBase() {
		this->mData.flags = (this->mData.flags & ~CV_MAT_TYPE_MASK) | cv::DataType<TPixel>::type;
	}

	/**
	 * @brief constructs a new image using width and height and type from
	 * Template type.
	 * note: the image data is allocated but not initialized
	 */
	ImgTypedBase(int width, int height) :
		Base(width, height, cv::DataType<TPixel>::type) {
	}

public:
	/**
	 * @brief Converts a typed image into another typed image
	 * If the channels and/or type of the new image and the source differ, 
	 * the image is converted and copied using the cv::mixChannels and
	 * cvConvertScale functions.
	 * If the channels and type equals the image is only copied if
	 * alwaysCopy is true.
	 * @param other The image that should be converted
	 * @param alwaysCopy If true the data is always copied even if channels and type match
	 * @return the converted image
	 */
	static TImg convertFrom(const cv::Mat& other, bool alwaysCopy = false) {
		int targetType = cv::DataType<TPixel>::type;

		// if channels and type equals and we don't want to copy - just return the image
		if (!alwaysCopy && (CV_MAKETYPE(other.type(), other.channels()) == targetType))
			return TImg(other);

		cv::Mat otherData = other;
		cv::Mat data(otherData.rows, otherData.cols, targetType);

		if (other.channels() != data.channels()) {
			// we need to convert the number of channels

			// create a temporary image that has the type of the other
			// image but the number of channels of our target type
			cv::Mat tmp(otherData.rows, otherData.cols,
			            CV_MAKETYPE(otherData.type(), data.channels()));

			// use MixChannels and build fromTo-mapping first
			std::vector<int> fromTo(data.channels() * 2);

			int srcChannel = 0;
			for (int k = 0; k < data.channels(); ++k, ++srcChannel) {
				fromTo[2 * k] = (srcChannel < otherData.channels() ? srcChannel : 0);
				fromTo[2 * k + 1] = k;
			}

			cv::mixChannels(&otherData, 1, &tmp, 1, fromTo.data(),
			                data.channels());

			otherData = tmp;
		}

#if ((CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION >= 3) || CV_MAJOR_VERSION >= 3)
		otherData.convertTo(data, targetType);
#else
		// Since this:
		//otherData.convertTo(data, targetType);
		// is buggy in the OpenCV 2.0-2.1, we have to use the
		// C-interface:
		CvMat src = otherData;
		CvMat dest = data;
		cvConvertScale(&src, &dest);
#endif
		return TImg(data);
	}

public:
	/// assigns a specific type to the image data (used to set pixel values)
	TImg& operator=(const TPixel& p) {
		castToMat() = p;
		return *This();
	}

	/**
	 * @brief access to image row
	 * @param y row to access. starting with 0 at top of image
	 */
	TPixel* operator[](int y) {
		return this->castToMat()[y];
	}

	/**
	 * @brief const access to image row
	 * @param y row to access. starting with 0 at top of image
	 */
	const TPixel* operator[](int y) const {
		return this->castToMat()[y];
	}

	/**
	 * @brief returns a ROI out of the image
	 * Returns a new image with set ROI on the image data.
	 * @note no image data is copied, just a new header is created.
	 */
	TImg operator()(const cv::Rect& roi) {
		return TImg(this->mData(roi));
	}

	/**
	 * @brief returns a ROI out of the image
	 * Returns a new image with set ROI on the image data.
	 * @note no image data is copied, just a new header is created.
	 */
	TImg operator()(const cv::Rect& roi) const {
		return TImg(this->mData(roi));
	}


	/**
	 * @brief access pixel(x,y)
	 */
	TPixel& operator()(int x, int y) {
		return this->castToMat()(y, x);
	}

	/**
	 * @brief const access pixel(x,y)
	 */
	const TPixel& operator()(int x, int y) const {
		return this->castToMat()(y, x);
	}

	/**
	 * @brief access to pixel given by point p
	 */
	TPixel& operator()(const cv::Point& p) {
		return this->castToMat()(p);
	}

	/**
	 * @brief const access to pixel given by point p
	 */
	const TPixel& operator()(const cv::Point& p) const {
		return this->castToMat()(p);
	}

public:

	typedef ImgIterator<TPixel> iterator;
	typedef ImgConstIterator<TPixel> const_iterator;

	/**
	 * Returns an iterator to the beginning of this image (image region)
	 * @see ImgIterator
	 */
	iterator begin() {
		return iterator(&castToMat());
	}

	/**
	 * Returns an iterator to the end of this image (image region)
	 * Note that the end iterator is NOT part of the image (image region)
	 * @see ImgIterator
	 */
	iterator end() {
		iterator it(&castToMat());
		it += this->total();
		return it;
	}

	/**
	 * Returns an const iterator to the beginning of this image (image region)
	 * @see ImgConstIterator
	 */
	const_iterator begin() const {
		return const_iterator(&castToMat());
	}

	/**
	 * Returns an const iterator to the end of this image (image region)
	 * Note that the end iterator is NOT part of the image (image region)
	 * @see ImgConstIterator
	 */
	const_iterator end() const {
		const_iterator it(&castToMat());
		it += this->total();
		return it;
	}

public:

	MIRA_NO_GENERIC_REFLECT_MEMBER(TImg)

	/** @brief reflect method for binaryStream serialization
	 * This method uses the reflect method of the ImageBase class
	 */
	template<typename Derived>
	void reflect(BinarySerializer<Derived>& r) {
		MIRA_REFLECT_BASE(r, Base);
	}

	/** @brief reflect method for binaryStream deserialization
	 * This method uses the reflect method of the ImageBase class
	 * and assures that the type of the deserialized image is correct.
	 * Otherwise an exception is thrown
	 */
	template<typename Derived>
	void reflect(BinaryDeserializer<Derived>& r) {
		MIRA_REFLECT_BASE(r, Base);
		// make sure that the type of the deserialized image matches
		// to our type, otherwise throw exception
		if (this->mData.type() != cv::DataType<TPixel>::type)
			MIRA_THROW(XIO, "The deserialized image type "
				"(depth: " << this->mData.depth() << ", channels: "
					<< this->mData.channels() << ") "
				"does not match the type of the Img class "
				"(depth: " << CV_MAT_DEPTH(cv::DataType<TPixel>::type)
					<< ", channels: " << CV_MAT_CN(cv::DataType<TPixel>::type)
					<< ")");
	}

	/** @brief reflect method for json serialization
	 * This method uses the reflect method of the ImageBase class
	 */
	void reflect(JSONSerializer& r) {
		MIRA_REFLECT_BASE(r, Base);
	}

protected:

	TImg* This() {
		return static_cast<TImg*> (this);
	}

	const TImg* This() const {
		return static_cast<const TImg*> (this);
	}

	/// operator for convenience access to image data as Mat
	cv::Mat_<TPixel>& castToMat() {
		return static_cast<cv::Mat_<TPixel>&> (this->mData);
	}

	/// operator for convenience const access to image data as Mat
	const cv::Mat_<TPixel>& castToMat() const {
		return static_cast<const cv::Mat_<TPixel>&> (this->mData);
	}

};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief class for typed images.
 * Class for images, where channels and type are known at compile time.
 * @ingroup ImageModule
 */
template<typename T = void, int TChannels = 1>
class Img: public ImgTypedBase<Img<T, TChannels> , ImgPixel<T, TChannels> > {
public:
	typedef ImgPixel<T, TChannels> Pixel;

protected:

	typedef Img<T, TChannels> Self;
	typedef ImgTypedBase<Self, Pixel> Base;

protected:

	friend class ImgBase<Self> ;
	friend class ImgTypedBase<Self, Pixel> ;

	// this must NOT be public, otherwise one could spoil our type safety
	Img(const cv::Mat& data) :
		Base(data) {
	}

public:

	/// The static number of channels of this typed image class.
	enum { Channels = TChannels };

public:

	Img() {
	}

	/**
	 * @brief constructs a new image of given size
	 * note: the image data is allocated but not initialized
	 * @param width width of image
	 * @param height height of image
	 */
	Img(int width, int height) :
		Base(width, height) {
	}
	/**
	 * @brief constructs a new image of given size
	 * note: the image data is allocated but not initialized
	 * @param size size of the image
	 */
	Img(const Size2i& size) :
		Base(size.width(), size.height()) {
	}

public:
	/**
	 * @brief assigns image with given Pixel
	 * assigns each pixel of the image to the given Pixel
	 *
	 * @param p Pixel which is assigned to each pixel of the image
	 * @return reference on image
	 */
	Self& operator=(const Pixel& p) {
		return Base::operator=(p);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief specialized typed image with 1 channel
 * This template class provides convenient way to create an image with known
 * type and only 1 channel by using Img<Type>
 */
template<typename T>
class Img<T, 1> : public ImgTypedBase<Img<T, 1> , T> {
public:
	typedef T Pixel;

protected:

	typedef Img<T, 1> Self;
	typedef ImgTypedBase<Self, Pixel> Base;

protected:

	friend class ImgBase<Self> ;
	friend class ImgTypedBase<Self, Pixel> ;

	// this must not be public, otherwise one could spoil our type safety
	Img(const cv::Mat& data) :
		Base(data) {
	}

public:

	/// The static number of channels of this typed image class.
	enum { Channels = 1 };

public:

	Img() {
	}
	/**
	 * @brief constructs a new image of given size
	 * note: the image data is allocated but not initialized
	 * @param width width of image
	 * @param height height of image
	 */
	Img(int width, int height) :
		Base(width, height) {
	}
	/**
	 * @brief constructs a new image of given size
	 * note: the image data is allocated but not initialized
	 * @param size size of the image
	 */
	Img(const Size2i& size) :
		Base(size.width(), size.height()) {
	}

public:
	/**
	 * @brief assigns image with given Pixel
	 * assigns each pixel of the image to the given Pixel
	 *
	 * @param p Pixel which is assigned to each pixel of the image
	 * @return reference on image
	 */
	Self& operator=(const Pixel& p) {
		return Base::operator=(p);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief untyped image class
 * This class provides a fully dynamic image, where type and channels are not
 * known at compile time.
 */
template<>
class Img<void, 1> : public ImgBase<Img<void, 1> > {
	typedef Img<void, 1> Self;
	typedef ImgBase<Self> Base;

protected:
	friend class ImgBase<Self> ;

public:

	Img() {
	}
	/**
	 * @brief constructs a new image of given size and type
	 * note: the image data is allocated but not initialized
	 * @param width width of image
	 * @param height height of image
	 * @param type CV_TYPE of the image
	 * @param channels number of channels
	 */
	Img(int width, int height, int type, int channels) :
		Base(width, height, CV_MAKETYPE(type, channels)) {
	}
	/**
	 * @brief constructs a new image of given size and type
	 * note: the image data is allocated but not initialized
	 * @param size size of image
	 * @param type CV_TYPE of the image
	 * @param channels number of channels
	 */
	Img(const Size2i& size, int type, int channels) :
		Base(size.width(), size.height(), CV_MAKETYPE(type, channels)) {
	}

	template<typename TImg>
	Img(const ImgBase<TImg>& other) :
		Base(other) {
	}

#if MIRA_USE_IPL_IMAGE > 0
	/** @brief creates Img from old-style IplImage
	 * For backward compatibility. Converts old-style IplImage to image,
	 * the data is not copied by default
	 * @param img The image data
	 * @param copyData specifies if the image data is copied, if false
	 * just a new header is created for the IplImage data.
	 */
	Img(const IplImage* img, bool copyData = false) :
		Base(img, copyData) {
	}
#endif

	/** @brief constructor from cv::Mat
	 * constructor is public, since we do not have a type safe image
	 */
	Img(const cv::Mat& data) :
		Base(data) {
	}

	///shallow copy
	template<typename TImg>
	Self& operator=(const ImgBase<TImg>& other) {
		mData = other.getMat();
		return *this;
	}


	/**
	 * @brief returns a ROI out of the image
	 * Returns a new image with set ROI on the image data.
	 * @note no image data is copied, just a new header is created.
	 */
	Img operator()(const cv::Rect& roi) {
		return Img(this->mData(roi));
	}

	/**
	 * @brief returns a ROI out of the image
	 * Returns a new image with set ROI on the image data.
	 * @note no image data is copied, just a new header is created.
	 */
	Img operator()(const cv::Rect& roi) const {
		return Img(this->mData(roi));
	}

	/**
	 * Returns true, if the untyped image internally has the specified
	 * type, i.e. TPixel type and TChannels match to the internal image type.
	 */
	template <typename TPixel, int TChannels>
	bool hasType() const
	{
		return cv::DataType<TPixel>::depth == this->depth() &&
				TChannels==this->channels();
	}

public:

	typedef UntypedImgIterator iterator;
	typedef UntypedImgConstIterator const_iterator;

	iterator begin() {
		return iterator(&this->mData);
	}

	iterator end() {
		iterator it(&this->mData);
		it.ptr = it.sliceEnd = (this->mData.data + this->mData.step*(this->mData.rows-1)) +
				this->mData.cols * this->mData.elemSize();
		return it;
	}

	const_iterator begin() const {
		return const_iterator(&this->mData);
	}

	const_iterator end() const {
		const_iterator it(&this->mData);
		it.ptr = it.sliceEnd = (this->mData.data + this->mData.step*(this->mData.rows-1)) +
				this->mData.cols * this->mData.elemSize();
		return it;
	}

};



/**
 * Casts an untyped Img<> into a typed image with the specified pixel type
 * and channel count.
 * @throw XBadCast exception, if the bit depth and number of channels  of the
 *        untyped image do not correspond to the specified target type
 *        and target channel count.
 */
template <typename TPixel, int TChannels>
Img<TPixel,TChannels> img_cast(Img<>& other)
{
	// if other has the same type here, return typed shallow copy
	if(other.hasType<TPixel,TChannels>())
		return Img<TPixel,TChannels>::convertFrom(other);

	// otherwise cannot cast
	MIRA_THROW(XBadCast, "Cannot cast this untyped image with of type " <<
	          other.depth() << "," << other.channels() <<
	          " to channel of type " << cv::DataType<TPixel>::depth <<
	          "," << TChannels);
}


/// Mark image classes as not meta-serializable
template <typename TPixel, int TChannels>
class IsNotMetaSerializable<Img<TPixel, TChannels>> : public std::true_type {};


/**
 * Image: unsigned 8-bit integers, 1 channel
 * @ingroup ImageModule
 */
typedef Img<uint8, 1> Img8U1;

/**
 * Image: unsigned 8-bit integers, 2 channels
 * @ingroup ImageModule
 */
typedef Img<uint8, 2> Img8U2;

/**
 * Image: unsigned 8-bit integers, 3 channel
 * @ingroup ImageModule
 */
typedef Img<uint8, 3> Img8U3;

/**
 * Image: unsigned 8-bit integers, 4 channel
 * @ingroup ImageModule
 */
typedef Img<uint8, 4> Img8U4;

/**
 * Image: unsigned 16-bit integers, 1 channel
 * @ingroup ImageModule
 */
typedef Img<uint16, 1> Img16U1;

/**
 * Image: unsigned 16-bit integers, 2 channel
 * @ingroup ImageModule
 */
typedef Img<uint16, 2> Img16U2;

/**
 * Image: unsigned 16-bit integers, 3 channel
 * @ingroup ImageModule
 */
typedef Img<uint16, 3> Img16U3;

/**
 * Image: unsigned 16-bit integers, 4 channel
 * @ingroup ImageModule
 */
typedef Img<uint16, 4> Img16U4;

/**
 * Image: 16-bit integers, 1 channel
 * @ingroup ImageModule
 */
typedef Img<int16, 1> Img16S1;

/**
 * Image: 16-bit integers, 2 channel
 * @ingroup ImageModule
 */
typedef Img<int16, 2> Img16S2;

/**
 * Image: 16-bit integers, 3 channel
 * @ingroup ImageModule
 */
typedef Img<int16, 3> Img16S3;

/**
 * Image: 16-bit integers, 4 channel
 * @ingroup ImageModule
 */
typedef Img<int16, 4> Img16S4;

/**
 * Image: floating-point numbers, 1 channel
 * @ingroup ImageModule
 */
typedef Img<float, 1> Img32F1;

/**
 * Image: floating-point numbers, 2 channel
 * @ingroup ImageModule
 */
typedef Img<float, 2> Img32F2;

/**
 * Image: floating-point numbers, 3 channel
 * @ingroup ImageModule
 */
typedef Img<float, 3> Img32F3;

/**
 * Image: floating-point numbers, 4 channel
 * @ingroup ImageModule
 */
typedef Img<float, 4> Img32F4;

/**
 * Image: double precision floating-point numbers, 1 channel
 * @ingroup ImageModule
 */
typedef Img<double, 1> Img64F1;

/**
 * Image: double precision floating-point numbers, 2 channel
 * @ingroup ImageModule
 */
typedef Img<double, 2> Img64F2;

/**
 * Image: double precision floating-point numbers, 3 channel
 * @ingroup ImageModule
 */
typedef Img<double, 3> Img64F3;

/**
 * Image: double precision floating-point numbers, 4 channel
 * @ingroup ImageModule
 */
typedef Img<double, 4> Img64F4;

///////////////////////////////////////////////////////////////////////////////

}//end namespace

#endif /* _MIRA_IMG_H_ */
