/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IconProvider.C
 *    Implementation of IconProvider.
 *
 * @author Ronny Stricker
 * @date   2012/09/21
 */

#include "gui/IconProvider.h"

#include <assert.h>
#include <iostream>

#include <QPainter>
#include <QBitmap>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/// simple helper function to overlay two QPixmaps
QPixmap overlay( QPixmap const& background, QPixmap const& foreground )
{
	// compose new pixmap using overlay
	assert( background.size() == foreground.size() );
	QImage overlay = QImage( background.size(), QImage::Format_ARGB32_Premultiplied );

	// make the background transparent
	QPainter painter( &overlay );
	painter.setCompositionMode( QPainter::CompositionMode_Source );
	painter.fillRect( overlay.rect(), Qt::transparent );

	// draw background image
	painter.setCompositionMode( QPainter::CompositionMode_SourceOver );
	painter.drawPixmap(0, 0, background );

	// draw foreground image
	painter.drawPixmap(0, 0, foreground );

	return QPixmap::fromImage( overlay );
}

///////////////////////////////////////////////////////////////////////////////

IconProvider::IconProvider()
{
	mInstalled = QPixmap(":/icons/pk-package-installed.png");
	mAvailable = QPixmap(":/icons/pk-package-available.png");

	mIconBase[ Package::ETC ] = QPixmap(":/icons/etc.png");
	mIconBase[ Package::BINARY ] = QPixmap(":/icons/binary.png");
	mIconBase[ Package::SOURCE ] = QPixmap(":/icons/source.png");

	QPixmap empty( mInstalled.width(), mInstalled.height() );
	empty.fill( Qt::white );
	empty.setAlphaChannel( empty.createMaskFromColor( Qt::black ) );

	mIconDepFlags[ Dependency::FACULTATIVE ] = QPixmap(":/icons/optional.png");
	mIconDepFlags[ Dependency::RUNTIME ] = QPixmap(":/icons/runtime.png");
	mIconDepFlags[ Dependency::VERSION_OK ] = empty;
	mIconDepFlags[ Dependency::VERSION_INCOMP ] = QPixmap(":/icons/attention.png");

	mInstall = overlay( mAvailable, QPixmap(":/icons/install.png") );
	mUninstall = overlay( mInstalled, QPixmap(":/icons/uninstall.png") );
}

QIcon const& IconProvider::getIcon( bool installed, PackageGroup::Type const& baseType,
		Dependency::DependencyFlags const& depFlags ) const
{
	// use base type in the higher bits of the index
	uint32 index = (uint32(baseType) << 8) + uint32( depFlags );

	// try to find icon in icon cache
	if ( (mIconCache.find( installed ) != mIconCache.end())
			 && (mIconCache[ installed ].find( index ) != mIconCache[ installed ].end() ) )
		return mIconCache[installed][index];

	// get icon base
	QPixmap currentIcon = installed ? mInstalled : mAvailable;

	// combine with type base
	if ( baseType != PackageGroup::UNSPECIFIED )
		currentIcon = overlay( currentIcon, mIconBase[ baseType ] );

	// combine icon from available dependency overlays
	for (int i=1;i<=Dependency::VERSION_INCOMP;i *= 2) {
		if ( i & depFlags ) {
			currentIcon = overlay( currentIcon, mIconDepFlags[ Dependency::DependencyFlags( i ) ] );
		}
	}

	// store icon in icon cache
	QIcon icon( currentIcon );
	mIconCache[ installed ][ index ] = icon;
	return mIconCache[ installed ][ index ];
}

QPixmap const& IconProvider::getIcon( Database::Action const& action )
{
	if ( action == Database::INSTALL )
		return mInstall;
	else if ( action == Database::UNINSTALL )
		return mUninstall;
	return mInstall;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

