/*
 * Copyright (C) 2019 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GitUtils.h
 *
 * @author Thomas Bauer
 * @date   2019/10/17
 */

#ifndef _MIRA_GITUTILS_H_
#define _MIRA_GITUTILS_H_

extern "C" {
// please read at least https://libgit2.org/docs/guides/101-samples/ before changing libgit2 related code
#include <git2.h>
}

#include "core/Credential.h"
#include <string>
#include <map>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class GitUtils
{

public:
	GitUtils();
	GitUtils(const GitUtils&) = default;

	~GitUtils();

	GitUtils& operator=(const GitUtils&) = default;

	/** @brief Return true if the given path is a git repository
	 *  @param[in] path, path on filesystem
	 *  @return true if path contains a git repository
	 */
	bool isGitRepository(const std::string& path);

	/** @brief returns the url of the remote repository origin.
	 *  @param[in] path, the path where the git repository resides in.
	 *  @param[in] the name of the remote to lookup, defaults to "origin"
	 *  @return url to remote or empty string if there is no such remote or mRepo is nullptr
	 */
	std::string remoteURL(const std::string& path, const std::string& remote = "origin");

	/** @brief checks if path is inside a repository and returns the .git directory if found
	 *  @param[in] a path on filesystem
	 *  @return .git directory path
	 */
	std::string findDotGit(const std::string& path);

	/** @brief check whether an URL hosts a git repository
	 *  @param[in] an URL
	 *  @return true if url hosts a git remote you can connect to
	 */
	bool checkURL(const std::string& URL);

	/** Iterate over URL sub path to find a git remote URL you can clone.
	 *  @brief iterate over URL sub path to find a git remote URL you can clone
	 *  @param[in] an URL
	 *  @return a URL clonable git repository URL or empty string
	 */
	std::string findRemoteURL(const std::string& URL);

	/** @brief clone git repository at an http url into path
	 *  @param[in] httpUrl, the url where the repository can be cloned, needs to be accessible via http
	 */
	bool clone(const std::string& httpUrl, const std::string& path, const Credential& cred);

	/** @brief checkout a certain commit.
	 *  @param[in] path, the path where the git repository resides in.
	 *  @param[in] ref, a commit-reference or commit id to set the head to
	 */
	bool detachHead(const std::string& path, const std::string& ref);

	/** @brief This makes the working copy and index to match the commit that HEAD points to.
	 *  @param[in] path, the path where the git repository resides in.
	 *  @param[in] force = true overrides all changes in working copy, while false will left modified files untouched.
	 */
	bool checkoutHead(const std::string& path, bool force = false);

	/** @brief get a list of files and their status in repository at
	 *  path.
	 *  @param[in] path, the path where the git repository resides in.
	 */
	std::map<std::string, unsigned int> status(const std::string& path);

	/** @brief Helper function to get a readable status string.
	 *  @param[in] sc, a libgit2 status code
	 *  @return a string containing the status readable for humans
	 */
	static std::string statusCode(const unsigned int sc);

	/** Lookup the branches of a git repository given by path.
	 *  @param[in] path, path to repository
	 *  @return a map with <std::string localbranch, std::string remotebranch>, remotebranch can be an empty string
	 */
	std::map<std::string, std::string> getBranchMap(const std::string& path);

	/**  @brief get a vector of branches which have commits, that are
	 *   not pushed to a given remote.
	 *   @param[in] path, path to repository
	 */
	std::vector<std::string> needPush(const std::string& path, const std::string& remote = "origin");

private:

	/** @brief Check err for libgit2 error code and prompt last error message if not 0
	 *  @param[in] libgit2 error code
	 *  @return nothing if err is 0 otherwise throw an exception
	 */
	void okayOrThrow(int err, const std::string& reason);

};

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif /* _MIRA_GITREPOSITORY_H_ */
