/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkGraphView.C
 *    Description.
 *
 * @author Erik Einhorn
 * @date   02/15/2014
 */

#include <views/FrameworkGraphView.h>

#include <boost/algorithm/string.hpp>

#include <QTimerEvent>
#include <QTimeLine>
#include <QGraphicsScene>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QCheckBox>
#include <QScrollBar>
#include <QMouseEvent>
#include <QLabel>
#include <QFileDialog>
#include <QGraphicsSvgItem>
#include <QSvgRenderer>


#undef USE_OPENGL
#ifdef USE_OPENGL // collides with our 3D view
#include <QGLWidget>
#endif

#include <serialization/Serialization.h>

#include <widgets/FlowLayout.h>

#include <thread/ThreadMonitor.h>

#include <common/DotToSvg.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

GraphViewUI::GraphViewUI(FrameworkGraphView* view, QWidget* parent) : GraphView(parent), mView(view)
{
	showText("Exploring graph ...");

	mDiscoverThread = boost::thread(boost::bind(&GraphViewUI::discoverGraphThread, this));
	connect(this, SIGNAL(startDiscoverGraph()), this, SLOT(onStartDiscoverGraph()), Qt::QueuedConnection);
	connect(this, SIGNAL(finishedDiscoverGraph()), this, SLOT(onFinishedDiscoverGraph()), Qt::QueuedConnection);
	connect(this, SIGNAL(graphHasChanged()), this, SLOT(onGraphHasChanged()), Qt::QueuedConnection);
}

GraphViewUI::~GraphViewUI()
{
	mDiscoverThread.interrupt();
	mDiscoverThread.join();

}

static std::string leafName(std::string name)
{
	std::size_t pos = name.rfind('/');
	if(pos!=std::string::npos)
		return name.substr(pos+1);
	return name;
}

template <typename Container>
bool hasMatchingItem(const Container& c, const std::string& s)
{
	foreach(auto& i, c)
		if(s.find(i)!=std::string::npos)
			return true;
	return false;
}

template <typename Container>
bool GraphViewUI::hideItem(const Container& c, const std::string& name)
{
	if(name.find('#')!=std::string::npos && !mView->mCbShowHiddenUnits->isChecked())
		return true;
	else
		return hasMatchingItem(c,name);
}


static std::pair<std::string,std::string> getColorForStatus(Status::StatusMode status)
{
	if (status == Status::BOOTUP)
		return std::pair<std::string,std::string>("#BFBFFF","#0000FF");
	if (status == Status::WARNING)
		return std::pair<std::string,std::string>("#FFFF7F","#FFFF00");
	if (status == Status::ERROR)
		return std::pair<std::string,std::string>("#FFBFBF","#FF0000");
	if (status == Status::RECOVER)
		return std::pair<std::string,std::string>("#FFBFFF","#FF00FF");
	if (status == STATUS_UNKNOWN)
		return std::pair<std::string,std::string>("#BFBFBF","#7F7F7F"); // grey

	return std::pair<std::string,std::string>("#BFFFBF","#00FF00");
}

void GraphViewUI::dotAddChannel(std::ostream& os, Graph::ChannelNodePtr c)
{
	Status::StatusMode status = c->status;
	os << "    \"" << c->id << "\" ["
			"fillcolor=\""<< getColorForStatus(status).first<< "\", "
			"color=\""<< getColorForStatus(status).second<< "\", "
	      "id=\"" << c->id << "\","
	      "label=\"" << leafName(c->id) << "\""
						"]"
				<< std::endl;
}

void GraphViewUI::dotAddAuthority(std::ostream& os, Graph::AuthorityNodePtr a)
{
	Status::StatusMode status = a->status;

	os << "    \"" << a->desc.getGlobalID() <<
			"\" [shape=box,"
			"fillcolor=\""<< getColorForStatus(status).first<< "\", "
			"color=\""<< getColorForStatus(status).second<< "\", "
			"id=\"" << a->desc.getGlobalID() << "\", "
			"label=\"" << a->desc.id << "\""
					"]"
			<< std::endl;
}

bool GraphViewUI::dotAddNamespace(std::ostream& os, Graph::NamespaceNodePtr n,
                                  int level, int& id,
                                  const std::set<Graph::AuthorityNodePtr>& visibleAuthorities,
                                  const std::set<Graph::ChannelNodePtr>& visibleChannels)
{
	bool isEmpty = true;
	bool isRoot = level==0;
	if(!isRoot) {
		const char* colors[16] = {
			"#F0F0FF", "#E0E0FF", "#D0D0FF", "#C0C0FF", "#B0B0FF",
			"#A0A0FF", "#9090FF", "#8080FF", "#7070FF", "#6060FF",
			"#5050FF", "#4040FF", "#3030FF", "#2020FF", "#1010FF", "#0000FF"
		};

		int c = std::min(level-1, 13);
		os <<
		"subgraph cluster" << (id++) << " {\n"
		"    label=\"/" << n->ns.leaf() << "\"\n"
		"    fillcolor=\"" << colors[c] << "\"\n"
		"    color=\"" << colors[c+2] << "\"\n"
		"    style=\"rounded,filled\"\n";
	}

	foreach(auto c, n->children)
	{
		std::stringstream tos;
		if(dotAddNamespace(tos, c, level+1, id, visibleAuthorities, visibleChannels)) {
			// add childs content only, if it is not empty
			os << tos.rdbuf();
			isEmpty = false;
		}
	}

	foreach(auto p, n->nodes)
	{
		Graph::AuthorityNodePtr a = boost::dynamic_pointer_cast<Graph::AuthorityNode>(p);
		if(a) {
			if(visibleAuthorities.count(a)==0)
				continue;

			isEmpty = false;
			dotAddAuthority(os, a);
		}

		if(!mView->mCbDependenciesOnly->isChecked()) {
			Graph::ChannelNodePtr c = boost::dynamic_pointer_cast<Graph::ChannelNode>(p);
			if(c) {
				if(visibleChannels.count(c)==0)
					continue;
				isEmpty = false;
				dotAddChannel(os, c);
			}
		}
	}

	if(!isRoot)
		os << "}"<<std::endl;

	return !isEmpty;
}


void GraphViewUI::discoverGraphThread()
{
	ThreadMonitor::instance().addThisThread("#FrameworkGraphViewDiscoverThread");
	while (!boost::this_thread::interruption_requested())
	{
		if(!mView->mBtUpdate->isChecked()) {
			MIRA_SLEEP(100);
			continue;
		}

		emit startDiscoverGraph();
		try {
			mGraph.discover(boost::bind(&GraphViewUI::graphHasChangedCb, this));
		} catch(...) {} // ignore possible errors

		// get the status of the authorities
		auto authorities = mGraph.getAuthorities();
		foreach(auto a, authorities)
		{
			a->status = STATUS_UNKNOWN;
			try {
				StatusManager::StatusMap statusmap = MIRA_FW.getAuthorityManager().getStatusMap(a->desc.getGlobalID());
				a->status = StatusManager::getOverallStatus(statusmap.begin(), statusmap.end());
			} catch(...) {}
		}

		// get the status of the channels
		auto channels = mGraph.getChannels();
		foreach(auto c, channels)
		{
			if(MIRA_FW.getChannelManager().hasChannel(c->id)) {
				uint64 updates = MIRA_FW.getChannelManager().getNrOfDataChanges(c->id);
				bool hasPublisher = MIRA_FW.getChannelManager().hasPublisher(c->id, true);
				bool hasSubscriber = !c->getSubscribedAuthorities().empty();

				uint64 deltaUpdate = updates - c->lastupdate;
				c->lastupdate = updates;

				if(hasSubscriber && !hasPublisher)
					c->status = Status::ERROR; // have subscriber but no publisher -> red
				else if(hasSubscriber && (deltaUpdate==0 || updates==0))
					c->status = Status::WARNING; // no (recent) update -> yellow
				else
					c->status = Status::OK; // okay -> green

			} else {
				c->status = STATUS_UNKNOWN; // no info
			}
		}
		emit finishedDiscoverGraph();
		MIRA_SLEEP(2000);
	}
	ThreadMonitor::instance().removeThisThread();
}

// called within discoverGraph thread
void GraphViewUI::graphHasChangedCb()
{
	emit graphHasChanged();
}

void GraphViewUI::onStartDiscoverGraph()
{
	mView->mBtUpdateAnimation->setPaused(false);
}

void GraphViewUI::onFinishedDiscoverGraph()
{
	updateGraph();
	mView->mBtUpdateAnimation->jumpToNextFrame();
	mView->mBtUpdateAnimation->setPaused(true);
}

void GraphViewUI::onGraphHasChanged()
{
	Time now = Time::now();
	if(!mLastChangedGraphUpdate.isValid() ||
	   (now-mLastChangedGraphUpdate) > Duration::milliseconds(200)) {
		mLastChangedGraphUpdate = now;
		updateGraph();
	}
}

// graph output is written into os stream
void GraphViewUI::generateDotFromGraph(std::ostream& os)
{
	const std::size_t cEdgeLabelElideLength = 32;

	std::list<std::string> filterUnits;
	std::string tmp = mView->mLeFilterUnits->text().toStdString();
	if(!tmp.empty())
		boost::split(filterUnits, tmp, boost::is_from_range(',',',') );

	std::set<std::string> hidden;
	tmp = mView->mLeHide->text().toStdString();
	if(!tmp.empty())
		boost::split(hidden, tmp, boost::is_from_range(',',',') );


	// first generate all edges, and collect the visible authorities and channels
	std::stringstream es;

	std::set<Graph::AuthorityNodePtr> visibleAuthorities;
	std::set<Graph::ChannelNodePtr> visibleChannels;

	std::set<std::pair<Graph::AuthorityNodePtr,Graph::AuthorityNodePtr>> edges;

	boost::mutex::scoped_lock lock(mGraph.mutex());
	auto authorities = mGraph.getAuthorities();

	foreach(auto a, authorities)
	{
		//StatusManager::StatusMap statusmap = MIRA_FW.getAuthorityManager().getStatusMap(a->desc.getGlobalID());
		//a->status = StatusManager::getOverallStatus(statusmap.begin(), statusmap.end());

		//std::cout << "processing: " << a->desc.getGlobalID() << std::endl;
		// skip authorities that are not in the units list
		if(!filterUnits.empty() && !hasMatchingItem(filterUnits,a->desc.getGlobalID()))
			continue;

		if(hideItem(hidden, a->desc.getGlobalID()))
			continue;

		if(!mView->mCbDependenciesOnly->isChecked()) {

			auto pub = a->getPublishedChannels();
			foreach(auto p, pub)
			{
				if(!hideItem(hidden, p->id)) {
					es << "    \"" << a->desc.getGlobalID() << "\" -> \"" << p->id << "\""<< std::endl;
					visibleAuthorities.insert(a);
					visibleChannels.insert(p);
				}
			}

			auto sub = a->getSubscribedChannels();
			foreach(auto s, sub)
			{
				if(!hideItem(hidden, s->id)) {
					es << "    \"" << s->id << "\" -> \"" << a->desc.getGlobalID() << "\"" << std::endl;
					visibleAuthorities.insert(a);
					visibleChannels.insert(s);
				}
			}

		} else {

			auto dependees = a->getDependees();
			foreach(const auto &p, dependees)
			{
				if(!hideItem(hidden, p.first->desc.getGlobalID()))
				{
					auto edge = std::make_pair(p.first,a);
					if(edges.count(edge)>0)
						continue; // skip this if we have written this edge already
					edges.insert(edge);

					// collect channel names
					std::string label;
					foreach(auto c, p.second)
					{
						if(c)
							label += (label.empty()? "" : ", ") + leafName(c->id);
					}

					if(mView->mCbElideEdgeLabels->isChecked()) {
						if(label.size()>cEdgeLabelElideLength)
							label = label.substr(0,cEdgeLabelElideLength-3) + "...";
					}

					es << "    \"" << p.first->desc.getGlobalID() << "\" -> \"" << a->desc.getGlobalID() << "\" [label=\"" << label << "\"]" << std::endl;
					visibleAuthorities.insert(a);
					visibleAuthorities.insert(p.first);
				}
			}

			auto dependers = a->getDependers();
			foreach(const auto &p, dependers)
			{
				if(!hideItem(hidden, p.first->desc.getGlobalID()))
				{
					auto edge = std::make_pair(a,p.first);
					if(edges.count(edge)>0)
						continue; // skip this if we have written this edge already
					edges.insert(edge);

					// collect channel names
					std::string label;
					foreach(auto c, p.second)
					{
						if(c)
							label += (label.empty()? "" : ", ") + leafName(c->id);
					}

					if(mView->mCbElideEdgeLabels->isChecked()) {
						if(label.size()>cEdgeLabelElideLength)
							label = label.substr(0,cEdgeLabelElideLength-3) + "...";
					}

					es << "    \"" << a->desc.getGlobalID() << "\" -> \"" << p.first->desc.getGlobalID() << "\" [label=\"" << label << "\"]" << std::endl;
					visibleAuthorities.insert(a);
					visibleAuthorities.insert(p.first);
				}
			}
		}
	}


	/////////////////////////////////////////////

	os << "digraph G {\n"
			  "    compound=true;\n"
			  "    style=filled fillcolor=\"#ffff00\" fontname=\"Helvetica\";\n"
	          "    node [style=\"filled\", fontname=\"Helvetica\"];\n"
	          "    edge [fontname=\"Helvetica\"];\n\n";

	// list all authorities and define their style
	if(mView->mCbShowNamespaces->isChecked()) {
		int id=0;
		std::stringstream tos;
		if(dotAddNamespace(tos, mGraph.getRootNamespace(), 0, id, visibleAuthorities, visibleChannels))
			os << tos.rdbuf();

	} else {
		foreach(auto a, visibleAuthorities)
		{
			if(hideItem(hidden, a->desc.getGlobalID()))
				continue;
			dotAddAuthority(os, a);
		}

		foreach(auto c, visibleChannels)
		{
			if(hideItem(hidden, c->id))
				continue;
			dotAddChannel(os, c);
		}
	}

	// copy edge content
	os << es.str();

	os << "}";

	lock.unlock();
}

void GraphViewUI::updateGraph()
{
	std::stringstream ss;
	generateDotFromGraph( ss );
	GraphView::updateGraph( QString::fromStdString( dotToSvg( ss.str() ) ) );
}

///////////////////////////////////////////////////////////////////////////////

FrameworkGraphView::FrameworkGraphView()
{
	ui = NULL;
}

FrameworkGraphView::~FrameworkGraphView()
{
	//std::cout << "delete FrameworkGraphView" << std::endl;
}

// member function that catches the frameChanged signal of the QMovie
void FrameworkGraphView::setButtonIcon(int frame)
{
	mBtUpdate->setIcon(QIcon(mBtUpdateAnimation->currentPixmap()));
}

QWidget* FrameworkGraphView::createPartControl()
{
	QWidget* w = new QWidget(this);

	QVBoxLayout* v = new QVBoxLayout();
	w->setLayout(v);

	FlowLayout* fl = new FlowLayout();

	QHBoxLayout* h = new QHBoxLayout();

	mBtUpdate = new QToolButton(w);
	mBtUpdate->setCheckable(true);
	mBtUpdate->setChecked(true);
	mBtUpdate->setToolTip("Enable Update");
	h->addWidget(mBtUpdate);


	mBtUpdateAnimation = new QMovie(":/icons/AnimatedArrows.gif");
	connect(mBtUpdateAnimation,SIGNAL(frameChanged(int)),this,SLOT(setButtonIcon(int)));
	mBtUpdateAnimation->start();
	mBtUpdateAnimation->setPaused(true);
	setButtonIcon(0);

	ui = new GraphViewUI(this, w);
	connect( ui, SIGNAL(autoScaleChanged(bool)), this, SLOT(setAutoScale(bool)) );

	mBtSaveGraph = new QToolButton(w);
	mBtSaveGraph->setToolTip("Save graph as graphviz dot file");
	mBtSaveGraph->setIcon(QIcon(":/icons/FileSave.png"));
	connect(mBtSaveGraph, SIGNAL(clicked()), ui, SLOT(saveGraph()));
	h->addWidget(mBtSaveGraph);

	mCbAutoScale = new QCheckBox("Auto Scale", w);
	mCbAutoScale->setChecked(true);
	connect(mCbAutoScale, SIGNAL(clicked(bool)), ui, SLOT(setAutoScale(bool)));

	h->addWidget(mCbAutoScale);

	fl->addItem(h);


	h = new QHBoxLayout();

	mCbDependenciesOnly = new QCheckBox("Dependencies Only", w);
	mCbDependenciesOnly->setChecked(true);
	connect(mCbDependenciesOnly, SIGNAL(stateChanged(int)), ui, SLOT(updateGraph()));
	h->addWidget(mCbDependenciesOnly);

	mCbShowNamespaces = new QCheckBox("Group Namespaces", w);
	mCbShowNamespaces->setChecked(true);
	connect(mCbShowNamespaces, SIGNAL(stateChanged(int)), ui, SLOT(updateGraph()));
	h->addWidget(mCbShowNamespaces);

	fl->addItem(h);

	h = new QHBoxLayout();

	mCbElideEdgeLabels = new QCheckBox("Elide Edge Labels", w);
	mCbElideEdgeLabels->setChecked(true);
	connect(mCbElideEdgeLabels, SIGNAL(stateChanged(int)), ui, SLOT(updateGraph()));
	h->addWidget(mCbElideEdgeLabels);

	mCbShowHiddenUnits = new QCheckBox("Show Hidden", w);
	mCbShowHiddenUnits->setChecked(false);
	connect(mCbShowHiddenUnits, SIGNAL(stateChanged(int)), ui, SLOT(updateGraph()));
	h->addWidget(mCbShowHiddenUnits);

	fl->addItem(h);

	h = new QHBoxLayout();

	h->addWidget(new QLabel("Filter Units:", w));
	mLeFilterUnits = new QLineEdit(w);
	connect(mLeFilterUnits, SIGNAL(editingFinished()), ui, SLOT(updateGraph()));
	h->addWidget(mLeFilterUnits);

	h->addWidget(new QLabel("Hide:", w));
	mLeHide = new QLineEdit(w);
	connect(mLeHide, SIGNAL(editingFinished()), ui, SLOT(updateGraph()));
	h->addWidget(mLeHide);

	fl->addItem(h,1);

	v->setMargin(0);
	v->setSpacing(0);
	v->addLayout(fl);
	v->addWidget(ui);

	return w;
}

void FrameworkGraphView::setAutoScale(bool enabled)
{
	mCbAutoScale->setCheckState( enabled ? Qt::Checked : Qt::Unchecked );
	ui->setAutoScale( enabled );
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

MIRA_CLASS_SERIALIZATION(mira::FrameworkGraphView, mira::EditorPart);

