/*
 * Copyright (C) 2017 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotGrid.h
 *    A visualization that shows a grid in the plot.
 *
 * @author Christof Schröter
 * @date   2017/12/05
 */

#ifndef _MIRA_VISUALIZATIONPLOTGRID_H_
#define _MIRA_VISUALIZATIONPLOTGRID_H_

#include <qwt_plot_grid.h>
#include <qwt_plot.h>

#include <serialization/adapters/Qt/QColor>

#include <visualization/VisualizationPlot.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <class BasePlotType>
class VisualizationPlotGrid : public BasePlotType
{

public:

	VisualizationPlotGrid() :
		mGrid(NULL),
		mColor(Qt::gray),
		mLineWidth(0.f),
		mLineStyle(Qt::DotLine),
		mXMajorEnabled(true),
		mXMinorEnabled(false),
		mYMajorEnabled(true),
		mYMinorEnabled(false)
	{
	}

	virtual ~VisualizationPlotGrid()
	{
		if (mGrid)
		{
			mGrid->detach();
			delete mGrid;
		}
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, VisualizationPlot);
		r.property("Color", mColor,
		           setter(&VisualizationPlotGrid::setColor, this),
		           "The color of the grid", Qt::gray);
		r.property("LineWidth", mLineWidth,
		           setter(&VisualizationPlotGrid::setLineWidth, this),
		           "The line width of the grid", 0.f,
		           PropertyHints::minimum(0.f));

		static const std::string styleHint = MakeString() << Qt::SolidLine << "=SolidLine;"
		                                                  << Qt::DashLine << "=DashLine;"
		                                                  << Qt::DotLine << "=DotLine;"
		                                                  << Qt::DashDotLine << "=DashDotLine;"
		                                                  << Qt::DashDotDotLine << "=DashDotDotLine";
		r.property("LineStyle", mLineStyle,
		           setter(&VisualizationPlotGrid::setLineStyle, this),
		           "The line style of the grid", Qt::DotLine,
		           PropertyHints::enumeration(styleHint));

		r.property("X Major Grid", mXMajorEnabled,
		           setter(&VisualizationPlotGrid::setXMajorEnabled, this),
		           "Enable major grid in x axis", true);
		r.property("X Minor Grid", mXMinorEnabled,
		           setter(&VisualizationPlotGrid::setXMinorEnabled, this),
		           "Enable minor grid in x axis", false);
		r.property("Y Major Grid", mYMajorEnabled,
		           setter(&VisualizationPlotGrid::setYMajorEnabled, this),
		           "Enable major grid in y axis", true);
		r.property("Y Minor Grid", mYMinorEnabled,
		           setter(&VisualizationPlotGrid::setYMinorEnabled, this),
		           "Enable minor grid in y axis", false);
	}

	virtual void setupScene(IVisualizationPlotSite* site)
	{
		mGrid = createGrid();
	}

	virtual void clear()
	{
	}

	virtual void setEnabled(bool enabled)
	{
		BasePlotType::setEnabled(enabled);
		if (!mGrid)
			return;

		mGrid->setVisible(BasePlotType::isEnabled());
		BasePlotType::getSite()->getPlot()->replot();
	}

	virtual int getYAxis()
	{
		return -1;
	}

	virtual QString getCurveTitle()
	{
		return "Grid";
	}

	void redraw()
	{
		if (!mGrid)
			return;

		mGrid->setPen(QPen(mColor, mLineWidth, mLineStyle));
		mGrid->enableX(mXMajorEnabled);
		mGrid->enableXMin(mXMinorEnabled);
		mGrid->enableY(mYMajorEnabled);
		mGrid->enableYMin(mYMinorEnabled);
		BasePlotType::getSite()->getPlot()->replot();
	}

	void setColor(const QColor& color)
	{
		mColor = color;
		redraw();
	}

	void setLineWidth(float width)
	{
		mLineWidth = width;
		redraw();
	}

	void setLineStyle(const Qt::PenStyle& style)
	{
		mLineStyle = style;
		redraw();
	}

	void setXMajorEnabled(bool enable)
	{
		mXMajorEnabled = enable;
		redraw();
	}

	void setXMinorEnabled(bool enable)
	{
		mXMinorEnabled = enable;
		redraw();
	}

	void setYMajorEnabled(bool enable)
	{
		mYMajorEnabled = enable;
		redraw();
	}

	void setYMinorEnabled(bool enable)
	{
		mYMinorEnabled = enable;
		redraw();
	}

	virtual QwtPlotItem* getPlotItem()
	{
		return mGrid;
	}

	QwtPlotGrid* createGrid()
	{
		mGrid = new QwtPlotGrid();
		mGrid->setPen(QPen(mColor, mLineWidth, mLineStyle));
		mGrid->setVisible(BasePlotType::isEnabled());
		mGrid->enableX(mXMajorEnabled);
		mGrid->enableXMin(mXMinorEnabled);
		mGrid->enableY(mYMajorEnabled);
		mGrid->enableYMin(mYMinorEnabled);
		mGrid->attach(BasePlotType::getSite()->getPlot());
		BasePlotType::getSite()->getPlot()->replot();
		mGrid->setZ(20.); // seems to be the default z for QwtPlotCurve (undocumented)
		return mGrid;
	}

protected:
	QwtPlotGrid* mGrid;
	QColor mColor;
	float mLineWidth;
	Qt::PenStyle mLineStyle;
	bool mXMajorEnabled;
	bool mXMinorEnabled;
	bool mYMajorEnabled;
	bool mYMinorEnabled;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
