/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VelocityTask.h
 *    A task that limits the maximum velocity.
 *
 * @author Tim Langner
 * @date   2014/04/30
 */

#ifndef _MIRA_VELOCITYTASK_H_
#define _MIRA_VELOCITYTASK_H_

#include <navigation/Task.h>
#include <transform/Velocity.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Subtask that has purpose to limit the speed of the robot.
 */
class MIRA_NAVIGATION_EXPORT VelocityTask : public SubTask
{
	MIRA_OBJECT(VelocityTask);
public:

	/**
	 * @brief Constructs a velocity task with a maximum translational velocity
	 *        of 1m/s and rotational velocity of 90deg/s.
	 */
	VelocityTask() :
		velocity(1.0f, 0.0, deg2rad(90.0f)) {}

	/**
	 * @brief Constructs a velocity task with given velocity
	 * @param[in] v The requested max velocity.
	 */
	VelocityTask(const Velocity2& v) :
		velocity(v)
	{}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SubTask);
		r.property("Velocity", velocity, "The requested max velocity.");
	}

public:
	/// The requested max velocity.
	Velocity2 velocity;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
