/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 *  and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Redistribution and modification of this code is strictly prohibited.
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TransformTreeVisualization.C
 *    Visualization of the whole transform tree.
 *
 * @author Tim Langner
 * @date   2012/04/03
 */

#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>

#include <fw/Framework.h>

#include <visualization/Visualization3D.h>
#include <visualization/3d/AxesObject.h>
#include <visualization/3d/ArrowObject.h>
#include <visualization/3d/TextObject.h>

#include "private/PoseCovVisualization.h"

#include <widgets/OgreUtils.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <serialization/DefaultInitializer.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

class TransformTreeVisualization : public Visualization3D
{
MIRA_META_OBJECT(TransformTreeVisualization,
	("Name", "TransformTree")
	("Description", "Shows all transform frames"))

public:

	struct FrameInfo
	{
		boost::shared_ptr<AxesObject> axes;
		boost::shared_ptr<ArrowObject> parentArrow;
		boost::shared_ptr<TextObject> text;
		boost::shared_ptr<CovVisualization> cov;
		std::string parent;
	};

	TransformTreeVisualization() :
		mNode(NULL),
		mLastUpdate(Time::unixEpoch())
	{
		MIRA_INITIALIZE_THIS;
	}

	virtual ~TransformTreeVisualization()
	{
		if(getSite()==NULL)
			return;
		mFrames.clear();
		getSite()->getSceneManager()->destroySceneNode(mNode);
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3D);
		r.property("UpdateRate", mUpdateRate, "How often are frames updated", Duration::seconds(1));
		r.property("TransformType", mTransformType, "Type of the transforms that are rendered",
		           POSE3, PropertyHints::enumeration("Pose2;PoseCov2;Pose3;PoseCov3"));
		r.property("ShowAxes", mShowAxes, "Should axes be rendered", true);
		r.property("AxesLength", mAxesLength, "Length of the axes", 0.2f);
		r.property("AxesRadius", mAxesRadius, "Radius of the axes", 0.01f);
		r.property("ShowText", mShowText, "Should text be rendered", true);
		r.property("TextSize", mTextSize, "The size of the text marks", 0.1f);
		r.property("TextColor", mTextColor, "The color of the text marks", Ogre::ColourValue::Black);
		r.property("ShowArrowHeads", mShowArrowHeads, "Should arrow heads be rendered", true);
		r.property("ShowParentArrows", mShowParentArrows, "Should arrows from child to parent be rendered", true);
		r.property("ParentArrowsRadius", mParentArrowsRadius, "Radius of the arrows from child to parent", 0.005f);
	}

public:

	virtual void setupScene(IVisualization3DSite* site) {
		Ogre::SceneManager* mgr = site->getSceneManager();
		mNode = mgr->getRootSceneNode()->createChildSceneNode();
		mNode->setVisible(isEnabled());
	}

	virtual Ogre::SceneNode* getNode()
	{
		return mNode;
	}

public:

	void setEnabled(bool enabled) {
		Visualization3D::setEnabled(enabled);
		if (mNode != NULL)
			mNode->setVisible(enabled);
	}

protected:

	virtual void update(Duration dt)
	{
		if (Time::now() - mLastUpdate > mUpdateRate)
		{
			checkFrames();
			mLastUpdate = Time::now();
		}
		if (isEnabled())
		{
			switch(mTransformType)
			{
			case POSE2:
				updateFrames<Pose2>();
				break;
			case POSE2COV:
				updateFrames<PoseCov2>();
				break;
			case POSE3:
				updateFrames<Pose3>();
				break;
			case POSE3COV:
				updateFrames<PoseCov3>();
				break;
			case INVALID : break;
			}
		}
	}

	void checkFrames();

	template <typename Pose>
	void updateFrames();

protected:

	void addFrame(const std::string& name, const std::string& parent = "");
	void addParent(FrameInfo& info, const std::string& parent);

	Ogre::SceneNode* mNode;
	Duration mUpdateRate;
	Time mLastUpdate;
	bool mShowAxes;
	float mAxesLength;
	float mAxesRadius;
	bool mShowText;
	float mTextSize;
	Ogre::ColourValue mTextColor;
	bool mShowArrowHeads;
	bool mShowParentArrows;
	float mParentArrowsRadius;
	TransformType mTransformType;
	std::map<std::string, FrameInfo> mFrames;
};

///////////////////////////////////////////////////////////////////////////////


void TransformTreeVisualization::addFrame(const std::string& name,
                                          const std::string& parent)
{
	FrameInfo info;
	info.axes.reset(new AxesObject(getSite()->getSceneManager(), mNode));
	info.axes->setLength(mAxesLength);
	info.axes->setRadius(mAxesRadius);
	info.axes->showArrows(mShowArrowHeads);
	info.axes->setVisible(mShowAxes && isEnabled());
	info.cov.reset(new CovVisualization());
	info.cov->setupScene(getSite()->getSceneManager(),info.axes.get());
	info.cov->setVisible(isEnabled() &&
	                     (mTransformType == POSE2COV || mTransformType == POSE3COV));
	info.text.reset(new TextObject(name, getSite()->getSceneManager(), mNode));
	info.text->setTextAlignment(TextObject::H_LEFT, TextObject::V_BELOW);
	info.text->setLocalTranslation(Ogre::Vector3(mAxesRadius, 0.f, 0.f));
	info.text->setCharacterHeight(mTextSize);
	info.text->setColor(mTextColor);
	info.text->setVisible(isEnabled() && mShowText);
	if (!parent.empty())
		addParent(info, parent);
	mFrames[name] = info;
}

void TransformTreeVisualization::addParent(FrameInfo& info, const std::string& parent)
{
	info.parent = parent;
	if (!info.parentArrow)
	{
		info.parentArrow.reset(new ArrowObject(getSite()->getSceneManager(), mNode));
		info.parentArrow->setRadius(mParentArrowsRadius);
		info.parentArrow->setVisible(mShowParentArrows && isEnabled());
		info.parentArrow->setColor(Ogre::ColourValue(1.0f,1.0f,0.0f,1.0f));
	}
}

void TransformTreeVisualization::checkFrames()
{
	auto nodes = MIRA_FW.getTransformer()->getNodes();
	foreach(TransformerBase::AbstractNodePtr node, nodes)
	{
		if (mFrames.count(node->getID()) == 0)
			addFrame(node->getID());
		auto children = node->getChildren();
		foreach(TransformerBase::AbstractNodePtr child, children)
		{
			auto it = mFrames.find(child->getID());
			if (it == mFrames.end())
				addFrame(child->getID(), node->getID());
			else
				addParent(it->second, node->getID());
		}
	}
}

template <typename Pose>
inline void TransformTreeVisualization::updateFrames()
{
	typedef std::pair<const std::string,FrameInfo> Pair;
	foreach(Pair& frame, mFrames)
	{
		Pose framePose;
		try
		{
			framePose = getAuthority().getTransform<Pose>(frame.first, getSite()->getCameraFrame());
			frame.second.axes->setTransform(framePose);
			frame.second.axes->setLength(mAxesLength);
			frame.second.axes->setRadius(mAxesRadius);
			frame.second.axes->setVisible(mShowAxes && isEnabled());
			frame.second.axes->showArrows(mShowArrowHeads);
			frame.second.cov->update(framePose);
			frame.second.cov->setVisible(isEnabled() && CovVisTrait<Pose>::poseHasCovariance);
			frame.second.text->setTransform(framePose);
			frame.second.text->setLocalTranslation(Ogre::Vector3(mAxesRadius, 0.f, 0.f));
			frame.second.text->setCharacterHeight(mTextSize);
			frame.second.text->setColor(mTextColor);
			frame.second.text->setVisible(mShowText && isEnabled());
			if (Pose::Dim>2)
				frame.second.axes->showZAxis(true);
			else
				frame.second.axes->showZAxis(false);
		}
		catch(Exception&)
		{
			frame.second.axes->setVisible(false);
			frame.second.text->setVisible(false);
			if (frame.second.parentArrow)
				frame.second.parentArrow->setVisible(false);
			continue;
		}
		if (!frame.second.parentArrow)
			continue;
		try
		{
			Pose parentPose = getAuthority().getTransform<Pose>(frame.second.parent, getSite()->getCameraFrame());
			Ogre::Vector3 parentPosition = OgreUtils::toOgreVector(parentPose);
			Ogre::Vector3 framePosition = OgreUtils::toOgreVector(framePose);

			Ogre::Vector3 direction = parentPosition - framePosition;
			float distance = direction.length();
			direction.normalise();

			Ogre::Quaternion orient = Ogre::Vector3::UNIT_Y.getRotationTo(direction);
			frame.second.parentArrow->setLength(distance);
			if ( distance > 0.001f )
				frame.second.parentArrow->setVisible(mShowParentArrows && isEnabled());
			else
				frame.second.parentArrow->setVisible(false);

			frame.second.parentArrow->setPosition(framePosition);
			frame.second.parentArrow->setOrientation(orient);
		}
		catch(Exception&)
		{
			frame.second.parentArrow->setVisible(false);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::gui::TransformTreeVisualization, mira::Visualization3D);

