/*
 * Copyright (C) 2013 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RangeCircleVisualization.C
 *
 * @author Erik Einhorn
 * @date   2013/06/02
 */

#include <iomanip>

#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>

#include <fw/TransformProperty.h>

#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <serialization/DefaultInitializer.h>
#include <serialization/SetterNotify.h>

#include <visualization/Visualization3D.h>
#include <visualization/3d/LineStripObject.h>
#include <visualization/3d/TextObject.h>
#include <widgets/OgreUtils.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

class RangeCircleVisualization :  public Visualization3D
{
MIRA_META_OBJECT(RangeCircleVisualization,
		("Name", "Range Circles")
		("Description", "Visualizes one or more range circles around the specified frame."))
public:

	RangeCircleVisualization() : mgr(NULL), node(NULL)
	{
		MIRA_INITIALIZE_THIS;
	}

	~RangeCircleVisualization()
	{
		if(mgr) {
			delete lineObject;
			mgr->destroySceneNode(node);
		}
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3D);
		r.property("Frame", frame, "Optional frame, where to center this visualization", TransformProperty());
		r.property("Count", count, setterNotify(count, &RangeCircleVisualization::updateGeometry, this),
		           "The number of circles", 1);
		r.property("Radius", radius, setterNotify(radius, &RangeCircleVisualization::updateGeometry, this),
		           "The radius of the first circle, the others will be visualized in multiples of this radius.", 1.0f);
		r.property("Color", color, setterNotify(color, &RangeCircleVisualization::updateGeometry, this),
		           "The color of the circles.", Ogre::ColourValue::Blue);
		r.property("Line Width", lineWidth, setterNotify(lineWidth, &RangeCircleVisualization::updateLineWidth, this),
		           "Line Width", 0.0f, PropertyHints::step(0.02f));
		r.property("Dash Spacing", dashSpacing, setterNotify(dashSpacing, &RangeCircleVisualization::updateGeometry, this),
		           "Enables dashed lines if > 0", 0.0f);
		r.property("Show Text", showText, setterNotify(showText, &RangeCircleVisualization::updateGeometry, this),
		           "Show textual marks", true);
		r.property("Font Size", fontSize, setterNotify(fontSize, &RangeCircleVisualization::updateGeometry, this),
		           "The size of the text marks", 0.2f);
	}

	virtual void setupScene(IVisualization3DSite* site)
	{
		mgr = site->getSceneManager();
		node = mgr->getRootSceneNode()->createChildSceneNode();
		lineObject = new LineStripObject(mgr, node);
		lineObject->setLineWidth(lineWidth);
		updateGeometry();
		setEnabled(isEnabled());
	}

	virtual Ogre::SceneNode* getNode()
	{
		return node;
	}

	virtual void update(Duration dt) {
		if(frame.isValid()) {
			RigidTransform3f d = getAuthority().getTransform<RigidTransform3f>(
				frame.getID(), Time(),
				getSite()->getCameraFrame(), Time(),
				getSite()->getFixedFrame());

			OgreUtils::setTransform(node, d);
		}
	}

	void setEnabled(bool enabled) {
		Visualization3D::setEnabled(enabled);
		if (node != NULL) {
			node->setVisible(enabled);
			lineObject->setVisible(enabled);
			for(std::size_t i=0; i<textObjects.size(); ++i)
				textObjects[i]->setVisible(enabled && showText && (i<(std::size_t)count));
		}
	}

	void updateGeometry()
	{
		if(count<1)
			count = 1;

		if(mgr==NULL) // not yet initialized
			return;

		createTextObjects();

		lineObject->clear();
		lineObject->setColor(color);

		int sectors=64;

		for(int c=0; c<count; ++c)
		{
			float r = radius * (c+1);
			float l = 2*pi<float>() * r;

			// update text
			std::stringstream ss;
			ss << std::fixed << std::setw(3) << std::setprecision(2) << r;
			textObjects[c]->setCaption(ss.str());
			textObjects[c]->setPosition(Ogre::Vector3(r,0.0f,0.0f));
			textObjects[c]->setCharacterHeight(fontSize);
			textObjects[c]->setColor(color);

			int dashes;
			int sectorsPerDash;
			float dbeta;
			float dalpha;

			if(dashSpacing>0.01f) {
				dashes = (int)std::floor(l / (dashSpacing * 2.0f));
				sectorsPerDash = std::max(sectors / dashes, 1);
				// delta angle between beginning of a dash and the next dash
				dbeta  = 2.0f*pi<float>() / dashes;
				dalpha = 0.5f*dbeta / sectorsPerDash;
			} else { // full circle (no dashed line)
				dashes = 1;
				sectorsPerDash = sectors;
				dbeta = 0.0f;
				dalpha = 2.0f*pi<float>() / sectors;
			}

			for(int d=0; d<dashes; ++d)
			{
				float alpha0 = d*dbeta;
				lineObject->begin();
				Ogre::Vector3 p(std::sin(alpha0)*r,std::cos(alpha0)*r,0.0f);
				lineObject->point(p);
				for(int i=0; i<sectorsPerDash; ++i)
				{
					float alpha = alpha0 + i*dalpha;

					Ogre::Vector3 p(std::sin(alpha+dalpha)*r,std::cos(alpha+dalpha)*r,0.0f);
					lineObject->point(p);
				}
				lineObject->end();
			}
		}
	}

	void updateLineWidth()
	{
		if(mgr==NULL) // not yet initialized
			return;

		lineObject->setLineWidth(lineWidth);
		setEnabled(isEnabled());
	}

	void createTextObjects()
	{
		assert(count>0);
		for(std::size_t i = textObjects.size(); i < (std::size_t)count; ++i) {
			TextObject* o = new TextObject("8.88", mgr, node);
			textObjects.push_back(o);
			o->setTextAlignment(TextObject::H_CENTER, TextObject::V_ABOVE);
		}
		setEnabled(isEnabled());
	}

	Ogre::SceneManager* mgr;
	Ogre::SceneNode* node;
	LineStripObject* lineObject;
	std::vector<TextObject*> textObjects;

	TransformProperty frame;
	int count;
	float radius;
	Ogre::ColourValue color;
	float lineWidth;
	float dashSpacing;
	bool showText;
	float fontSize;
};


///////////////////////////////////////////////////////////////////////////////

} } // namespace


MIRA_CLASS_SERIALIZATION(mira::gui::RangeCircleVisualization, mira::Visualization3D);

