/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MeshVisualization.C
 *    Description.
 *
 * @author Erik Einhorn
 * @date   2011/01/22
 */

#include <boost/scoped_ptr.hpp>

#include <utils/Path.h>
#include <serialization/Serialization.h>

#include <visualization/3d/MeshObject.h>
#include <visualization/3d/MeshFactory.h>
#include <visualization/Visualization3D.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <serialization/SetterNotify.h>
#include <serialization/DefaultInitializer.h>

#include <widgets/OgreUtils.h>

#include <transform/RigidTransform.h>
#include <fw/TransformProperty.h>

#include <OGRE/OgreSceneManager.h>

using namespace Eigen;
using namespace std;

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

class Pose3Property : public Pose3{
public:
	Pose3Property() : Pose3(), mYaw(0.0), mPitch(0.0), mRoll(0.0){};

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Pose3);
		r.property("Yaw", mYaw,
			setterNotify(mYaw, &Pose3Property::updateOrientation, this),
			"Description",0.0);
		r.property("Pitch", mPitch,
			setterNotify(mPitch, &Pose3Property::updateOrientation, this),
			"Description",0.0);
		r.property("Roll", mRoll,
			setterNotify(mRoll, &Pose3Property::updateOrientation, this),
			"Description",0.0);
	}
private:
	void updateOrientation(){
		this->r = quaternionFromYawPitchRoll(deg2rad(mYaw),deg2rad(mPitch),deg2rad(mRoll));
	}

	float mYaw;
	float mPitch;
	float mRoll;
};


class MeshVisualization :  public Visualization3D
{
	MIRA_META_OBJECT(MeshVisualization,
		("Name", "Mesh")
		("Description", "Visualization of a mesh that can be loaded from file."))

public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor
	MeshVisualization() :
		mMesh(NULL)
	{
		const std::vector<MeshFactory::MeshFormatInfo> formats =
				MeshFactory::instance().getSupportedFormats();

		mSupportedExtensions = "";
		std::string allFormats = "Supported Meshes (";
		foreach(const MeshFactory::MeshFormatInfo& format, formats)
		{
			mSupportedExtensions += ";;" + format.description + " (*" + format.extension +")";
			allFormats += "*" + format.extension + " ";
		}

		mSupportedExtensions = allFormats + ")" + mSupportedExtensions;

		MIRA_INITIALIZE_THIS;
	}

	/// The destructor.
	virtual ~MeshVisualization()
	{
		if(getSite()==NULL)
			return;

		delete mMesh;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3D);

		r.property("Mesh",  mMeshPath, setter(&MeshVisualization::setPath, this),
		           "The path to the file containing the mesh.", "",
		           PropertyHints::file(mSupportedExtensions));
		r.property("Frame", mFrameID, "The frame id to obtain the pose", "");
		r.property("Pose",  mPose,
		           "The pose of the mesh (can be specified manually if no frame is given)",
		           RigidTransform3f());
		r.property("Color", mColor, "The color of the mesh visualization (Only takes affect on meshes without materials).",
		           Ogre::ColourValue::Blue);
		r.property("Offset", mOffset, "Offset of the mesh relative to the frame", Pose3Property());
		r.property("Scale", mScale, "Scale factor of the mesh", 1.f);
	}

	//@}

public:
	/** @name Public implementation of Visualization3D and Visualization*/
	//@{

	virtual void setupScene(IVisualization3DSite* site)
	{
		setPath(mMeshPath);
	}

	virtual Ogre::SceneNode* getNode()
	{
		if (!mMesh)
			return NULL;

		return mMesh->getNode();
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		if(mMesh!=NULL)
			mMesh->setVisible(enabled);
	}

	//@}

public:
	/** @name Setters */
	//@{

	void setPath(const Path& path)
	{
		mMeshPath = path;

		if(path.empty() || getSite()==NULL)
			return;

		delete mMesh;

		Ogre::SceneManager* mgr = getSite()->getSceneManager();
		std::string s = mMeshPath.string();

		try {
			mMesh = new MeshObject(s, mgr, mgr->getRootSceneNode());
			mMesh->setVisible(isEnabled());
		} catch(Exception& ex) {
			error("Mesh", ex.message());
			mMesh = NULL;
		}
		ok("Mesh");
	}

	//@}

protected:
	/** @name Protected implementation Visualization*/
	//@{

	virtual void update(Duration dt)
	{
		if(mMesh==NULL)
			MIRA_THROW(XInvalidRead, "Mesh not set or not loaded");

		// get the transform of the range scan
		if(mFrameID.isValid())
			mPose = getAuthority().getTransform<RigidTransform3f>(mFrameID.getID(), getSite()->getCameraFrame());

		mPose *= mOffset;

		if(mMesh!=NULL){
			mMesh->setTransform(mPose);
			mMesh->setColor(mColor);
			mMesh->setScale(Ogre::Vector3(mScale , mScale, mScale));
		}

	}

	//@}

private:
	TransformProperty mFrameID;
	std::string mSupportedExtensions;

	MeshObject* mMesh;
	Path mMeshPath;
	RigidTransform3f mPose;
	Ogre::ColourValue mColor;
	float mScale;
	Pose3Property mOffset;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::MeshVisualization, mira::Visualization3D);
