#!/bin/bash

# we need miracd from mirabash, so import that file
DIR=$(dirname "${BASH_SOURCE[0]}")
source $DIR/../mirabash

# exit on Ctrl+C
trap ctrl_c INT
function ctrl_c() {
	exit 2
}


function usage
{
	echo -e "\
Usage: miragit <command> [<package> [<package> ...]] [options]\n\
Runs 'git' for the specified packages or for all packages (if no package is specified).\n\n\
General Options:\n\
  --at <location>: only process packages in a certain location (directory). Can be used with absolute or relative paths (also e.g. '.')
  --auth: interactively query username and email and provide to git (useful for committing on a machine where these are not global settings)
\n\
Available commands:\n\
    status - status of working tree\n\
    commit - record changes to repository(use with care!)\n\
    push   - update remote refs(use with care!)\n\
    pull   - fetch from and integrate with remote repository\n\
\n\
    all other git commands are supported too, type 'git help <command>' for details.\n"
}

function gitpackage
{
	miracd $1 &> /dev/null
	if [[ $? != 0 ]]; then
		echo -e "\e[01;31mPackage '$1' No such package found\e[00m"
	else
		here=`realpath $PWD`
		if [[ $here"/" != "$4"/* ]]; then
			return
		fi

		### check if we are in a git directory
		git status 2>&1 1>&/dev/null; ret=$?

		if [[ $ret == 0 ]]; then

			echo -e "\e[01;32mPackage '$1' \e[00;32m[$PWD]\e[00m"
			
			if [[ $2 = "pull" ]]; then
				OUTPUT=`git $2 $3`
				echo "$OUTPUT" | egrep --invert-match -e "Already up-to-date." --invert-match -e "Already up to date."
			else
				[ -v "$5" ] && git -c user.name="$5" -c user.email=$6 $2 $3 || git $2 $3 
			fi
		fi
	fi
}

if [[ $1 = "--help" ]] || [[ $# -lt 1 ]]; then
	usage
	exit 0
fi

configfile=${HOME}/.gitconfig
configfilebak=${HOME}/.gitconfig.miragit
configfile_exists=0
 
if [[ -f ${configfile} ]]; then
	#echo "config exists - backing up"
	mv ${configfile} ${configfilebak} # keep original timestamp
	cp ${configfilebak} ${configfile}
	configfile_exists=1
fi

git config --global credential.helper cache

# first param is the command
command="$1"

# other params may be packages, the others are the options for git
# find the first parameter, that does contain a -, it will be the first option to 
# pass to git
paramcount=$#
firstoption=$((${paramcount}+1))

for (( i=2; i<=${paramcount}; i++ ))
do
	if [[ ${!i} == -* ]]; then
		firstoption=$i
		break
	fi
done

firstgitoption=$firstoption

username=""
email=""
at=""

while :
do
	if [[ ${!firstgitoption} == "--auth" ]]; then
		read -p "user.name (Firstname Secondname): " username
		read -p "user.email: " email
		echo ""
		firstgitoption=$(($firstgitoption+1))
	elif [[ ${!firstgitoption} == "--at" ]]; then
		firstgitoption=$(($firstgitoption+1))
		at=`realpath ${!firstgitoption}`
		firstgitoption=$(($firstgitoption+1))
		echo "Restrict to packages in directory" $at
	else
		break
	fi
done

gitoptions="${@:${firstgitoption}}"

# handle additional commands
if [[ $command == "changes" ]]; then
	command="log -r BASE:HEAD"
fi

if [[ $firstoption == 2 ]]; then
	echo "Running 'git $command $gitoptions' for all packages ..."
	echo "Reindexing packages ..." 
	$_mirafindpkg --reindex
	allpackages=($(bash $_mirafindpkg | awk -- '{print $1}'))
	for package in "${allpackages[@]}"	
	do
		gitpackage $package "$command" "$gitoptions" "$at" "$username" $email
	done
else 
	# for each given package
	echo "Running 'git $command $gitoptions' ..."
	for (( i=2; i<${firstoption}; i++ ))
	do
		package="${!i}"
		gitpackage $package "$command" "$gitoptions" "$at" "$username" $email
	done
fi

git credential-cache exit

rm ${configfile}
if [[ ${configfile_exists} -eq 1 ]]; then
	#echo "restoring config"
	mv ${configfilebak} ${configfile} 
fi

