/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TreeViewFilter.C
 *    Implementation of TreeViewFilter.
 *
 * @author Erik Einhorn
 * @date   2012/02/19
 */

#include <widgets/TreeViewFilter.h>

#include <assert.h>

#include <QTimerEvent>
#include <QVBoxLayout>
#include <QTreeView>

#include <widgets/LineEditClear.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TreeViewFilter::TreeViewFilter(QTreeView* view, QWidget* parent) :
	QWidget(parent), mTreeView(view)
{
	assert(mTreeView!=NULL);

	mExpandAll = false;
	mAutoHide = false;
	mExpandChildren = true;
	mUseFilterRole = false;

	mFilterEdit = new LineEditClear(this);
	mFilterEdit->setToolTip(tr("Enter a search string to show matching items only"));

	QVBoxLayout* verticalLayout = new QVBoxLayout(this);
	verticalLayout->setSpacing(0);
	verticalLayout->setMargin(0);
	verticalLayout->addWidget(mFilterEdit);

	connect(mFilterEdit, SIGNAL(textChanged(const QString &)),
	               this, SLOT(onFilterTextChanged(const QString &)));

	mFilterTimerId = -1;
	mHideTimerId = startTimer(1000);
	setFocusProxy(mFilterEdit);
}

void TreeViewFilter::timerEvent(QTimerEvent *e)
{
	if(e->timerId()==mFilterTimerId) {
		applyFilter(mFilterEdit->text(), QModelIndex(), false);
		killTimer(mFilterTimerId);
		mFilterTimerId = -1;
		return;
	}
	else if(e->timerId()==mHideTimerId) {
		autoHide();
	}
}

void TreeViewFilter::setAutoHide(bool hide)
{
	mAutoHide = hide;
	if(hide)
		autoHide();
	else
		showFilter();
}

void TreeViewFilter::showFilter()
{
	mFilterEdit->show();
	mFilterEdit->setFocus();
}

void TreeViewFilter::hideFilter()
{
	mFilterEdit->clear();
	mFilterEdit->hide();
}

void TreeViewFilter::useFilterRole(bool use)
{
	if ( mUseFilterRole != use ) {
		mUseFilterRole = use;
		startFilterUpdateTimer(0);
	}
}

bool TreeViewFilter::getAutoHide() const
{
	return mAutoHide;
}

void TreeViewFilter::setExpandAll(bool expandAll)
{
	mExpandAll = expandAll;
}

void TreeViewFilter::setExpandChildren(bool expand)
{
	mExpandChildren = expand;
}

void TreeViewFilter::setFilterColumns(const std::vector<int>& cols)
{
	mFilterColumns = cols;
}

void TreeViewFilter::setFilterColumns(int col1)
{
	mFilterColumns.resize(1);
	mFilterColumns[0] = col1;
}

void TreeViewFilter::setFilterColumns(int col1, int col2)
{
	mFilterColumns.resize(2);
	mFilterColumns[0] = col1;
	mFilterColumns[1] = col2;
}

void TreeViewFilter::setFilterColumns(int col1, int col2, int col3)
{
	mFilterColumns.resize(3);
	mFilterColumns[0] = col1;
	mFilterColumns[1] = col2;
	mFilterColumns[2] = col3;
}

void TreeViewFilter::autoHide()
{
	if(!mAutoHide)
		return;

	if(!mFilterEdit->hasFocus() && mFilterEdit->text().isEmpty())
		hideFilter();
}

void TreeViewFilter::onFilterTextChanged(const QString& text)
{
	startFilterUpdateTimer(500);
}

bool TreeViewFilter::applyFilter(const QString& filter,
                                 const QModelIndex& parent, bool forceVisible)
{
	bool visible = false;

	QAbstractItemModel* model = mTreeView->model();

	int childCount = model->rowCount(parent);
	int colCount = model->columnCount(parent);

	for(int i=0;i<childCount;++i)
	{
		QModelIndex child = model->index(i,0,parent);
		assert(child.isValid());

		bool childVisible = false;
		bool match = false;

		if(!filter.isEmpty() && !forceVisible)
		{
			for(int j=0; ;++j)
			{
				int col = 0;
				// if mFilterColumns is empty, then we check all existing columns
				if(mFilterColumns.empty())
				{
					if(j>=colCount)
						break; // end
					col = j;
				} else { // othwerise, we check all columns, specified in the vector
					if(j>=(int)mFilterColumns.size())
						break; // end

					col = mFilterColumns[j];
					if(col<0 || col>=colCount)
						continue;
				}


				QModelIndex idx = model->index(i,col,parent);

				// if we have a filter pattern specified, check if child's text matches.
				// skip matching, if we are forced to be visible anyway

				// get the displayed / filter text
				QVariant v = model->data(idx, mUseFilterRole ? (Qt::ItemDataRole)FilterRole : Qt::DisplayRole);
				if(v.isValid()) {
					QString text = v.toString();
					int r = text.indexOf(filter, 0, Qt::CaseInsensitive);
					if(r>=0) {
						match = true;
						break;
					}
				}
			}
		}

		if(filter.isEmpty() || match || forceVisible) {
			childVisible = true;
			// we have a match (or no pattern), so show this child and force
			// all of its sub children to be shown
			applyFilter(filter, child, mExpandChildren);
		} else {
			// no match, so check its sub children, if there is a match
			// our children will be shown only, if at least one of its
			// sub children matches.
			childVisible = applyFilter(filter, child, forceVisible);
		}

		mTreeView->setRowHidden(i,parent, !childVisible);
		visible |= childVisible;
	}

	if(mExpandAll || !filter.isEmpty())
		mTreeView->setExpanded(parent, visible);

	return visible;
}

void TreeViewFilter::startFilterUpdateTimer(int delay)
{
	if(mFilterTimerId>=0)
		killTimer(mFilterTimerId);

	mFilterTimerId = startTimer(500);
}

///////////////////////////////////////////////////////////////////////////////

}
