/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPropertyDelegate.C
 *    Implementation of VisualizationPropertyDelegate.
 *
 * @author Erik Einhorn
 * @date   2011/01/05
 */

#include <QIcon>
#include <QApplication>
#include <QPainter>
#include <QInputDialog>
#include <QStyle>
#include <QStyleOptionButton>

#include <visualization/private/VisualizationPropertyDelegate.h>
#include <widgets/PropertyEditor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// Return an icon containing a check box indicator
static QIcon drawCheckBox(bool value)
{
	QStyleOptionButton opt;
	opt.state |= value ? QStyle::State_On : QStyle::State_Off;
	opt.state |= QStyle::State_Enabled;
	const QStyle *style = QApplication::style();
	// Figure out size of an indicator and make sure it is not scaled down in a list view item
	// by making the pixmap as big as a list view icon and centering the indicator in it.
	// (if it is smaller, it can't be helped)
	const int indicatorWidth = style->pixelMetric(QStyle::PM_IndicatorWidth, &opt);
	const int indicatorHeight = style->pixelMetric(QStyle::PM_IndicatorHeight, &opt);
	const int listViewIconSize = indicatorWidth;
	const int pixmapWidth = indicatorWidth;
	const int pixmapHeight = qMax(indicatorHeight, listViewIconSize);

	opt.rect = QRect(0, 0, indicatorWidth, indicatorHeight);
	QPixmap pixmap = QPixmap(pixmapWidth, pixmapHeight);
	pixmap.fill(Qt::transparent);
	{
		const int xoff = (pixmapWidth  > indicatorWidth)  ? (pixmapWidth  - indicatorWidth)  / 2 : 0;
		const int yoff = (pixmapHeight > indicatorHeight) ? (pixmapHeight - indicatorHeight) / 2 : 0;
		QPainter painter(&pixmap);
		painter.translate(xoff, yoff);
		style->drawPrimitive(QStyle::PE_IndicatorCheckBox, &opt, &painter);
	}
	return QIcon(pixmap);
}

///////////////////////////////////////////////////////////////////////////////

typedef TypedPropertyNode<Visualization*> PropertyType;

VisualizationPropertyDelegate_CheckBox::VisualizationPropertyDelegate_CheckBox(PropertyNode* p,
                                                                               QWidget* parent) :
	QCheckBox(parent)
{
	property = p->toTyped<Visualization*>();
	assert(property!=NULL);

	bool enabled = property->get()->isEnabled();
	setChecked(enabled);
	if(enabled)
		setText("enabled");
	else
		setText("disabled");

	connect(this, SIGNAL(clicked(bool)), this, SLOT(slotSetValue(bool)));
}

void VisualizationPropertyDelegate_CheckBox::slotSetValue(bool value)
{
	property->get()->setEnabled(value);
	if(value)
		setText("enabled");
	else
		setText("disabled");
}

///////////////////////////////////////////////////////////////////////////////

class VisualizationPropertyDelegate : public PropertyEditor::Delegate
{
	MIRA_OBJECT(VisualizationPropertyDelegate)

public:

	virtual SupportedTypes supportedTypes() const
	{
		return makeSupportedType<Visualization*>(SupportedType::TEXT |
		                                         SupportedType::ICON |
		                                         SupportedType::EDITOR |
		                                         SupportedType::COLOR |
		                                         SupportedType::DIALOG |
		                                         SupportedType::NAME );
	}

	virtual std::string getText(const PropertyNode* property)
	{
		const PropertyType* p = property->toTyped<Visualization*>();
		assert(p!=NULL);
		if(p->get()->isEnabled())
			return "enabled";
		else
			return "disabled";
		return "";
	}

	virtual QIcon getIcon(const PropertyNode* property)
	{
		const PropertyType* p = property->toTyped<Visualization*>();
		assert(p!=NULL);
		return drawCheckBox(p->get()->isEnabled());
	}

	virtual QColor getBackgroundColor(const PropertyNode* property)
	{
		const PropertyType* p = property->toTyped<Visualization*>();
		assert(p!=NULL);

		Status::StatusMode status = p->get()->getStatusManager().getStatus();

		if(status == Status::WARNING)
			return PropertyEditor::lightYellow;
		else if (status == Status::ERROR)
			return PropertyEditor::lightRed;
		else if (status == Status::BOOTUP)
			return PropertyEditor::lightBlue;
		else if (status == Status::RECOVER)
			return PropertyEditor::lightPurple;
		else return QColor(); // if okay, then use default color
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent)
	{
		return new VisualizationPropertyDelegate_CheckBox(property, parent);
	}

	virtual bool execDialog(PropertyNode* property, QWidget* parent)
	{
		const PropertyType* p = property->toTyped<Visualization*>();
		assert(p!=NULL);
		Visualization* vis = p->get();
		QString text = QInputDialog::getText(parent, QWidget::tr("Rename Visualization"),
		                                     QWidget::tr("Name:"), QLineEdit::Normal,
		                                     vis->getName().c_str());
		if(!text.isNull()) {
			std::string name = text.toStdString();
			vis->setName(name);
		}

		return true;
	}

	std::string getName(const PropertyNode* property)
	{
		const PropertyType* p = property->toTyped<Visualization*>();
		assert(p!=NULL);
		Visualization* vis = p->get();
		return vis->getName();
	}
};

///////////////////////////////////////////////////////////////////////////////

typedef TypedPropertyNode<StatusManager*> StatusPropertyType;

class VisualizationStatusPropertyDelegate : public PropertyEditor::Delegate
{
	MIRA_OBJECT(VisualizationStatusPropertyDelegate)

public:

	virtual SupportedTypes supportedTypes() const
	{
		return makeSupportedType<StatusManager*>(SupportedType::TEXT |
		                                         SupportedType::COLOR);
	}

	virtual std::string getText(const PropertyNode* property)
	{
		const StatusPropertyType* p = property->toTyped<StatusManager*>();
		assert(p!=NULL);
		Status::StatusMode m = p->get()->getStatus();
		if (m==Status::OK)
			return "OK";
		StatusManager::StatusMap status = p->get()->getStatusMap();
		if (status.size() > 0)
		{
			if (status.rbegin()->second.trText.empty())
				return status.rbegin()->second.message;
			else
				return status.rbegin()->second.trText+": "+status.rbegin()->second.message;
		}
		return "Unknown";
	}

	virtual QColor getBackgroundColor(const PropertyNode* property)
	{
		const StatusPropertyType* p = property->toTyped<StatusManager*>();
		assert(p!=NULL);
		Status::StatusMode status = p->get()->getStatus();
		if(status == Status::WARNING)
			return PropertyEditor::lightYellow;
		else if (status == Status::ERROR)
			return PropertyEditor::lightRed;
		else if (status == Status::BOOTUP)
			return PropertyEditor::lightBlue;
		else if (status == Status::RECOVER)
			return PropertyEditor::lightPurple;
		else return QColor(); // if okay, then use default color
	}
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::VisualizationPropertyDelegate,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::VisualizationStatusPropertyDelegate,
                    mira::PropertyEditor::Delegate);
