/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonTool.C
 *    Implementation of PolygonTool.
 *
 * @author Tim Langner
 * @date   2012/10/05
 */

#include <QMouseEvent>

#include <visualization/3d/PolygonTool.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void PolygonTool::onMouseReleased(QMouseEvent* e)
{
	if (e->button() == Qt::LeftButton)
	{
		auto i = getIntersection(e->posF().x(), e->posF().y(), mSelectedObject);
		if (!i.first.empty())
		{
			if ((mPolygon.size() >= 3) && isNearFirstPoint(e, i.second)) {
				finishPolygon();
			} else {
				mSelectedObject = i.first;
				mPolygon.push_back(Point2f(i.second.x, i.second.y));
				mPolygonObject->setPolygon(mPolygon);
			}
		}
	}
	else if (e->button() == Qt::RightButton)
		reset();
}

void PolygonTool::onMouseMoved(QMouseEvent* e)
{
	if (mSelectedObject.empty()) {
		updateToolTip(e, "Set 3 or more points with left-click to specify polygon."
		                 "\nRight-click to restart (clear points).");
		return;
	}

	auto i = getIntersection(e->posF().x(), e->posF().y(), mSelectedObject);
	if (!i.first.empty())
	{
		if ((mPolygon.size() > 2) && isNearFirstPoint(e, i.second)) {
			mPolygonObject->setPolygon(mPolygon);
			updateToolTip(e, "FINISH (left-click here)");
			return;
		}

		Polygon2f poly = mPolygon;
		poly.push_back(Point2f(i.second.x, i.second.y));
		mPolygonObject->setPolygon(poly);
	}

	if (mPolygon.size() < 3) {
		updateToolTip(e, "Set 3 or more points with left-click to specify polygon."
		                 "\nRight-click to restart (clear points).");
	} else
		updateToolTip(e, "Add more points or click first point again to finish input."
		                 "\nRight-click to restart (clear points).");
}

/// called when the user has activated the tool.
void PolygonTool::activate()
{
	mPolygonObject->setVisible(true);
}

/// called when the user has deactivated the tool by selecting another one.
void PolygonTool::deactivate()
{
	mPolygonObject->setVisible(false);
}

std::pair<std::string, Ogre::Vector3> PolygonTool::getIntersection(float x, float y,
                                                                   const std::string& mask)
{
	return mask.empty() ? SceneQueryTool::getIntersection(x, y, "ImageObject")
                        : SceneQueryTool::getIntersection(x, y, mask, true);
}

void PolygonTool::setupScene(IVisualization3DSite* site)
{
	SceneQueryTool::setupScene(site);
	mPolygonObject.reset(new PolygonObject<float>(site->getSceneManager()));
	mPolygonObject->setVisible(false);
	mPolygonObject->setOutlineColor(mOutlineColor);
	mSelectedObject = "";
}

void PolygonTool::setPolygon(const Polygon2f& polygon)
{
	mPolygon = polygon;
	if (mPolygonObject)
		mPolygonObject->setPolygon(mPolygon);
}

void PolygonTool::setOutlineColor(const Ogre::ColourValue& color)
{
	mOutlineColor = color;
	if (mPolygonObject)
		mPolygonObject->setOutlineColor(color);
}

void PolygonTool::reset()
{
	mSelectedObject = "";
	mPolygon.clear();
	mPolygonObject->setPolygon(mPolygon);
}

void PolygonTool::finishPolygon()
{
	// actually it shouldn't be possible anymore to finish with <3 points
	if (mPolygon.size() < 3)
		MIRA_THROW(XInvalidConfig, "Polygon must have at least 3 points");

	boost::geometry::correct(mPolygon);
	try {
		onNewPolygon(mPolygon);
	}
	catch(...) {}
	reset();
}

bool PolygonTool::isNearFirstPoint(QMouseEvent* e, const Ogre::Vector3& p)
{
	assert(!mPolygon.empty());

	// since we cannot project back the polygon point to screen, we use a workaround:
	// project into the scene at different screen positions and compare the hit positions
	// on the object (map) to extrapolate the pixel distance

	Ogre::Vector3 p2;
	std::pair<std::string, Ogre::Vector3> i;
	i = getIntersection(e->posF().x()+1.0, e->posF().y()+1.0, mSelectedObject);
	if (!i.first.empty()) {
		p2 = i.second;
	} else {
		i = getIntersection(e->posF().x()-1.0, e->posF().y()-1.0, mSelectedObject);
		if (!i.first.empty()) {
			p2 = i.second;
		} else
			return false;
	}

	float dNorm = (p2 - p).length() / 1.4; // distance on map for 1 pixel offset

	float dFirst = (p - Ogre::Vector3(mPolygon.front().x(), mPolygon.front().y(), p.z)).length();

	// compare and check if the first polygon point is in a 10 pixel proximity
	if (dFirst / dNorm < 10.f)
		return true;

	return false;
}

///////////////////////////////////////////////////////////////////////////////

}
