/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationCompatibilityTapeTest.C
 *    Test cases for reading old tapes (v0 binary and typemeta format).
 *    Actually a test for serialization functionality, but uses Tape as
 *    vehicle for v0 serialized data (which is also of course the main
 *    use case for this compatibility feature).
 *
 * @author Christof Schröter
 * @date   2019/03/23
 */

#include <boost/test/unit_test.hpp>

#include <transform/Pose.h>
#include <utils/PathFinder.h>
#include <utils/Stamped.h>

#include <serialization/BinaryJSONConverter.h>

#include <fw/Framework.h>
#include <fw/Tape.h>

using namespace mira;

template <typename T>
void readMessageBinary(Tape& tape, const Tape::MessageIndex& index, Stamped<T>& oValue)
{
	std::cout << "readMessageBinary" << std::endl;
	Buffer<uint8> buffer;
	Duration time;
	tape.readMessage(index, oValue.frameID, oValue.sequenceID, buffer, time);
	oValue.timestamp = tape.getStartTime() + time;

	BinaryBufferDeserializer d(&buffer);
	d.deserialize(oValue.internalValueRep(), false);
}

void readMessageJSON(Tape& tape, const Tape::MessageIndex& index,
                     TypeMetaPtr meta, const MetaTypeDatabase& metaDB,
                     json::Value& oValue)
{
	std::cout << "readMessageJSON" << std::endl;
	Buffer<uint8> buffer;
	std::string frameID;
	uint32 sequenceID;
	Duration time;
	tape.readMessage(index, frameID, sequenceID, buffer, time);

	BinaryJSONConverter::binaryToJSON(buffer, false, *meta, metaDB, oValue);
}

typedef std::vector<std::list<Pose2>> ChannelType;

void checkObject(const ChannelType& v, int n)
{
	BOOST_REQUIRE_EQUAL(v.size(), n);
	for (int l = 0; l < n; ++l) {
		BOOST_REQUIRE_EQUAL(v[l].size(), l);
		std::list<Pose2>::const_iterator it = v[l].cbegin();
		for (int e = 0; e < l; ++e) {
			BOOST_CHECK_EQUAL(it->x(), (float)l);
			BOOST_CHECK_EQUAL(it->y(), (float)e);
			BOOST_CHECK_EQUAL(it->phi(), deg2rad((float)(e+l)));
			++it;
		}
	}
}

void testTape(const Path& tapefile)
{
	Tape t;
	t.open(tapefile, Tape::READ);

	const Tape::ChannelMap& channels = t.getChannels();
	BOOST_REQUIRE_EQUAL(channels.size(), 1);
//	BOOST_CHECK_EQUAL(channels.begin()->first, "/Vector");
	
	const Tape::ChannelInfo& info = channels.begin()->second;

	BOOST_CHECK_EQUAL(info.type, typeName<ChannelType>());

	if (info.meta->version() == 0) {
		BOOST_CHECK_EQUAL(info.meta->toString(), "[][]mira::RigidTransform<float,2> @v0");
		BOOST_CHECK_EQUAL(info.meta->getIdentifier(), "mira::RigidTransform<float,2> @v0");
	} else {
		MetaTypeDatabase db;
		MetaSerializer ms(db);
		ChannelType t;
		TypeMetaPtr meta = ms.addMeta(t);

		BOOST_CHECK_EQUAL(info.meta->toString(), meta->toString());
		BOOST_CHECK_EQUAL(info.meta->getIdentifier(), meta->getIdentifier());
	}

	BOOST_REQUIRE_EQUAL(info.messages.size(), 10);

	Stamped<ChannelType> sv;
	json::Value jv;

	for (int n = 0; n < 10; ++n) {
		const Tape::MessageIndex& index = info.messages[n];
		readMessageBinary(t, index, sv);
		checkObject(sv, n);

		readMessageJSON(t, index, info.meta, info.metaDB, jv);

		JSONDeserializer d(jv);
		ChannelType v;
		d.deserialize(v);
		checkObject(v, n);
	}
}

BOOST_AUTO_TEST_CASE(TapeReadTest)
{
	testTape(findProjectFile("framework/tests/fw/etc/v0.tape"));
}

///////////////////////////////////////////////////////////////////////////////

class Subscriber : public mira::Authority
{
public:
	Subscriber(const std::string& ns) : Authority(ns, "Subscriber", Authority::ANONYMOUS), dataCount(0) {}
	bool finished() { return dataCount >= 10; }
	void reset() { dataCount = 0; }

protected:
	int dataCount;
};

class TypedSubscriber : public Subscriber
{
public:
	TypedSubscriber(const std::string& ns = "/typed") : Subscriber(ns) {
		subscribe<ChannelType>("Vector", &TypedSubscriber::onData);
	}

private:
	void onData(ChannelRead<ChannelType> data) {
		if (dataCount == 0) {
			TypeMetaPtr channelMeta = MIRA_FW.getChannelManager().getTypeMeta(getNamespace()+"/Vector");

			MetaTypeDatabase db;
			MetaSerializer ms(db);
			ChannelType t;
			TypeMetaPtr meta = ms.addMeta(t);

			BOOST_CHECK_EQUAL(channelMeta->toString(), meta->toString());
			BOOST_CHECK_EQUAL(channelMeta->getIdentifier(), meta->getIdentifier());
		}
		checkObject(data, dataCount++);
	}
};

class UntypedSubscriber : public Subscriber
{
public:
	UntypedSubscriber() : Subscriber("/untyped") {
		subscribe<void>("Vector", &UntypedSubscriber::onData);
	}

private:
	void onData(ChannelRead<void> data)	{

		if (dataCount == 0) {
			TypeMetaPtr channelMeta = MIRA_FW.getChannelManager().getTypeMeta(getNamespace()+"/Vector");

			BOOST_CHECK_EQUAL(channelMeta->toString(), "[][]mira::RigidTransform<float,2> @v0");
			BOOST_CHECK_EQUAL(channelMeta->getIdentifier(), "mira::RigidTransform<float,2> @v0");
		}

		json::Value j;
		data.readJSON(j);

		ChannelType o;
		JSONDeserializer d(j);
		d.deserialize(o);

		checkObject(o, dataCount++);
	}
};

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(TapePlayTest)
{
	const char* argv[] = {"TapePlayTest",
	                      "--no-autopromote",
	                      "--config-string",
	                      "<root>"
	                      "  <namespace name=\"typed\">"
	                      "    <unit id=\"TapePlayer\" class=\"mira::TapePlayerUnit\"/>"
	                      "    <unit id=\"TapeRecorder\" class=\"mira::TapeRecorderUnit\"/>"
	                      "  </namespace>"
	                      "  <namespace name=\"untyped\">"
	                      "    <unit id=\"TapePlayer\" class=\"mira::TapePlayerUnit\"/>"
	                      "    <unit id=\"TapeRecorder\" class=\"mira::TapeRecorderUnit\"/>"
	                      "  </namespace>"
	                      "  <namespace name=\"remote\">"
	                      "    <process>"
	                      "      <unit id=\"TapePlayer\" class=\"mira::TapePlayerUnit\"/>"
	                      "    </process>"
	                      "    <unit id=\"TapeRecorder\" class=\"mira::TapeRecorderUnit\"/>"
	                      "  </namespace>"
	                      "</root>"};
	Framework fw(4,(char**)argv);
	fw.load();

	fw.start();

	std::string tapefile = findProjectFile("framework/tests/fw/etc/v0.tape").string();

	{
		UntypedSubscriber untyped;

		// wait for required services
		if (!untyped.waitForService("TapePlayer", Duration::seconds(10)) ||
		    !untyped.waitForService("TapeRecorder", Duration::seconds(10)))
			BOOST_FAIL("Missing TapePlayer or TapeRecorder");

		// configure services
		untyped.callService<void>("TapePlayer#builtin", "setProperty",
		                          std::string("NamespacePrefix"), std::string("untyped")).get();
		untyped.callService<void>("TapePlayer#builtin", "setProperty",
		                      std::string("TimeScale"), std::string("10.0")).get();

		// play v0 tape to untyped channel, read+check, record
		untyped.callService<void>("TapePlayer", "load", tapefile).get();
		untyped.callService<void>("TapeRecorder", "addChannel", std::string("/untyped/Vector")).get();

		untyped.callService<void>("TapeRecorder", "record", std::string("v0_playeduntyped_recorded.tape")).get();
		untyped.callService<void>("TapePlayer", "play").get();

		int c = 0;
		while (!untyped.finished()) {
			if (c++ > 15)
				BOOST_FAIL("Waiting too long, tape playback seems not to work");
			MIRA_SLEEP(1000);
		}

		untyped.callService<void>("TapePlayer", "stop").get();
		untyped.callService<void>("TapeRecorder", "stop").get();

		// untyped subscriber deleted here
	}

	{
		// subscribe typed to previously untyped channel -> promoted to typed!
		TypedSubscriber promoted("/untyped");

		// Channel's type meta is updated in promotion (or, if empty, on first write)
		// (as promotion is creating a new ChannelBuffer internally)

		// play v0 tape to the previously untyped, now typed channel, read+check
		promoted.callService<void>("TapePlayer", "load", tapefile).get();
		promoted.callService<void>("TapeRecorder", "addChannel", std::string("/untyped/Vector")).get();

		promoted.callService<void>("TapePlayer", "step").get(); // to make sure we don't record previous channel updates
		promoted.callService<void>("TapeRecorder", "record", std::string("v0_playedpromoted_recorded.tape")).get();

		promoted.callService<void>("TapePlayer", "play").get();

		int c = 0;
		while (!promoted.finished()) {
			if (c++ > 15) {
				BOOST_FAIL("Waiting too long, tape playback seems not to work");
			}
			MIRA_SLEEP(1000);
		}

		promoted.callService<void>("TapePlayer", "stop").get();
		promoted.callService<void>("TapeRecorder", "stop").get();
	}

	{
		TypedSubscriber typed;

		// wait for required services
		if (!typed.waitForService("TapePlayer", Duration::seconds(10)) ||
		    !typed.waitForService("TapeRecorder", Duration::seconds(10)))
			BOOST_FAIL("Missing TapePlayer or TapeRecorder");

		// configure services
		typed.callService<void>("TapePlayer#builtin", "setProperty",
		                        std::string("NamespacePrefix"), std::string("typed")).get();
		typed.callService<void>("TapePlayer#builtin", "setProperty",
		                        std::string("TimeScale"), std::string("10.0")).get();


		// play v0 tape to typed channel, read+check, record
		typed.callService<void>("TapePlayer", "load", tapefile).get();
		typed.callService<void>("TapeRecorder", "addChannel", std::string("/typed/Vector")).get();

		typed.callService<void>("TapeRecorder", "record", std::string("v0_playedtyped_recorded.tape")).get();
		typed.callService<void>("TapePlayer", "play").get();

		int c = 0;
		while (!typed.finished()) {
			if (c++ > 15) {
				BOOST_FAIL("Waiting too long, tape playback seems not to work");
			}
			MIRA_SLEEP(1000);
		}

		typed.callService<void>("TapePlayer", "stop").get();
		typed.callService<void>("TapeRecorder", "stop").get();
	}

	{
		TypedSubscriber remote("/remote");

		// wait for required services
		if (!remote.waitForService("TapePlayer", Duration::seconds(10)) ||
		    !remote.waitForService("TapeRecorder", Duration::seconds(10)))
			BOOST_FAIL("Missing TapePlayer or TapeRecorder");

		// configure services
		remote.callService<void>("TapePlayer#builtin", "setProperty",
		                         std::string("NamespacePrefix"), std::string("remote")).get();
		remote.callService<void>("TapePlayer#builtin", "setProperty",
		                         std::string("TimeScale"), std::string("10.0")).get();

		// play v0 tape to untyped channel in remote framework, locally read+check, record
		remote.callService<void>("TapePlayer", "load", tapefile).get();
		remote.callService<void>("TapeRecorder", "addChannel", std::string("/remote/Vector")).get();

		remote.callService<void>("TapeRecorder", "record", std::string("v0_playeduntyped_recordedremote.tape")).get();
		remote.callService<void>("TapePlayer", "play").get();

		int c = 0;
		while (!remote.finished()) {
			if (c++ > 15) {
				BOOST_FAIL("Waiting too long, tape playback seems not to work");
			}
			MIRA_SLEEP(1000);
		}

		remote.callService<void>("TapePlayer", "stop").get();
		remote.callService<void>("TapeRecorder", "stop").get();
	}
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(RecordedUntypedTest)
{
	testTape("v0_playeduntyped_recorded.tape"); // created in TapePlayTest
}

BOOST_AUTO_TEST_CASE(RecordedPromotedTest)
{
	testTape("v0_playedpromoted_recorded.tape");   // created in TapePlayTest
}

BOOST_AUTO_TEST_CASE(RecordedTypedTest)
{
	testTape("v0_playedtyped_recorded.tape");   // created in TapePlayTest
}

BOOST_AUTO_TEST_CASE(RecordedTypedRemoteTest)
{
	testTape("v0_playeduntyped_recordedremote.tape");   // created in TapePlayTest
}

struct ForceLegacyTestClass
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("int16", i16, "");
		r.member("int8",  i8, "");
		r.member("int",   i, "");
	}

	bool operator==(const ForceLegacyTestClass& other) const {
		return (i16 == other.i16) && (i8 == other.i8) && (i == other.i);
	}

	uint16 i16;
	uint8  i8;
	int    i;
};

// serialized data in v0 format that looks like v2
BOOST_AUTO_TEST_CASE(DeserializeV0MimicV2Test)
{
	{
		ForceLegacyTestClass c;
		c.i16 = 1;
		c.i8 = BinaryBufferSerializer::getSerializerFormatVersion();
		c.i = 12345;

		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializerLegacy s(&buffer);
		s.serialize(c, false);

		
		BinaryBufferDeserializer d(&buffer);
		ForceLegacyTestClass c2;
		d.deserialize(c2, false);

		BOOST_CHECK(c == c2);
	}
	{
		// write v0 binary format, but mimic v2!
		ForceLegacyTestClass c;
		c.i16 = BINARY_VERSION_MARKER;
		c.i8 = BinaryBufferSerializer::getSerializerFormatVersion();
		c.i = 12345;

		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializerLegacy s(&buffer);
		s.serialize(c, false);

		
		BinaryBufferDeserializer d(&buffer);
		ForceLegacyTestClass c2;
		BOOST_CHECK_THROW(d.deserialize(c2, false), XIO);
	}
}

BOOST_AUTO_TEST_CASE(SerializeReadV0Test)
{
	{
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer s(&buffer);

		int i = 12345;
		s.serialize(i, false);

		i = 54321;
		s.serialize(i, false);

		BinaryBufferDeserializer d(&buffer);

		d.deserialize(i, false);
		BOOST_CHECK_EQUAL(i, 12345);

		d.deserialize(i, false);
		BOOST_CHECK_EQUAL(i, 54321);
	}
	{
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer s(&buffer);

		int i = 12345;
		s.serialize(i, false);

		i = 54321;
		s.serialize(i, false);

		// try to read the serialized value with legacy deserializer
		BinaryBufferDeserializerLegacy d(&buffer);

		// we don't read back the original serialized values, but the first and second 4 bytes
		// of this sequence:
		// BINARY_VERSION_MARKER (2 bytes), version 2(1 byte), 12345(4 bytes),
		// BINARY_VERSION_MARKER (2 bytes), version 2(1 byte), 54321(4 bytes)
		d.deserialize(i, false);
		BOOST_CHECK_EQUAL(i,
		                 (((12345%256 << 8) + BinaryBufferSerializer::getSerializerFormatVersion()) << 16)
		                 + BINARY_VERSION_MARKER);

		d.deserialize(i, false);
		BOOST_CHECK_EQUAL(i, (BINARY_VERSION_MARKER % 256 << 24) + 12345/256);
	}
}
