/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCTest.C
 *    Test cases for RPC base classes
 *
 * @author Erik Einhorn
 * @date   2012/01/23
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

using namespace mira;
using namespace std;

const char* argv[] = {"FrameworkBasicTest", "-d3", "--no-colors",
                      "--config-string",
                      "<root><process args=\"--fw-name=remotefw\"/></root>"};
Framework fw(5,(char**)argv);

///////////////////////////////////////////////////////////////////////////////

class MyService
{
public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.method("method", &MyService::method, this, "A test method");
	}

	int test1;
	float test2;

	void method(int param1, float param2)
	{
		MIRA_LOG(NOTICE) << "MyService::method() called " << ", " << param1 << ", " << param2;
		test1 = param1;
		test2 = param2;
	}
};

///////////////////////////////////////////////////////////////////////////////

class MyErrorService
{
public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.method("method", &MyErrorService::method, this, "A test method");
	}

	int method()
	{
		MIRA_LOG(NOTICE) << "MyErrorService::method() called";
		MIRA_THROW(XInvalidParameter, "Exception!");
		return 0;
	}
};

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(RPCTests)
{
	Authority authority1("/", std::string("Authority1"));
	Authority authority2("/", std::string("Authority2"));

	MyService service;
	authority1.publishService(service);

	MyErrorService errorService;
	authority2.publishService(errorService);

	RPCFuture<int> fi = authority1.callService<int>("/Authority2", "method");
	fi.wait();
	BOOST_CHECK_EQUAL(fi.hasException(), true);
	BOOST_CHECK_THROW(fi.get(), Exception);

	//////////// JSON tests //////////

	// Test call with service, method and string arguments
	service.test1 = 0;
	service.test2 = 0.0f;
	auto fj = authority2.callServiceJSON("/Authority1", "method", std::string("345, 3.141"));
	fj.wait();
	BOOST_CHECK_EQUAL(service.test1, 345);
	BOOST_CHECK_CLOSE(service.test2, 3.141f, 0.0001f);

	// Test call with service, method and json array argument
	service.test1 = 0;
	service.test2 = 0.0f;
	json::Array va;
	va.push_back(json::Value(345));
	va.push_back(json::Value(3.141));
	fj = authority2.callServiceJSON("/Authority1", "method", va);
	fj.wait();
	BOOST_CHECK_EQUAL(service.test1, 345);
	BOOST_CHECK_CLOSE(service.test2, 3.141f, 0.0001f);

	// Test call with own generated JSON RPC 2.0 conform request in json format
	service.test1 = 0;
	service.test2 = 0.0f;
	json::Object jsonCall;
	jsonCall["jsonrpc"] = json::Value("2.0");
	jsonCall["id"] = json::Value(255);
	jsonCall["method"] = json::Value("/Authority1.method");
	jsonCall["params"] = va;
	fj = authority2.callServiceJSON(jsonCall);
	fj.wait();
	BOOST_CHECK_EQUAL(service.test1, 345);
	BOOST_CHECK_CLOSE(service.test2, 3.141f, 0.0001f);

	// Test call with own generated JSON RPC 2.0 conform request in string format
	service.test1 = 0;
	service.test2 = 0.0f;
	std::string strCall = "{ \"jsonrpc\": \"2.0\","
	                      "  \"id\": 0,"
	                      "  \"method\": \"/Authority1.method\","
	                      "  \"params\": [345, 3.141] }";
	fj = authority2.callServiceJSON(strCall);
	fj.wait();
	BOOST_CHECK_EQUAL(service.test1, 345);
	BOOST_CHECK_CLOSE(service.test2, 3.141f, 0.0001f);

	fj = authority1.callServiceJSON("/Authority2", "method",json::Value());
	fj.wait();
	BOOST_CHECK_EQUAL(fj.hasException(), true);
	try {
		fj.get();
		BOOST_CHECK_EQUAL("", "Should have thrown an XRPC before reaching here!");
	}
	catch(XRPC& ex) {
		BOOST_CHECK_EQUAL(std::string(ex.what(), 10), "Exception!");
		std::ostringstream ss;
		ss << ex.callStack()[0];
		std::string trace0 = ss.str();
		BOOST_CHECK(trace0.find("MyErrorService") != std::string::npos);

		// Hint: This test will fail on old CentOS-6 with gcc 4.4.7.
		// For some unknown reason, on this system the final call of
		// "method" is missing in the backtrace. But we keep this
		// test to check this behaviour on other systems.
		BOOST_CHECK(trace0.find("method") != std::string::npos);

		BOOST_CHECK(trace0.find("RPCTest") != std::string::npos);
	}
}

BOOST_AUTO_TEST_CASE(RemoteRPCTests)
{
	fw.load();
	fw.start();

	RPCManager& man = fw.getRPCManager();

	BOOST_REQUIRE(man.waitForService("/remotefw", Duration::seconds(3)));

	RPCFuture<Duration> fb = man.call<Duration>("/remotefw", "getUptime");
	BOOST_REQUIRE(fb.timedWait(Duration::seconds(3)));
	std::cout << "uptime of remote fw (binary rpc): " << fb.get() << std::endl;

	RPCFuture<JSONRPCResponse> fj = man.callJSON("/remotefw", "getUptime");
	BOOST_REQUIRE(fj.timedWait(Duration::seconds(3)));
	std::cout << "uptime of remote fw (json rpc): " << fj.get() << std::endl;
}

/*
class ChattyObject
{
public:
	ChattyObject() : name("<null>") {
		refcount(1);
	}

	ChattyObject(const std::string& iName) : name(iName) {
		std::cout << "ChattyObject(): " << name << std::endl;
		refcount(1);
	}
	ChattyObject(const ChattyObject& other) : name(other.name) {
		std::cout << "ChattyObject() copy: " << name << std::endl;
		refcount(1);
	}
	ChattyObject& operator=(const ChattyObject& other) {
		name = other.name;
		std::cout << "ChattyObject() assignment: " << name << std::endl;
		return *this;
	}

	~ChattyObject() {
		refcount(-1);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("name", name, "");
	}

	std::string name;

	static int refcount(int mode=0) {
		static boost::mutex mutex;
		boost::mutex::scoped_lock lock(mutex);

		static int sCount = 0;
		switch(mode) {
			case 0:  return sCount;
			case 1:  return ++sCount;
			case -1: return --sCount;
			default: return sCount;
		}
	}
};

class MyClass
{
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("ITest");
		r.method("return0", return0, "");
		//r.method("return1", return1, "");
		r.method("return2", return2, "");
		r.method("return3", return3, "");

		r.method("param0", param0, "");
		r.method("param1", param1, "");
		r.method("param2", param2, "");
		r.method("param3", param3, "");
	}

	static ChattyObject return0()
	{
		std::cout << "return0" << std::endl;
		return ChattyObject("return0 obj");
	}

	static ChattyObject& return1()
	{
		static ChattyObject obj("return1");
		std::cout << "return1" << std::endl;
		return obj;
	}

	static ChattyObject* return2()
	{
		std::cout << "return2" << std::endl;

		static ChattyObject obj("return2");
		return &obj;
	}

	static boost::shared_ptr<ChattyObject> return3()
	{
		std::cout << "return3" << std::endl;
		return boost::shared_ptr<ChattyObject>(new ChattyObject("return3 obj"));
	}

	static void param0(ChattyObject val)
	{
		std::cout << "param0:" << std::endl;
		std::cout << val.name << std::endl;
	}

	static void param1(ChattyObject& val)
	{
		std::cout << "param1:" << std::endl;
		std::cout << val.name << std::endl;
	}

	static void param2(ChattyObject* val)
	{
		std::cout << "param2:" << std::endl;
		std::cout << val->name << std::endl;
	}

	static void param3(boost::shared_ptr<ChattyObject> val)
	{
		std::cout << "param3:" << std::endl;
		std::cout << val->name << std::endl;
	}

};




int main(int argc, char** argv)
{
	Framework fw(argc,argv);

	Authority service("/","Service");

	MyClass myServiceObj;
	service.publishService(myServiceObj);


	Authority caller("/", "Caller");

//	{
//	std::cout << "Invoking return0" << std::endl;
//	RPCFuture<ChattyObject> f0 = caller.callService<ChattyObject>("Service", "return0");
//	std::cout << "Get: " << f0.get().name << std::endl;
//	}

	{
	// returning references results in compiler error
	// but we keep this code for documentation
	//std::cout << "Invoking return1" << std::endl;
	//RPCFuture<ChattyObject&> f1 = caller.callService<ChattyObject&>("Service", "return1");
	//std::cout << "Get: " << f1.get().name << std::endl;
	}

	{
	std::cout << "Invoking return2" << std::endl;
	RPCFuture<ChattyObject*> f2 = caller.callService<ChattyObject*>("Service", "return2");
	std::cout << "Get: " << f2.get()->name << std::endl;
	delete f2.get();
	}

	// internally, one instance of ChattyObject is kept as static object in
	// return2(), hence the reference count: +1
	std::cout << "Ref count: " << ChattyObject::refcount() << std::endl;


	{
	std::cout << "Invoking return3" << std::endl;
	RPCFuture<boost::shared_ptr<ChattyObject>> f3 =
			caller.callService<boost::shared_ptr<ChattyObject>>("Service", "return3");
	std::cout << "Get: " << f3.get()->name << std::endl;
	}

	// internally, the shared pointer management will keep a pointer to a
	// chatty object, hence the reference count: +1 -> (total 2)
	std::cout << "Ref count: " << ChattyObject::refcount() << std::endl;

	std::cout << "Finished1" << std::endl;

	{
	std::cout << "Invoking param0" << std::endl;
	ChattyObject param0("param0");
	caller.callService<void>("Service", "param0", param0).get();
	}

	{
	std::cout << "Invoking param1" << std::endl;
	ChattyObject param1("param1");
	caller.callService<void>("Service", "param1", param1).get();
	}

	{
	std::cout << "Invoking param2" << std::endl;
	ChattyObject param2("param2");
	caller.callService<void>("Service", "param2", &param2).get();
	}

	{
	std::cout << "Invoking param3" << std::endl;
	caller.callService<void>("Service", "param3",
			boost::shared_ptr<ChattyObject>(new ChattyObject("param3"))).get();
	}

	std::cout << "Ref count: " << ChattyObject::refcount() << std::endl;
	std::cout << "Finished2" << std::endl;

	return 0;//fw.run();
}*/

