/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolymorphicChannelsTest.C
 *    Test for polymorphic channels
 *
 * @author Christof Schröter
 * @date   2020/06/12
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

using namespace mira;

class TestA : public Object
{
	MIRA_OBJECT(TestA)
public:
	TestA(const std::string& ia = "")
		: a(ia) {}
	virtual ~TestA() {}

	template<typename Reflector> void reflect(Reflector& r) {}

	virtual std::string id() const { return "A"; }
	virtual std::string content() const { return a; }

protected:
	std::string a;
};

class TestB : public TestA
{
	MIRA_OBJECT(TestB)
public:
	TestB(const std::string& ia = "", const std::string& ib = "")
		: TestA(ia), b(ib) {}
	virtual ~TestB() {}

	virtual std::string id() const { return "B"; }
	virtual std::string content() const { return TestA::content() + b; }

protected:
	std::string b;
};

class TestC : public TestB
{
	MIRA_OBJECT(TestC)
public:
	TestC(const std::string& ia = "", const std::string& ib = "", const std::string& ic = "")
		: TestB(ia, ib), c(ic) {}
	virtual ~TestC() {}

	virtual std::string id() const { return "C"; }
	virtual std::string content() const { return TestB::content() + c; }

protected:
	std::string c;
};

MIRA_CLASS_SERIALIZATION(TestA, mira::Object);
MIRA_CLASS_SERIALIZATION(TestB, TestA);
MIRA_CLASS_SERIALIZATION(TestC, TestB);

// Test rules for polymorphic channels

BOOST_AUTO_TEST_CASE(PolymorphicChannelsTest1)
{
	const char* argv[] = {"PolymorphicChannelsTest", "-p1234"};
	Framework fw(2,(char**)argv);

	// now start processing
	fw.start();

	Authority auth("/", "TestAuthority");

	{
		// The first publish() fixes the type,
		// to the most derived of all publish/subscribe types at that time.

		Channel<TestA*> channelA = auth.subscribe<TestA*>("Test1"); // promote to A*
		BOOST_CHECK_THROW(channelA.getTypename(), XLogical);

		Channel<TestB*> channelB = auth.subscribe<TestB*>("Test1"); // promote to B*
		BOOST_CHECK_THROW(channelB.getTypename(), XLogical);

		Channel<TestA*> channelA2 = auth.publish<TestA*>("Test1");  // fix -> B*
		
		BOOST_CHECK_EQUAL(channelA.getTypename(), "TestB*");
		BOOST_CHECK_EQUAL(channelB.getTypename(), "TestB*");
		BOOST_CHECK_EQUAL(channelA2.getTypename(), "TestB*");

		// Once fixed, the channel cannot be published or subscribed
		// with a more derived type.

		BOOST_CHECK_THROW(auth.subscribe<TestC*>("Test1"), XBadCast);
		BOOST_CHECK_THROW(auth.publish<TestC*>("Test1"), XBadCast);

		// Polymorphic channels act like dynamic_cast:
		// when publishing a Base object, a Derived subscriber will receive a null pointer

		channelA2.post(new TestA("1"));
		{
			ChannelRead<TestA*> readA = channelA.waitForData(Duration::milliseconds(100));
			BOOST_CHECK(readA.isValid());
			const TestA* a = readA->value();
			BOOST_CHECK((a != NULL) && (a->id() == "A"));
			BOOST_CHECK((a != NULL) && (a->content() == "1"));

			ChannelRead<TestB*> readB = channelB.waitForData(Duration::milliseconds(100));
			BOOST_CHECK(readB.isValid());
			BOOST_CHECK_EQUAL(readA->timestamp, readB->timestamp);

			const TestB* b = readB->value();
			BOOST_CHECK((b == NULL));
		}

		channelA2.post((TestA*)(new TestB("1", "2")));
		{		
			ChannelRead<TestA*> readA = channelA.read();
			BOOST_CHECK(readA.isValid());
			const TestA* a = readA->value();
			BOOST_CHECK((a != NULL) && (a->id() == "B"));
			BOOST_CHECK((a != NULL) && (a->content() == "12"));

			ChannelRead<TestB*> readB = channelB.read();
			BOOST_CHECK(readB.isValid());
			BOOST_CHECK_EQUAL(readA->timestamp, readB->timestamp);

			const TestB* b = readB->value();
			BOOST_CHECK((b != NULL) && (b->id() == "B"));
			BOOST_CHECK((b != NULL) && (b->content() == "12"));
		}

		// Quite interesting that this works. What relevance does the fixed type have then?
		channelA2.post((TestA*)(new TestC("1", "2", "3")));
		{		
			ChannelRead<TestA*> readA = channelA.read();
			BOOST_CHECK(readA.isValid());
			const TestA* a = readA->value();
			BOOST_CHECK((a != NULL) && (a->id() == "C"));
			BOOST_CHECK((a != NULL) && (a->content() == "123"));

			ChannelRead<TestB*> readB = channelB.read();
			BOOST_CHECK(readB.isValid());
			BOOST_CHECK_EQUAL(readA->timestamp, readB->timestamp);

			const TestB* b = readB->value();
			BOOST_CHECK((b != NULL) && (b->id() == "C"));
			BOOST_CHECK((b != NULL) && (b->content() == "123"));
		}
	}
	
	{
		// An authority can publish a channel multiple times,
		// but only with the same type.

		Channel<TestA*> channelA = auth.subscribe<TestA*>("Test2");  // promote to A*
		Channel<TestB*> channelB = auth.subscribe<TestB*>("Test2");  // promote to B*
		Channel<TestC*> channelC = auth.publish<TestC*>("Test2");    // promote to C* , fix -> C*
		Channel<TestC*> channelC2 = auth.publish<TestC*>("Test2");
		BOOST_CHECK_THROW(auth.publish<TestB*>("Test2"), XLogical);

		BOOST_CHECK_EQUAL(channelA.getTypename(), "TestC*");
		BOOST_CHECK_EQUAL(channelB.getTypename(), "TestC*");
		BOOST_CHECK_EQUAL(channelC.getTypename(), "TestC*");
		BOOST_CHECK_EQUAL(channelC2.getTypename(), "TestC*");

		// A different authority can publish a channel with the fixed type
		// or a base class of the fixed type.
		
		// A polymorphic channel cannot be subscribed/published with a monomorphic
		// (non-pointer) type.

		Authority auth2("/", "TestAuthority2");
		BOOST_CHECK_THROW(auth2.publish<TestB>("Test2"), XBadCast);
		Channel<TestB*> channelB2 = auth2.publish<TestB*>("Test2");
		BOOST_CHECK_EQUAL(channelB2.getTypename(), "TestC*");

		// when publishing a Derived object, a Base subscriber will receive a Base pointer to the Derived object

		channelC.post(new TestC("1", "2", "3"));
		{
			ChannelRead<TestA*> readA = channelA.waitForData(Duration::milliseconds(100));
			BOOST_CHECK(readA.isValid());
			const TestA* a = readA->value();
			BOOST_CHECK((a != NULL) && (a->id() == "C"));
			BOOST_CHECK((a != NULL) && (a->content() == "123"));

			ChannelRead<TestB*> readB = channelB.waitForData(Duration::milliseconds(100));
			BOOST_CHECK(readB.isValid());
			BOOST_CHECK_EQUAL(readA->timestamp, readB->timestamp);

			const TestB* b = readB->value();
			BOOST_CHECK((b != NULL) && (b->id() == "C"));
			BOOST_CHECK((b != NULL) && (b->content() == "123"));
		}

		channelB2.post(new TestB("1", "2"));
		{
			ChannelRead<TestA*> readA = channelA.read();
			BOOST_CHECK(readA.isValid());
			const TestA* a = readA->value();
			BOOST_CHECK((a != NULL) && (a->id() == "B"));
			BOOST_CHECK((a != NULL) && (a->content() == "12"));

			ChannelRead<TestB*> readB = channelB.read();
			BOOST_CHECK(readB.isValid());
			BOOST_CHECK_EQUAL(readA->timestamp, readB->timestamp);

			const TestB* b = readB->value();
			BOOST_CHECK((b != NULL) && (b->id() == "B"));
			BOOST_CHECK((b != NULL) && (b->content() == "12"));
		}

		// Again: when publishing a Base object, a Derived subscriber will receive a null pointer

		channelB2.post(new TestA("1"));
		{
			ChannelRead<TestA*> readA = channelA.read();
			BOOST_CHECK(readA.isValid());
			const TestA* a = readA->value();
			BOOST_CHECK((a != NULL) && (a->id() == "A"));
			BOOST_CHECK((a != NULL) && (a->content() == "1"));

			ChannelRead<TestB*> readB = channelB.read();
			BOOST_CHECK(readB.isValid());
			BOOST_CHECK_EQUAL(readA->timestamp, readB->timestamp);

			const TestB* b = readB->value();
			BOOST_CHECK(b == NULL);
		}
	}
}
