/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkTransformer.C
 *    Implementation of FrameworkTransformer.h.
 *
 * @author Erik Einhorn
 * @date   2010/10/11
 */

#include <fw/FrameworkTransformer.h>

#include <error/Logging.h>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

FrameworkTransformer::FrameworkTransformer(Authority& authority) :
	mAuthority(authority)
{
	mAuthority.publishService(*this);
	mAuthority.registerCallbackForInterface("IFrameworkTransformer",
	                                        &FrameworkTransformer::registerInterface,
	                                        this);
}

FrameworkTransformer::~FrameworkTransformer()
{}

///////////////////////////////////////////////////////////////////////////////

void FrameworkTransformer::addLink(const std::string& childID,
                                   const std::string& parentID)
{
	MIRA_LOG(DEBUG) << "Added transform link: " << childID << " <- " << parentID;
	NodePtr child  = getNode(childID);
	NodePtr parent = getNode(parentID);

	// create child and/or parent if they do not exist, and subscribe on channels
	if(!child) {
		child=new FrameworkTransformerNode(childID, mAuthority);
		this->addNode(child);
		child->mChannel = mAuthority.subscribe<void>(childID);
	}
	if(!parent) {
		parent=new FrameworkTransformerNode(parentID, mAuthority);
		this->addNode(parent);
		parent->mChannel = mAuthority.subscribe<void>(parentID);
	}

	if(Base::addLink(child, parent)) {
		// if link was not already known, then synchronize all known transformers
		auto l = mAuthority.queryServicesForInterface("IFrameworkTransformer");
		foreach(const std::string& otherTransformer, l)
			if(otherTransformer!=mAuthority.getID())
				mAuthority.callService<void>(otherTransformer, "addLink",
				                             child->getID(), parent->getID());
	}
}

void FrameworkTransformer::storeTransformTree(const Path& filename)
{
	XMLDom xml;
	foreach(IDToNodeMap::value_type& n, mNodes)
	{
		if (n.second->getParent())
		{
			XMLDom::sibling_iterator i = xml.root().add_child("link");
			i.add_attribute("parent", n.second->getParent()->getID());
			i.add_attribute("child", n.second->getID());
		}
	}

	xml.saveToFile(filename);
}

void FrameworkTransformer::registerInterface(const std::string& interface,
                                             const std::string& otherTransformer)
{
	assert(interface=="IFrameworkTransformer");
	MIRA_LOG(DEBUG) << "Found other IFrameworkTransformer: " << otherTransformer;

	// tell the other transformer all of our links
	auto links = getLinks();
	foreach(auto& link, links) // links is a list of (parent,child) pairs
	{
		AbstractNodePtr parent = link.first;
		AbstractNodePtr child = link.second;
		mAuthority.callService<void>(otherTransformer, "addLink",
		                             child->getID(), parent->getID());
	}
}

///////////////////////////////////////////////////////////////////////////////

RigidTransform2f FrameworkTransformer::getTransform2(const std::string& target,
                                                     const std::string& source)
{
	return mAuthority.getTransform<RigidTransform2f>(target, source, Time::now());
}

RigidTransformCov2f FrameworkTransformer::getTransformCov2(const std::string& target,
                                                           const std::string& source)
{
	return mAuthority.getTransform<RigidTransformCov2f>(target,source, Time::now());
}

RigidTransform3f FrameworkTransformer::getTransform3(const std::string& target,
                                                     const std::string& source)
{
	return mAuthority.getTransform<RigidTransformCov3f>(target, source, Time::now());
}

RigidTransformCov3f FrameworkTransformer::getTransformCov3(const std::string& target,
                                                           const std::string& source)
{
	return mAuthority.getTransform<RigidTransformCov3f>(target, source, Time::now());
}

void FrameworkTransformer::publishTransform2(const std::string& nodeID,
                                             const RigidTransform2f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

void FrameworkTransformer::publishTransformCov2(const std::string& nodeID,
                                                const RigidTransformCov2f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

void FrameworkTransformer::publishTransform3(const std::string& nodeID,
                                             const RigidTransform3f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

void FrameworkTransformer::publishTransformCov3(const std::string& nodeID,
                                                const RigidTransformCov3f& transform)
{
	mAuthority.publishTransform(nodeID, transform, Time::now());
}

///////////////////////////////////////////////////////////////////////////////

// the specialized versions of FrameworkTransformReadAccessor and
// FrameworkTransformWriteAccessor are not defined inline in the header to
// reduce the compilation time dramatically

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, NearestNeighborInterpolator>::
get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, NearestNeighborInterpolator&) {
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolator>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolator& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolatorNoExtrapolation>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolatorNoExtrapolation& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolatorNearestNeighbourExtrapolator>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolatorNearestNeighbourExtrapolator& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

template <typename ChannelTransformType>
void FrameworkTransformReadAccessor<ChannelTransformType, LinearInterpolatorExtrapolationLimit>::
	get(Channel<void>& channel, const Time& time, ChannelTransformType& oTransform, LinearInterpolatorExtrapolationLimit& filter)
{
	Channel<ChannelTransformType> ch = channel_cast<ChannelTransformType>(channel);
	if(time.isValid())
		oTransform = ch.get(time, filter);
	else // get the latest
		oTransform = ch.get();
}

// explicit instantiation
template struct FrameworkTransformReadAccessor<RigidTransform2f,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,NearestNeighborInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolatorNoExtrapolation>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolatorNearestNeighbourExtrapolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2f,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2f,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransform3f,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3f,LinearInterpolatorExtrapolationLimit>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,NearestNeighborInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,NearestNeighborInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolatorNoExtrapolation>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolatorNoExtrapolation>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolatorNearestNeighbourExtrapolator>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolatorNearestNeighbourExtrapolator>;

template struct FrameworkTransformReadAccessor<RigidTransform2d,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov2d,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransform3d,LinearInterpolatorExtrapolationLimit>;
template struct FrameworkTransformReadAccessor<RigidTransformCov3d,LinearInterpolatorExtrapolationLimit>;


template <typename Transform>
void FrameworkTransformWriteAccessor<Transform>::set(Authority& authority, const std::string& nodeID,
                                                     const Transform& transform, const Time& time)
{
	Channel<Transform> channel = authority.publish<Transform>(nodeID);
	ChannelWrite<Transform> data = channel.write();

	std::string frameID;

	// determine node's parent as frame id (if it is a node and has a parent)
	FrameworkTransformer::NodePtr n = MIRA_FW.getTransformer()->getNode(nodeID);
	if (n) {
		AbstractTransformerNode* p = n->getParent();
		if (p)
			frameID = p->getID();
	}

	data = makeStamped(transform, time, frameID);
}

template struct FrameworkTransformWriteAccessor<RigidTransform2f>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov2f>;
template struct FrameworkTransformWriteAccessor<RigidTransform3f>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov3f>;
template struct FrameworkTransformWriteAccessor<RigidTransform2d>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov2d>;
template struct FrameworkTransformWriteAccessor<RigidTransform3d>;
template struct FrameworkTransformWriteAccessor<RigidTransformCov3d>;

///////////////////////////////////////////////////////////////////////////////

}
