/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Authority.C
 *    Implementation of AuthorityInstance, AuthorityInfo and Authority classes.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/10/04
 */

#include <fw/Framework.h>
#include <fw/Authority.h>

#include <serialization/PropertyTree.h>
#include <utils/UUID.h>

#include <fw/Framework.h>
#include <fw/RemoteModule.h>


namespace mira {

///////////////////////////////////////////////////////////////////////////////

Authority::Authority(Flags flags) :
	mFlags(flags) // initialization of other members goes to init()!
{
	init();
}


Authority::Authority(const ResourceName& fullname, Flags flags) : mFlags(flags)
{
	init();
	checkin(fullname);
	start();
}

Authority::Authority(const ResourceName& ns, const std::string& name,
                     Flags flags) :
	mFlags(flags)
{
	init();
	checkin(ns,name);
	start();
}

Authority::Authority(Authority& parent, const std::string& name, Flags flags) :
	mFlags(flags)
{
	init();
	checkin(parent, name);
}

void Authority::init()
{
	mValid = false;
	mStarted = false;
	mSubsciberEnabled = false;
	mParent = NULL;
	mStatusManager.registerDiagnostics("General", this);
}

Authority::~Authority()
{
	checkout();

	// destroy our RPCHandler
	mRPCHandler.reset();
}
void Authority::checkin(const ResourceName& fullname, AuthorityRuntimePtr runtime)
{
	if(mValid)
		MIRA_THROW(XLogical, "Trying to checkin authority instance as '" << fullname << "' " <<
			"which is already checked in as '" << getGlobalName() << "'");

	ResourceName ns = fullname.parent();
	std::string name = fullname.leaf();

	if(!ns.isFullyQualified())
		MIRA_THROW(XInvalidParameter, "Namespace '" << ns << "' of authority '" << name << "' is not fully qualified");

	std::string autoID = name;
	// Add prefix if we are hidden
	if ( mFlags & HIDDEN )
		autoID = MIRA_HIDDEN_AUTHORITY_PREFIX + autoID;
	// generate an auto id based on uuids, if we are anonymous
	if ( mFlags & ANONYMOUS )
		autoID += "_"+toString(boost::uuids::random_generator()());

	mDescription.ns = ns;
	mDescription.id = autoID;
	mDescription.name = name;
	mIsInternal = (mFlags & INTERNAL) == INTERNAL;
	mIsInvisiblePublisherSubscriber = (mFlags & INVISIBLE_PUBLISHER_SUBSCRIBER) > 0;

	DiagnosticsModule::setName(getGlobalID());

	// use the specified runtime, if any, otherwise create our own
	if(runtime) {
		mOwnsRuntime = false;
		mRuntime = runtime;
	}
	else {
		mOwnsRuntime = true;
		mRuntime.reset(new AuthorityRuntime(getGlobalID()));
	}

	if(mFlags & INDEPENDENT_SUBSCRIBER_THREAD) {
		// create independent dispatcher thread and handler
		mSubscriberDispatcher.reset(new DispatcherThread(getGlobalID()+"_subscribers"));
	}

	mServiceInterfaceHandler.reset(new ServiceInterfaceHandler(&getRuntime()->getMainDispatcher(), this));

	if(mFlags & INDEPENDENT_RPC_THREAD) {
		// create independent dispatcher thread
		mRPCDispatcher.reset(new DispatcherThread(getGlobalID()+"_rpc"));
		mRPCHandler.reset(new RPCHandler(mRPCDispatcher.get(), this));
	} else
		mRPCHandler.reset(new RPCHandler(&getRuntime()->getMainDispatcher(), this));


	mStatusManager.setID(getGlobalID());

	mValid = true;
	// add us to AuthorityManager
	MIRA_FW.getAuthorityManager().addAuthority(this);

	// Register ourselves at all connected remote frameworks
	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if (remote && !mIsInternal)
		remote->publishAuthority(getDescription());

	MIRA_LOG(NOTICE) << "Checked in authority '" << getGlobalID() << "'";
}

void Authority::checkin(const ResourceName& ns, const std::string& name,
                        AuthorityRuntimePtr runtime)
{
	checkin(ns/name, runtime);
}

void Authority::checkin(Authority& parent, const ResourceName& ns,
                        const std::string& name)
{
	checkin(ns, name, parent.getRuntime());
	parent.mChildren.insert(this);
	mParent = &parent;
	mParent->getStatusManager().registerDiagnostics(getName(), this);
}

void Authority::checkin(Authority& parent, const std::string& name)
{
	checkin(parent, parent.getNamespace(), name);
}

void Authority::checkout()
{
	if ( !mValid )
		return;

	// TODO: why not call stop here???

	foreach(Authority* child, mChildren)
		child->mParent = NULL;
	if (mParent)
	{
		mParent->getStatusManager().unregisterDiagnostics(getName());
		mParent->mChildren.erase(this);
	}

	// deactivate all subscribers
	enableSubscriberCallbacks(false);

	// stop all independent SubsriberHandlers and their dispatchers.
	foreach(DispatcherThreadPtr& s, mIndependentSubscriberDispatchers)
		s->stop();

	// stop our other dispatchers
	if(mSubscriberDispatcher)
		mSubscriberDispatcher->stop();

	if(mRPCDispatcher)
		mRPCDispatcher->stop();

	// remove and destroy our subscribers
	foreach(AbstractChannelSubscriberPtr subscriber, mSubscriberList)
	{
		// make sure the all signaled callbacks to this subscriber are done/removed
		subscriber->unsignal();
		// remove the subscriber from its channel
		subscriber->getChannel()->removeSubscriber(subscriber);
	}
	mSubscriberList.clear();


	// destroy the independent dispatchers
	mIndependentSubscriberDispatchers.clear();

	// Unregister our service interface handler
	MIRA_FW.getRPCManager().unregisterCallback(mServiceInterfaceHandler);
	mServiceInterfaceHandler.reset();

	// destroy our RPCHandler
	mRPCHandler.reset();


	RemoteModulePtr remote = MIRA_FW.getRemoteModule();

	// unpublish and unsubscribe from all channels
	if (remote && !mIsInternal)
		remote->unpublishAuthority(getDescription());

	MIRA_FW.getChannelManager().unpublish(getGlobalID(), mIsInvisiblePublisherSubscriber);
	MIRA_FW.getChannelManager().unsubscribe(getGlobalID());

	// remove our published services
	foreach(const std::string& name, mPublishedServices)
	{
		MIRA_FW.getRPCManager().removeLocalService(name);
		if ( remote )
			remote->unpublishService(name);
	}

	mPublishedServices.clear();
	mValid = false;

	MIRA_FW.getAuthorityManager().removeAuthority(this);
	MIRA_LOG(NOTICE) << "Checked out authority '" << getGlobalID() << "'";
}

void Authority::addImmediateHandler(DiagnosticRunnablePtr runnable)
{
	validate();
	runnable->setDiagnosticsModule(this);
	mRuntime->getMainDispatcher().addImmediateHandler(runnable);
}

void Authority::addFinalizeHandler(DiagnosticRunnablePtr runnable)
{
	validate();
	runnable->setDiagnosticsModule(this);
	mRuntime->getMainDispatcher().addFinalizeHandler(runnable);
}

TimerPtr Authority::createTimer(Duration period, TimerCallback callback,
                                bool oneshot)
{
	validate();
	return mRuntime->getMainDispatcher().createTimer(period, callback, oneshot);
}

TimerPtr Authority::createTimer(Duration period, Duration tolerance, 
                                TimerCallback callback, bool oneshot)
{
	validate();
	return mRuntime->getMainDispatcher().createTimer(period, tolerance,
	                                                 callback, oneshot);
}

TimerPtr Authority::createTimer(Time time, TimerCallback callback)
{
	validate();
	return mRuntime->getMainDispatcher().createTimer(time, callback);
}

void Authority::removeTimer(TimerPtr timer)
{
	validate();
	mRuntime->getMainDispatcher().removeTimer(timer);
}

void Authority::start()
{
	validate();

	if(mStarted)
		return;

	mStarted = true;

	if (mOwnsRuntime)
		mRuntime->getMainDispatcher().start(!(mFlags & NO_MAIN_THREAD));

	if(mSubscriberDispatcher)
		mSubscriberDispatcher->start();

	if(mRPCDispatcher)
		mRPCDispatcher->start();

	foreach(DispatcherThreadPtr& s, mIndependentSubscriberDispatchers)
		s->start();

	enableSubscriberCallbacks(true);

	foreach(Authority* child, mChildren)
		child->start();
}

void Authority::stop()
{
	validate();

	if(!mStarted)
		return;

	mStarted = false;

	foreach(Authority* child, mChildren)
		child->stop();

	enableSubscriberCallbacks(false);

	if (mOwnsRuntime)
		mRuntime->getMainDispatcher().stop();

	if(mSubscriberDispatcher)
		mSubscriberDispatcher->stop();

	if(mRPCDispatcher)
		mRPCDispatcher->stop();

	foreach(DispatcherThreadPtr& s, mIndependentSubscriberDispatchers)
		s->stop();

}

bool Authority::spin(const Duration& maxWait)
{
	if (mOwnsRuntime)
		return mRuntime->getMainDispatcher().spin(maxWait);
	return false;
}

bool Authority::hasUnrecoverableFailure() const
{
	validate();
	return mRuntime->getMainDispatcher().hasUnrecoverableFailure();
}

bool Authority::isStarted() const
{
	return mStarted;
}

bool Authority::isRunning() const
{
	if (!isValid())
		return false;
	return mRuntime->getMainDispatcher().isRunning();
}

///////////////////////////////////////////////////////////////////////////////

void Authority::enableSubscriberCallbacks(bool enable)
{
	validate();

	if (mSubsciberEnabled != enable) {
		mSubsciberEnabled = enable;
		foreach (AbstractChannelSubscriberPtr ptr, mSubscriberList)
			ptr->enable(enable);
	}
}

///////////////////////////////////////////////////////////////////////////////

void Authority::unpublish(const std::string& iChannelID)
{
	validate();

	// resolve channel's fully qualified name
	std::string channelID = resolveName(iChannelID);
	eraseChannelNameMapping(mPublishedChannelNames, iChannelID, channelID);

	// check if really published
	if ( !MIRA_FW.getChannelManager().hasPublished(getGlobalID(), channelID) )
		return;

	MIRA_FW.getChannelManager().unpublish(channelID, getGlobalID(), mIsInvisiblePublisherSubscriber);
	removeChannelWriteAccess(channelID);
}

bool Authority::isSubscribedOn(const std::string& channelID) const
{
	return MIRA_FW.getChannelManager().isSubscribedOn(getGlobalID(),
	                                                   resolveName(channelID));
}

bool Authority::hasPublished(const std::string& channelID) const
{
	return MIRA_FW.getChannelManager().hasPublished(getGlobalID(),
	                                                resolveName(channelID));
}

bool Authority::doesChannelExist(const std::string& channelID) const
{
	return MIRA_FW.getChannelManager().hasChannel(resolveName(channelID));
}

bool Authority::waitForChannel(const std::string& iChannelID, const Duration& timeout) const
{
	Time end;
	if(!timeout.isValid() || timeout.isInfinity())
		end = Time::eternity();
	else
		end = Time::now() + timeout;

	while (!boost::this_thread::interruption_requested())
	{
		if(doesChannelExist(iChannelID))
			return true;
		if(Time::now()>end) // handle timeout
			break;
		MIRA_SLEEP(50);
	}
	return false;
}

std::string Authority::resolveName(const std::string& name) const
{
	try
	{
		return MIRA_FW.getNameRegistry().resolve(name, getNamespace());
	}
	catch(Exception& ex)
	{
		MIRA_RETHROW(ex, "while trying to resolve name in Authority " << getGlobalID());
	}
	return "";
}

std::string Authority::resolveServiceName(const std::string& name) const
{
	// 8 is the length of #builtin suffix
	if(name.length() > 8 && name.substr(name.length() - 8) == "#builtin" ){
		std::string service = name.substr(0, name.length() - 8);
		return resolveName(service) + "#builtin";
	}
	else
		return resolveName(name);
}

void Authority::remotePublishService(const std::string& service) const
{
	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if ( remote )
		remote->publishService(service);
}

void Authority::setChannelReadAccess(const std::string& channelID)
{
	mAccessMap[channelID].flags |= CHANNEL_ACCESS_READ;
}

void Authority::setChannelWriteAccess(const std::string& channelID)
{
	mAccessMap[channelID].flags |= CHANNEL_ACCESS_WRITE;
}

void Authority::removeChannelReadAccess(const std::string& channelID)
{
	mAccessMap[channelID].flags ^= CHANNEL_ACCESS_READ;
}

void Authority::removeChannelWriteAccess(const std::string& channelID)
{
	mAccessMap[channelID].flags ^= CHANNEL_ACCESS_WRITE;
}

void Authority::insertChannelNameMapping(AbstractAuthority::ChannelNameMapping& map,
                                         const std::string& local,
                                         const std::string& global)
{
	boost::mutex::scoped_lock lock(mNameMutex);
	auto range = map.equal_range(global);
	AbstractAuthority::ChannelNameMapping::iterator it = range.first;
	for (; it != range.second; ++it)
		if (it->second == local)
			return;
	map.insert(std::make_pair(global, local));
}

void Authority::eraseChannelNameMapping(AbstractAuthority::ChannelNameMapping& map,
                                        const std::string& local,
                                        const std::string& global)
{
	boost::mutex::scoped_lock lock(mNameMutex);
	auto range = map.equal_range(global);
	AbstractAuthority::ChannelNameMapping::iterator it = range.first;
	for (; it != range.second; ++it)
		if (it->second == local)
		{
			map.erase(it);
			return;
		}
}

///////////////////////////////////////////////////////////////////////////////

bool Authority::isTransformAvailable(const std::string& targetID,
                                     const std::string& sourceID) const
{
	try
	{
		FrameworkTransformerNode* target = getTransformNode(targetID);
		FrameworkTransformerNode* source = getTransformNode(sourceID);
		if (MIRA_FW.getTransformer()->isTransformAvailable(target, source))
			return true;
	}
	catch (...)
	{}
	return false;
}

bool Authority::isTransformAvailable(const std::string& targetID,
                                     const std::string& sourceID,
                                     const std::string& fixedID) const
{
	try
	{
		FrameworkTransformerNode* target = getTransformNode(targetID);
		FrameworkTransformerNode* source = getTransformNode(sourceID);
		FrameworkTransformerNode* fixed = getTransformNode(fixedID);
		if (MIRA_FW.getTransformer()->isTransformAvailable(target, fixed) &&
			MIRA_FW.getTransformer()->isTransformAvailable(fixed, source))
			return true;
	}
	catch (...)
	{}
	return false;
}

bool Authority::waitForTransform(const std::string& targetID,
                                 const std::string& sourceID,
                                 Duration timeout) const
{
	validate();
	Time end;
	if(!timeout.isValid())
		end = Time::eternity();
	else
		end = Time::now() + timeout;

	while(!boost::this_thread::interruption_requested())
	{
		try
		{
			FrameworkTransformerNode* target = getTransformNode(targetID);
			FrameworkTransformerNode* source = getTransformNode(sourceID);
			if (MIRA_FW.getTransformer()->isTransformAvailable(target, source))
				return true;
		}
		catch (...)
		{}
		if(Time::now()>end) // handle timeout
			break;
		MIRA_SLEEP(100)
	}
	return false;
}

bool Authority::waitForTransform(const std::string& targetID,
                                 const std::string& sourceID,
                                 const std::string& fixedID, Duration timeout) const
{
	validate();
	Time end;
	if(!timeout.isValid())
		end = Time::eternity();
	else
		end = Time::now() + timeout;

	while(!boost::this_thread::interruption_requested())
	{
		try
		{
			FrameworkTransformerNode* target = getTransformNode(targetID);
			FrameworkTransformerNode* source = getTransformNode(sourceID);
			FrameworkTransformerNode* fixed = getTransformNode(fixedID);
			if (MIRA_FW.getTransformer()->isTransformAvailable(target, fixed) &&
				MIRA_FW.getTransformer()->isTransformAvailable(fixed, source))
				return true;
		}
		catch (...)
		{}
		if(Time::now()>end) // handle timeout
			break;
		MIRA_SLEEP(100)
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////////

void Authority::unpublishService()
{
	validate();
	unpublishService(this->getGlobalID());
}

void Authority::unpublishService(const std::string& name)
{
	std::string resolvedname = resolveServiceName(name);
	if (mPublishedServices.count(resolvedname) == 0)
	{
		MIRA_LOG(WARNING) << "Unpublishing service " 
			<< resolvedname << " that was not published before";
		return;
	}

	MIRA_FW.getRPCManager().removeLocalService(resolvedname);
	mPublishedServices.erase(resolvedname);
	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if (remote)
		remote->unpublishService(resolvedname);
}

RPCFuture<JSONRPCResponse> Authority::callServiceJSON(const json::Value& jsonRequest) const
{
	validate();
	return MIRA_FW.getRPCManager().callJSON(jsonRequest);
}

RPCFuture<JSONRPCResponse> Authority::callServiceJSON(const std::string& jsonString) const
{
	validate();
	return MIRA_FW.getRPCManager().callJSON(jsonString);
}

RPCFuture<JSONRPCResponse> Authority::callServiceJSON(const std::string& service,
                                                      const std::string& method,
                                                      const json::Value& params) const
{
	validate();
	return MIRA_FW.getRPCManager().callJSON(resolveServiceName(service), method, params);
}

RPCFuture<JSONRPCResponse> Authority::callServiceJSON(const std::string& service,
                                                      const std::string& method,
                                                      const std::string& params) const
{
	validate();
	return MIRA_FW.getRPCManager().callJSON(resolveServiceName(service), method, params);
}

RPCFuture<JSONRPCResponse> Authority::callServiceJSON(const RPCCallDefinition& rpc) const
{
	validate();
	return MIRA_FW.getRPCManager().callJSON(resolveServiceName(rpc.service),
	                                        rpc.method, rpc.params);
}

bool Authority::existsService(const std::string& name) const
{
	validate();
	std::string resolvedname = resolveServiceName(name);
	return MIRA_FW.getRPCManager().existsService(resolvedname);
}

bool Authority::implementsInterface(const std::string& name, const std::string& interface) const
{
	validate();
	std::string resolvedname = resolveServiceName(name);
	return MIRA_FW.getRPCManager().implementsInterface(resolvedname, interface);
}

bool Authority::waitForService(const std::string& name, Duration timeout) const
{
	validate();
	std::string resolvedname = resolveServiceName(name);
	return MIRA_FW.getRPCManager().waitForService(resolvedname, timeout);
}

std::string Authority::waitForServiceInterface(const std::string& interface,
                                               Duration timeout) const
{
	validate();
	return MIRA_FW.getRPCManager().waitForServiceInterface(interface, timeout);
}

std::list<std::string> Authority::queryServicesForInterface(const std::string& interface) const
{
	validate();
	return MIRA_FW.getRPCManager().queryServicesForInterface(interface);
}

void Authority::registerCallbackForInterface(const std::string& interface,
                                             ServiceInterfaceHandler::Callback cb)
{
	validate();
	mServiceInterfaceHandler->registerCallback(interface, cb);
	MIRA_FW.getRPCManager().registerCallback(interface, mServiceInterfaceHandler);
}

//////////////////////////////////////////////////////////////////////////////

PropertyTree Authority::getPropertiesRPC()
{
	boost::shared_ptr<PropertyNode> properties = getProperties();
	if (!properties)
		MIRA_THROW(XInvalidParameter, "Authority '" << getGlobalID() <<
		           "' does not have any properties");
	return PropertyTree(properties.get());
}

void Authority::setProperty(const std::string& property, const std::string& value)
{
	boost::shared_ptr<PropertyNode> properties = getProperties();
	if (!properties)
		MIRA_THROW(XInvalidParameter, "Authority '" << getGlobalID() <<
		           "' does not have any properties");
	PropertyNode* node = properties->findChildNode(property);
	if (node == NULL)
		MIRA_THROW(XInvalidParameter, "The property '" << property <<
		           "' could not be found in '" << getGlobalID() << "'");

	node->setFromString(value);
}

std::string Authority::getProperty(const std::string& property) const
{
	Authority* This = const_cast<Authority*>(this);
	boost::shared_ptr<PropertyNode> properties = This->getProperties();
	if (!properties)
		MIRA_THROW(XInvalidParameter, "Authority '" << getGlobalID() <<
		           "' does not have any properties");
	PropertyNode* node = properties->findChildNode(property);
	if (node == NULL)
		MIRA_THROW(XInvalidParameter, "The property '" << property <<
		           "' could not be found in '" << getGlobalID() << "'");

	return node->getAsString();
}

void Authority::setPropertyJSON(const std::string& property, const json::Value& value)
{
	boost::shared_ptr<PropertyNode> properties = getProperties();
	if (!properties)
		MIRA_THROW(XInvalidParameter, "Authority '" << getGlobalID() <<
		           "' does not have any properties");
	PropertyNode* node = properties->findChildNode(property);
	if (node == NULL)
		MIRA_THROW(XInvalidParameter, "The property '" << property <<
		           "' could not be found in '" << getGlobalID() << "'");

	node->setFromJSON(value);
}

json::Value Authority::getPropertyJSON(const std::string& property) const
{
	Authority* This = const_cast<Authority*>(this);
	boost::shared_ptr<PropertyNode> properties = This->getProperties();
	if (!properties)
		MIRA_THROW(XInvalidParameter, "Authority '" << getGlobalID() <<
		           "' does not have any properties");
	PropertyNode* node = properties->findChildNode(property);
	if (node == NULL)
		MIRA_THROW(XInvalidParameter, "The property '" << property <<
		           "' could not be found in '" << getGlobalID() << "'");

	return node->getAsJSON();
}

//////////////////////////////////////////////////////////////////////////////

StatusManager& Authority::getStatusManager()
{
	return mStatusManager;
}

StatusManager::StatusMap Authority::getStatusMap()
{
	return mStatusManager.getStatusMap();
}

boost::shared_ptr<PropertyNode> Authority::getProperties()
{
	return boost::shared_ptr<PropertyNode>();
}

std::set<std::string> Authority::getPublishedChannels()
{
	std::set<std::string> r = MIRA_FW.getChannelManager().getPublishedChannelsBy(getGlobalID());
	foreach(Authority* child, mChildren)
	{
		std::set<std::string> temp;
		std::set<std::string> other = MIRA_FW.getChannelManager().getPublishedChannelsBy(child->getGlobalID());
		std::set_union(r.begin(), r.end(),
		               other.begin(), other.end(),
		               std::inserter(temp, temp.begin()));
		r.swap(temp);
	}
	return r;
}

std::set<std::string> Authority::getSubscribedChannels()
{
	std::set<std::string> r = MIRA_FW.getChannelManager().getSubscribedChannelsBy(getGlobalID());
	foreach(Authority* child, mChildren)
	{
		std::set<std::string> temp;
		std::set<std::string> other = MIRA_FW.getChannelManager().getSubscribedChannelsBy(child->getGlobalID());
		std::set_union(r.begin(), r.end(),
		               other.begin(), other.end(),
		               std::inserter(temp, temp.begin()));
		r.swap(temp);
	}
	return r;
}

std::set<std::string> Authority::getServiceInterfaces()
{
	std::set<std::string> r;
	try {
		RPCServer::Service const& service = MIRA_FW.getRPCManager().getLocalService(getGlobalID());
		r.insert( service.interfaces.begin(), service.interfaces.end() );
	}
	catch (XInvalidParameter&) {
		// ignore this error since it is not guaranteed that an authority provides a service interface
	}
	return r;
}

AbstractAuthority::ChannelNameMapping Authority::getPublishedChannelNames()
{
	boost::mutex::scoped_lock lock(mNameMutex);
	ChannelNameMapping r = mPublishedChannelNames;
	foreach(Authority* child, mChildren)
	{
		ChannelNameMapping other = child->getPublishedChannelNames();
		r.insert(other.begin(), other.end());
	}
	return r;
}

AbstractAuthority::ChannelNameMapping Authority::getSubscribedChannelNames()
{
	boost::mutex::scoped_lock lock(mNameMutex);
	ChannelNameMapping r = mSubscribedChannelNames;
	foreach(Authority* child, mChildren)
	{
		ChannelNameMapping other = child->getSubscribedChannelNames();
		r.insert(other.begin(), other.end());
	}
	return r;
}

///////////////////////////////////////////////////////////////////////////////

}
