/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CommonTest.h
 *
 * @author Erik Einhorn
 */

#ifndef _MIRA_COMMONTEST_H_
#define _MIRA_COMMONTEST_H_

#ifndef Q_MOC_RUN
#include <boost/format.hpp>
#endif

#include <serialization/XMLSerializer.h>
#include <serialization/JSONSerializer.h>
#include <serialization/BinarySerializer.h>
#include <serialization/Print.h>

#include <stream/GzStream.h>

using namespace mira;
using namespace std;


template<typename Class>
void testXmlFromString(const string& s)
{
	// deserialize test class
	Class c2; // with 0
	XMLDom xml2;
	xml2.loadFromString(s);
	XMLDeserializer s2(xml2.croot());
	s2.deserialize("Test", c2);
	c2.check();

	cout << "Test" << print(c2) << endl;
}

template<typename Class>
void testXml(const char* baseName, int num)
{
	{
		string filename = (boost::format("%s%d.xml") % baseName % num).str();

		// serialize test class
		Class c(true);
		XMLDom xml;
		XMLSerializer s(xml);
		s.serialize("Test", c);
		xml.saveToFile(filename);

		// deserialize test class
		Class c2; // with 0
		XMLDom xml2;
		xml2.loadFromFile(filename);
		XMLDeserializer s2(xml2.croot());
		s2.deserialize("Test", c2);
		c2.check();

		cout << "Test:" << print(c2) << endl;
	}
	{
		string filename = (boost::format("%s%d_compressed.xml") % baseName % num).str();

		// serialize test class
		Class c(true);
		XMLDom xml;
		XMLSerializer s(xml, XMLSerializer::COMPRESSED);
		s.serialize("Test", c);
		xml.saveToFile(filename);

		// deserialize test class
		Class c2; // with 0
		XMLDom xml2;
		xml2.loadFromFile(filename);
		XMLDeserializer s2(xml2.croot());
		s2.deserialize("Test", c2);
		c2.check();

		cout << "Test:" << print(c2) << endl;
	}
}

template<typename Class>
void testJSON(const char* baseName, int num)
{
	{
		string filename = (boost::format("%s%d.json") % baseName % num).str();
		// serialize test class
		Class c(true);
		JSONSerializer s;
		try
		{
			json::Value v = s.serialize(c);
			std::ofstream of(filename.c_str());
			json::write(v, of, true);
			Class c2; // with 0
			JSONDeserializer d(v);
			d.deserialize(c2);
			c2.check();
			cout << "Test:" << print(c2) << endl;
		}
		catch(XInvalidParameter&)
		{
			// We can't handle pointers so catch this exception and return.
			return;
		}
	}
	{
		string filename = (boost::format("%s%d.json") % baseName % num).str();
		// serialize test class
		Class c(true);
		JSONSerializer s(false, JSONSerializer::STRING_MAP_AS_OBJECT);
		try
		{
			json::Value v = s.serialize(c);
			std::ofstream of(filename.c_str());
			json::write(v, of, true);
			Class c2; // with 0
			JSONDeserializer d(v, JSONDeserializer::ACCEPT_STRING_MAP_AS_OBJECT);
			d.deserialize(c2);
			c2.check();
			cout << "Test:" << print(c2) << endl;
		}
		catch(XInvalidParameter&)
		{
			// We can't handle pointers so catch this exception and return.
			return;
		}
	}	
}

template<typename Class>
void testBinaryBuffer()
{
	// serialize test class
	Class c(true);
	BinaryBufferOstream::buffer_type buffer;
	BinaryBufferSerializer s(&buffer);
	s.serialize(c);

	//cout << ">" << print(buffer) << "<" << endl;

	// deserialize test class
	Class c2; // with 0

	BinaryBufferDeserializer s2(&buffer);
	s2.deserialize(c2);
	c2.check();
}

template<typename Class>
void testBinaryStream()
{
	// serialize test class
	Class c(true);
	std::stringbuf buffer;
	BinaryStreamSerializer s(&buffer);

	s.serialize(c);
	//cout << ">" << print(buffer) << "<" << endl;

	// deserialize test class
	Class c2; // with 0

	BinaryStreamDeserializer s2(&buffer);
	s2.deserialize(c2);
	c2.check();
}

template<typename Class>
void testBinaryGZStream()
{
	// serialize test class
	Class c(true);

	{
		ogzstream os("test.gz", std::fstream::out | std::fstream::binary);
		BinaryStlOstream obuffer(os);
		BufferedBinaryStreamSerializer s(obuffer);

		s.serialize(c);
	}

	//cout << ">" << print(buffer) << "<" << endl;

	// deserialize test class
	Class c2; // with 0

	igzstream is("test.gz", std::fstream::in | std::fstream::binary);
	BinaryStlIstream ibuffer(is);
	BinaryStreamDeserializer s2(ibuffer);

	s2.deserialize(c2);
	c2.check();
}

template<typename Class>
void testAll(const char* baseName, int num)
{
	cout << "TEST: " << baseName << " - " << num << endl;
	testXml<Class>(baseName, num);
	testJSON<Class>(baseName, num);
	testBinaryBuffer<Class>();
	testBinaryStream<Class>();
	testBinaryGZStream<Class>();
}

#endif
