/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AngleTest.C
 *    Testcases for the Angle classes.
 *
 * @author Erik Einhorn
 * @date   2010/08/30
 */

#include <iostream>

#include <boost/test/unit_test.hpp>
#if BOOST_VERSION >= 107100
#include <boost/test/tools/floating_point_comparison.hpp>
#else
#include <boost/test/floating_point_comparison.hpp>
#endif

#include <math/Angle.h>

#include <serialization/XMLSerializer.h>

#include <utils/Time.h>

using namespace std;
using namespace mira;


// tests constructors and conversation (basically checks if everything compiles)
BOOST_AUTO_TEST_CASE( ConstructionTest )
{
	Radian<float> rf1;
	Radian<float> rf2(1.0f);
	Radian<float> rf3(rf2);
	Radian<float> rf4(Degree<float>(180));
	Radian<float> rf5(pi<float>()*1000.f+0.5f);
	Radian<float> rf6(-pi<float>()*1000.f+0.5f);
	Degreei rf7(-3600);

	BOOST_CHECK_EQUAL(rf2.value(),1.0f);
	BOOST_CHECK_EQUAL(rf3.value(),1.0f);
	BOOST_CHECK_CLOSE(rf4.value(),pi<float>(),0.001f);
	BOOST_CHECK_CLOSE(rf5.value(),0.5f,0.01f);
	BOOST_CHECK_CLOSE(rf6.value(),0.5f,0.01f);
	BOOST_CHECK_EQUAL(rf7.value(),0.0f);

	Degree<int> ri1;
	Degree<int> ri2(10);
	Degree<int> ri3(ri2);
	// Remember when converting from float to int to add 0.5
	// for correct rounding 
	Degree<int> ri4(Radian<double>(pi<double>()*0.5) + Degree<double>(0.5));

	BOOST_CHECK_EQUAL(ri2.value(),10);
	BOOST_CHECK_EQUAL(ri3.value(),10);
	BOOST_CHECK_EQUAL(ri4.value(),90);
}


BOOST_AUTO_TEST_CASE( AngleFloatTest )
{
	Anglef a = Degreef(300);
	BOOST_CHECK_CLOSE(a.rad(), 5.23599f, 0.0001f);

	Anglef b = Degreef(70);
	BOOST_CHECK_CLOSE(b.rad(), 1.22173f, 0.0001f);

	Anglef c;
	c = a + b;

	BOOST_CHECK_CLOSE(c.deg(), 10.0f, 0.01f);

	// some compiler tests
	Degreef d(80);
	c = a + d;
	c = a+ 10.34f;
	c = 10.34f + a;
	c = 10.34f + Degreef(90);
}

BOOST_AUTO_TEST_CASE( AngleDoubleTest )
{
	Angled a = Degreef(300);
	BOOST_CHECK_CLOSE(a.rad(), 5.23599, 0.0001);

	Angled b = Degreef(70);
	BOOST_CHECK_CLOSE(b.rad(), 1.22173, 0.0001);

	Angled c;
	c = a + b;

	BOOST_CHECK_CLOSE(c.deg(), 10.0, 0.01);
}

// tests the operators
BOOST_AUTO_TEST_CASE( OperatorTest )
{
	Radian<float> a(1);
	Radian<float> b(2);
	Radian<float> c(2);

	// comparison
	BOOST_CHECK_EQUAL(a>b,false);
	BOOST_CHECK_EQUAL(a>=b,false);
	BOOST_CHECK_EQUAL(a<b,true);
	BOOST_CHECK_EQUAL(a<=b,true);
	BOOST_CHECK_EQUAL(a==b,false);
	BOOST_CHECK_EQUAL(a!=b,true);
	BOOST_CHECK_EQUAL(b==c,true);


	BOOST_CHECK_EQUAL(a+b,Radian<float>(3.0f));
	BOOST_CHECK_EQUAL(b-a,Radian<float>(1.0f));
	BOOST_CHECK_EQUAL(b*5.0f,Radian<float>(10.0f));

	a+=b;
	BOOST_CHECK_EQUAL(a,Radian<float>(3.0f));

	a-=b;
	BOOST_CHECK_EQUAL(a,Radian<float>(1.0f));

	b*=4;
	BOOST_CHECK_EQUAL(b,Radian<float>(8.0f));

}

BOOST_AUTO_TEST_CASE( AngleSignedFloatTest )
{
	SignedAnglef a = SignedDegreef(270);

	BOOST_CHECK_CLOSE(a.rad(), -1.570796f, 0.0001f);

	SignedAnglef b = SignedDegreef(90);
	BOOST_CHECK_CLOSE(b.rad(), 1.570796f, 0.0001f);

	SignedAnglef c;
	c = a + b + SignedDegreef(10);
	BOOST_CHECK_CLOSE(c.deg(), 10.f, 0.0001f);

	SignedAnglef minus = SignedDegreef(-135);
	SignedAnglef plus = SignedDegreef(135);
	Anglef result = plus - minus;
	BOOST_CHECK_CLOSE(result.deg(), 270.f, 0.0001f);
}

BOOST_AUTO_TEST_CASE( AngleSignedDoubleTest )
{
	SignedAngled a = SignedDegreed(270);

	BOOST_CHECK_CLOSE(a.rad(), -1.570796, 0.0001);

	SignedAngled b = SignedDegreed(90);
	BOOST_CHECK_CLOSE(b.rad(), 1.570796, 0.0001);

	SignedAngled c;
	c = a + b + SignedDegreed(10);
	BOOST_CHECK_CLOSE(c.deg(), 10, 0.0001);
}

BOOST_AUTO_TEST_CASE( AngleSerializationTest )
{
	SignedAnglef a = SignedDegreef(70);
	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("Angle", a);

	std::cout << xml.saveToString() << std::endl;

	SignedAnglef b;
	XMLDeserializer d(xml);
	d.deserialize("Angle", b);
	BOOST_CHECK(a==b);
}

BOOST_AUTO_TEST_CASE( AngleCoverageTest )
{
	Anglef a = Radianf(pi<float>()/2.0f);
	BOOST_CHECK_CLOSE(a.rad(), pi<float>()/2.0f, 0.0001f);

	SignedAnglef b(a);
	Anglef c(a);
	BOOST_CHECK_CLOSE(a.rad(),b.rad(),0.0001f);
	BOOST_CHECK_CLOSE(c.rad(),a.rad(),0.0001f);
}



BOOST_AUTO_TEST_CASE( AngleBenchmarkFloat )
{
	std::cout << "With Radian<float>:" << std::endl;
	Radian<float> r1(0.0f);
	Degree<float> inc1(0.01f);

	// test if it compiles correctly
	r1 = r1;
	r1 = inc1;
	r1 = 0.0f;

	Time start = Time::now();
	for (int i=0; i<4000000; ++i)
		r1 = r1 + inc1;

	std::cout << "took: " << Time::now()-start << std::endl;
	std::cout << r1.value() << std::endl;


	std::cout << "With plain floats:" << std::endl;
	float r2(0.0f);
	float inc2(0.01f);


	start = Time::now();
	for (int i=0; i<4000000; ++i)
		r2 = r2+inc2;

	std::cout << "took: " << Time::now()-start << std::endl;
	std::cout << r2 << std::endl;
}

BOOST_AUTO_TEST_CASE( AngleModuloBenchmark )
{
	for(int i=0;i<1000;i++)
	{
		float w=1+ pi<float>()*i;
		Time start = Time::now();
		float sum=0;
		for(int j=0;j<100000;j++)
		{
			float k=w;
			while(k>pi<float>()*2)
				k=k-pi<float>()*2;
			sum+=k;
		}
		auto t1=Time::now()-start;
		start = Time::now();
		std::cout<<"sum "<<sum<<std::endl;
		float lower=0.5;
		float upper=0.5+pi<float>()*2;
		sum=0;
		for(int j=0;j<100000;j++)
		{
			float k=w;
			k -= pi<float>()*2 * floor( (k - lower) / (pi<float>()*2) );
			sum+=k;
		}
		std::cout<<"sum "<<sum<<std::endl;
		auto t2=Time::now()-start;
		if(t2<t1)
		{
			std::cout<<"for i="<<i<<" fmod is faster"<<std::endl;
			break;
		}
	}
}


BOOST_AUTO_TEST_CASE( AngleDifferenceTest )
{
	// will yield 340 degrees (since 350-10=340)
	Degreei d1 = Degreei(350) - Degreei(10);
	// will yield -20 degrees (since 10-350=-340 corresponding to 20 degrees)
	Degreei d2 = Degreei(10)  - Degreei(350);
	std::cout<<"d2"<<d2<<std::endl;

	BOOST_CHECK(d1 == 340);
	BOOST_CHECK(d2 == 20);

	// will yield -20 degrees (since 350-10=340 corresponding to 340 degrees)
	SignedDegreei d3 = Degreei(350) - Degreei(10);
	 // will yield 20 degrees (since 10-350=20 corresponding to 20 degrees)
	SignedDegreei d4 = Degreei(10)  - Degreei(350);
	std::cout<<"d4"<<d4<<std::endl;

	BOOST_CHECK(d3 == -20);
	BOOST_CHECK(d4 == 20);

	// will yield -20 degrees (since 350-10=340 corresponding to 340 degrees)
	SignedDegreei d5 = Degreei(350).smallestDifference(Degreei(10));
	// will yield 20 degrees (since 10-350=20 corresponding to 20 degrees)
	SignedDegreei d6 = Degreei(10).smallestDifference(Degreei(350));
	BOOST_CHECK(d5 == -20);
	BOOST_CHECK(d6 == 20);

}

BOOST_AUTO_TEST_CASE( AngleAbsTest )
{
	BOOST_CHECK_EQUAL(std::abs(SignedAnglef(-1.0f)),SignedAnglef(1.0f));

	BOOST_CHECK_EQUAL(std::abs(SignedRadianf(-1.0f)),SignedRadianf(1.0f));

	BOOST_CHECK_EQUAL(std::abs(SignedDegreef(-45.0f)),SignedDegreef(45.0f));
	BOOST_CHECK_EQUAL(std::abs(Degreef(-45.0f)),Degreef(315.0f));
}

BOOST_AUTO_TEST_CASE( AngleBenchmarkInt )
{
	std::cout << "With Degree<int>:" << std::endl;
	Degree<int> r1(0);
	Degree<int> inc1(1);

	r1 = r1;
	r1 = inc1;
	r1 = 0.0f;

	Time start = Time::now();
	for (int i=0; i<4000000; ++i)
		r1 = r1 + inc1;

	std::cout << "took: " << Time::now()-start << std::endl;
	std::cout << r1.value() << std::endl;


	std::cout << "With plain ints:" << std::endl;
	int r2(0);
	int inc2(1);

	start = Time::now();
	for (int i=0; i<4000000; ++i)
		r2 = r2+i;

	std::cout << "took: " << Time::now()-start << std::endl;
	std::cout << r2 << std::endl;
}
