/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCManager.C
 *    Implements RPCManager.
 *
 * @author Erik Einhorn
 * @date   2010/11/25
 */

#include <rpc/RPCManager.h>

#include <thread/Thread.h>
#include <utils/Foreach.h>
#include <utils/MakeString.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename Backend>
struct RPCBackendAdapter
{
	static typename Backend::ContainerType fromBuffer(Buffer<uint8> buffer);
	static Buffer<uint8> toBuffer(typename Backend::ContainerType value);
};

template<uint8 BinaryFormatVersion>
struct RPCBackendAdapter<BinaryRPCBackendTempl<BinaryFormatVersion>>
{
	typedef BinaryRPCBackendTempl<BinaryFormatVersion> BinaryRPCBackend;
	static typename BinaryRPCBackend::ContainerType fromBuffer(Buffer<uint8> buffer) {
		return std::move(buffer);
	}

	static Buffer<uint8> toBuffer(typename BinaryRPCBackend::ContainerType value){
		return std::move(value);
	}
};

template<>
struct RPCBackendAdapter<JSONRPCBackend>
{
	static JSONRPCBackend::ContainerType fromBuffer(Buffer<uint8> buffer) {
		JSONRPCBackend::ContainerType value;
		std::string s((char*)buffer.data());
		json::read(s,value);
		return value;
	}

	static Buffer<uint8> toBuffer(JSONRPCBackend::ContainerType value){
		std::string s = json::write(value, false, 0);
		Buffer<uint8> buffer((uint8*)s.c_str(),s.size());

		// We have to copy the buffer!
		Buffer<uint8> result;
		result.copy(buffer);
		return result;
	}
};

///////////////////////////////////////////////////////////////////////////////


template<typename Backend>
struct RPCManager::RequestContext : public AbstractRequestContext
{
	RequestContext(typename Backend::ContainerType iBuffer,
	               RemoteFinishHandlerPtr iHandler, bool iRemote) :
		AbstractRequestContext(iHandler, iRemote),
		requestBuffer(std::move(iBuffer)),
		request(&requestBuffer), response(&responseBuffer) {}

	virtual void onFinished(RPCClient& client)
	{
		if(!remote) {
			typename Backend::ClientResponse response(&responseBuffer);
			client.handleResponse<Backend>(response);
		} else {
			if(handler) {
				handler->onRPCfinished(RPCBackendAdapter<Backend>::toBuffer(responseBuffer));
			}
		}
	}

	typename Backend::ContainerType  requestBuffer;
	typename Backend::ContainerType  responseBuffer;
	typename Backend::ServerRequest  request;
	typename Backend::ServerResponse response;
};

///////////////////////////////////////////////////////////////////////////////

void RPCManager::removeLocalService(const std::string& name)
{
	boost::mutex::scoped_lock lock(mServiceManagementMutex);

	if (mLocalServices.count(name) == 0)
		return;

	// remove all service methods from that service
	for(auto it=mLocalServiceMethods.begin(); it!=mLocalServiceMethods.end();)
	{
		std::vector<std::string> strs;
		boost::split(strs, it->first, boost::is_from_range('.','.'));
		if(!strs.empty() && strs[0]==name) {
			auto deleteIt = it;
			++it;
			mLocalServiceMethods.erase(deleteIt);
		} else
			++it;
	}

	mLocalServices.erase(name);
	mServer.unregisterService(name);
}

std::set<std::string> RPCManager::getLocalServices() const
{
	boost::mutex::scoped_lock lock(mServiceManagementMutex);

	return mLocalServices;
}

///////////////////////////////////////////////////////////////////////////////

void RPCManager::addRemoteService(const RemoteService& serviceInfo,
                                  RemoteRequestHandlerPtr handler,
                                  uint8 binaryFormatVersion)
{
	boost::mutex::scoped_lock lock(mServiceManagementMutex);
	auto existing = mRemoteServices.find(serviceInfo.name);

	std::set<std::string> existingInterfaces;
	// new service -> insert
	if (existing == mRemoteServices.end())
		mRemoteServices.insert(std::make_pair(serviceInfo.name,
		                                      RemoteServiceDetail(serviceInfo, handler,
		                                                          binaryFormatVersion)));
	// if service was already registered with this name check if rpc handler matches
	else
	{
		if (existing->second.handler != handler)
			MIRA_THROW(XLogical, "The remote service '"
			           << serviceInfo.name
			           << "' was already registered with a different handler");
		existingInterfaces = existing->second.interfaces;
		// overwrite the informations about this service with new ones
		existing->second = RemoteServiceDetail(serviceInfo, handler, binaryFormatVersion);
	}
	std::set<std::string> newInterfaces;
	std::set_difference(serviceInfo.interfaces.begin(), serviceInfo.interfaces.end(),
	                    existingInterfaces.begin(), existingInterfaces.end(),
	                    std::inserter(newInterfaces, newInterfaces.begin()));

	lock.unlock();
	// loop through all new interfaces of the service
	foreach(const std::string& ifc, newInterfaces)
		handleCallbacks(ifc, serviceInfo.name);
}

void RPCManager::removeRemoteService(const std::string& name)
{
	boost::mutex::scoped_lock lock(mServiceManagementMutex);
	mRemoteServices.erase(name);
}

std::set<std::string> RPCManager::getRemoteServices() const
{
	boost::mutex::scoped_lock lock(mServiceManagementMutex);
	std::set<std::string> res;
	foreach(auto const &v, mRemoteServices)
		res.insert(v.first);
	return res;
}

const RPCManager::RemoteService& RPCManager::getRemoteService(const std::string& name) const
{
	boost::mutex::scoped_lock lock(mServiceManagementMutex);

	auto i = mRemoteServices.find(name);
	if (i == mRemoteServices.end())
		MIRA_THROW(XInvalidParameter, "No such service '" << name << "'");

	return i->second;
}

void RPCManager::registerCallback(const std::string& interface,
                                  AbstractInterfaceCallbackHandlerPtr callback)
{
	boost::mutex::scoped_lock lock(mCallbacksMutex);
	mInterfaceCallbacks.insert(std::make_pair(interface, callback));
}

void RPCManager::unregisterCallback(const std::string& interface,
                                    AbstractInterfaceCallbackHandlerPtr callback)
{
	auto cbrange = mInterfaceCallbacks.equal_range(interface);
	for(auto it=cbrange.first; it!=cbrange.second;)
	{
		if (it->second == callback)
		{
			auto tmp = it;
			++it;
			mInterfaceCallbacks.erase(tmp);
		}
		else
			++it;
	}
}

void RPCManager::unregisterCallback(AbstractInterfaceCallbackHandlerPtr callback)
{
	for(auto it=mInterfaceCallbacks.begin(); it!=mInterfaceCallbacks.end();)
	{
		if (it->second == callback)
		{
			auto tmp = it;
			++it;
			mInterfaceCallbacks.erase(tmp);
		}
		else
			++it;
	}
}

bool RPCManager::waitForService(const std::string& name, Duration timeout) const
{
	Time end;
	if(!timeout.isValid() || timeout.isInfinity())
		end = Time::eternity();
	else
		end = Time::now() + timeout;

	while(!boost::this_thread::interruption_requested())
	{
		if(existsService(name))
			return true;

		if(Time::now()>end) // handle timeout
			break;
		MIRA_SLEEP(100)
	}
	return false;
}

std::string RPCManager::waitForServiceInterface(const std::string& interface,
                                               Duration timeout) const
{
	Time end;
	if(!timeout.isValid() || timeout.isInfinity())
		end = Time::eternity();
	else
		end = Time::now() + timeout;

	while(!boost::this_thread::interruption_requested())
	{
		auto l = queryServicesForInterface(interface);

		if (!l.empty()) // found a service, so return it
			return *l.begin();

		if(Time::now()>end)
			break;
		MIRA_SLEEP(100)
	}

	// timeout or interruption, before we found a service
	return std::string();
}

void RPCManager::handleCallbacks(const std::string& interface,
                                 const std::string& servicename)
{
	auto cbrange = mInterfaceCallbacks.equal_range(interface);
	for(auto it=cbrange.first; it!=cbrange.second; ++it)
		it->second->newServiceInterface(interface, servicename);
}

///////////////////////////////////////////////////////////////////////////////

template <typename Backend>
void RPCManager::handleLocalRequest(typename Backend::ContainerType buffer,
                                    const std::string& service)
{
	// unpack the method signature
	// TODO: this only is used to get the name of the method, in order to be
	//       able to find the servicemethod within our mLocalServiceMethods
	//       set. We could speed up the whole thing, if our callers would pass
	//       that method name.
	typename Backend::ServerRequest rq(&buffer);
	std::string id, s;
	rq.getHeader(id,s);
	RPCSignature sig = rq.getSignature();
	std::string servicemethod = service + "." + sig.name;


	boost::mutex::scoped_lock lock(mServiceManagementMutex);

	// get the handler for this servicemethod
	auto it = mLocalServiceMethods.find(servicemethod);
	if(it==mLocalServiceMethods.end())
		MIRA_THROW(XRPC, "Local service method '" << servicemethod << "' does not exist.");

	AbstractRPCHandlerPtr handler = it->second;

	// free the lock, since below processing could block
	lock.unlock();

	/**
	 * SPECIAL CASE:
	 * if RPC method was called within the same thread where the handling of
	 * the RPC calls is done, we need to call invoke RPC method immediately,
	 * to avoid deadlocks, since the caller might wait for the returned future
	 * and would block, since the requested RPC call will never be processed.
	 */
	if(handler->getThreadID() == boost::this_thread::get_id()) {
		typename Backend::ServerRequest  request(&buffer);
		typename Backend::ContainerType  responseBuffer;
		typename Backend::ServerResponse response(&responseBuffer);
		mServer.processCall<Backend>(request, response); // invoke call immediately!
		typename Backend::ClientResponse clientresponse(&responseBuffer);
		mClient.handleResponse<Backend>(clientresponse);
		return;
	}

	// otherwise treat local call as normal request
	handleRequest<Backend>(std::move(buffer), RemoteFinishHandlerPtr(), false);
}

// explicite instantiation needed so we can call it from call(), defined in RPCManager.h
template void RPCManager::handleLocalRequest<BinaryRPCBackend>(BinaryRPCBackend::ContainerType buffer,
                                                               const std::string& service);

template <typename Backend>
void RPCManager::handleRequest(typename Backend::ContainerType buffer,
                               RemoteFinishHandlerPtr finishHandler, bool remote)
{
	boost::mutex::scoped_lock lock1(mServiceManagementMutex);

	RequestContext<Backend>* c = new RequestContext<Backend>(std::move(buffer),
	                                                         finishHandler, remote);
	AbstractRequestContextPtr ctx(c);

	AbstractDeferredInvokerPtr invoker = mServer.processCallDeferred<Backend>(c->request,
	                                                                          c->response);

	// if no invoker, then something went wrong in processCallDeferred, then
	// error will be reported in response, so finish the call here and send
	// the error response
	if(!invoker) {
		ctx->onFinished(mClient);
		return;
	}

	// get the handler for this service method
	std::string method = invoker->getServiceName() + "." + invoker->getMethodName();

	auto it = mLocalServiceMethods.find(method);
	assert(it!=mLocalServiceMethods.end()); // should never happen, mServer.processCallDeferred() would throw an exception already

	AbstractRPCHandlerPtr handler = it->second;

	lock1.unlock();

	invoker->setFinishHandler(this);

	boost::mutex::scoped_lock lock2(mPendingRequestsMutex);
	mPendingRequests.insert(std::make_pair(invoker.get(), ctx));
	handler->queueRequest(invoker);
}

void RPCManager::handleRemoteRequest(Buffer<uint8> buffer, RemoteFinishHandlerPtr handler)
{
	if(buffer.empty())
		return;

	BinaryBufferDeserializer ds(&buffer);	
	uint8 marker;
	ds.deserialize(marker, false);

	if(marker==BinaryRPCBackend::BINARY_RPC_MARKER) {
		handleRequest<BinaryRPCBackend>(std::move(buffer), handler, true);
	} else {
		handleRequest<JSONRPCBackend>(std::move(RPCBackendAdapter<JSONRPCBackend>::fromBuffer(buffer)),
 		                              handler, true);
	}
}

void RPCManager::handleRemoteRequest(Buffer<uint8> buffer,
                                     RemoteFinishHandlerPtr handler,
                                     uint8 binaryFormatVersion)
{
	if(buffer.empty())
		return;

	uint8 serializerVersion = BinaryBufferSerializer::getSerializerFormatVersion();

	if (binaryFormatVersion == serializerVersion) {
		handleRemoteRequest(buffer, handler);
	} else if (binaryFormatVersion == 0) {
		BinaryBufferDeserializer ds(&buffer);	
		uint8 marker;
		ds.deserialize(marker, false);

		if(marker==BinaryRPCBackendLegacy::BINARY_RPC_MARKER) {
			handleRequest<BinaryRPCBackendLegacy>(std::move(buffer), handler, true);
		} else {
			handleRequest<JSONRPCBackend>(std::move(RPCBackendAdapter<JSONRPCBackend>::fromBuffer(buffer)),
			                              handler, true);
		}
	} else {
		MIRA_THROW(XIO, "handleRemoteRequest called with binary format version " << (int)binaryFormatVersion
		      	         << ". Only implemented for versions 0, " << (int)serializerVersion << ".");
	}
}

///////////////////////////////////////////////////////////////////////////////

void RPCManager::handleRemoteResponse(Buffer<uint8> buffer)
{
	BinaryBufferDeserializer ds(&buffer);	
	uint8 marker;
	ds.deserialize(marker, false);

	assert(!buffer.empty());
	if (marker == BinaryRPCBackend::BINARY_RPC_MARKER) {
		BinaryRPCBackend::ClientResponse response(&buffer);
		mClient.handleResponse<BinaryRPCBackend>(response);
	} else {
		JSONRPCBackend::ContainerType value = RPCBackendAdapter<JSONRPCBackend>::fromBuffer(buffer);
		JSONRPCBackend::ClientResponse response(&value);
		mClient.handleResponse<JSONRPCBackend>(response);
	}
}

void RPCManager::handleRemoteResponse(Buffer<uint8> buffer, uint8 binaryFormatVersion)
{
	uint8 serializerVersion = BinaryBufferSerializer::getSerializerFormatVersion();

	if (binaryFormatVersion == serializerVersion) {
		handleRemoteResponse(buffer);
	} else if (binaryFormatVersion == 0) {
		BinaryBufferDeserializer ds(&buffer);	
		uint8 marker;
		ds.deserialize(marker, false);

		assert(!buffer.empty());
		if (marker == BinaryRPCBackendLegacy::BINARY_RPC_MARKER) {
			BinaryRPCBackendLegacy::ClientResponse response(&buffer);
			mClient.handleResponse<BinaryRPCBackendLegacy>(response);
		} else {
			JSONRPCBackend::ContainerType value = RPCBackendAdapter<JSONRPCBackend>::fromBuffer(buffer);
			JSONRPCBackend::ClientResponse response(&value);
			mClient.handleResponse<JSONRPCBackend>(response);
		}
	} else {
		MIRA_THROW(XIO, "handleRemoteResponse called with binary format version " << (int)binaryFormatVersion
		                 << ". Only implemented for versions 0, " << (int)serializerVersion << ".");
	}
}

void RPCManager::onRPCfinished(AbstractDeferredInvoker* invoker)
{
	boost::mutex::scoped_lock lock(mPendingRequestsMutex);

	auto it = mPendingRequests.find(invoker);
	if(it==mPendingRequests.end()) {
		MIRA_LOG(ERROR) << "Got RPC finished callback for unknown call invoker " << invoker;
		return;
	}

	AbstractRequestContextPtr context = it->second;
	mPendingRequests.erase(it);

	lock.unlock(); // unlock before (potentially lengthy) onFinished

	// delegate to corresponding request context
	context->onFinished(mClient);
}

///////////////////////////////////////////////////////////////////////////////

RPCFuture<JSONRPCResponse> RPCManager::callJSON(const std::string& jsonCall)
{
	json::Value request;
	json::read(jsonCall, request);
	return callJSON(request);
}

RPCFuture<JSONRPCResponse> RPCManager::callJSON(const json::Value& jsonCall)
{
	// get service and call id from call
	JSONRPCBackend::ServerRequest backend(&jsonCall);
	std::string callID, service;
	backend.getHeader(callID, service);

	return callJSONIntern(service, callID, jsonCall);
}

RPCFuture<JSONRPCResponse> RPCManager::callJSON(const std::string& service,
                                                const std::string& method,
                                                const json::Value& params)
{
	if (params.type() != json_spirit::null_type &&
		params.type() != json_spirit::array_type)
		MIRA_THROW(XInvalidParameter, "The parameters object for calling service '"
		           << service << "' must be a json array.");

	// build a JSON RPC request according to the JSON RPC 2.0 specification:
	//   http://jsonrpc.org/specification
	static uint32 sID = 0;
	json::Value request = json::Object();
	request.get_obj()["jsonrpc"] = json::Value("2.0");
	request.get_obj()["id"] = json::Value(std::string("JSONRM") + toString(atomic::inc(&sID)));
	request.get_obj()["method"] = json::Value(service + "." + method);
	// add params, if some were specified
	if (params.type() != json_spirit::null_type)
		request.get_obj()["params"] = params;
	return callJSONIntern(service, json::write(request.get_obj()["id"]), request);
}

RPCFuture<JSONRPCResponse> RPCManager::callJSON(const std::string& service,
                                                const std::string& method,
                                                const std::string& params)
{
	// wrap comma separated parameters into a JSON array ...
	json::Value pv;
	if (!params.empty())
	{
		std::string paramStr = MakeString() << "[" << params << "]";
		json::read(paramStr, pv);
	}
	// ... and call the above method
	return callJSON(service, method, pv);
}

RPCFuture<JSONRPCResponse> RPCManager::callJSON(const RPCCallDefinition& rpc)
{
	return callJSON(rpc.service, rpc.method, rpc.params);
}

RPCFuture<JSONRPCResponse> RPCManager::callJSONIntern(const std::string& service,
                                                      const std::string& callID,
                                                      const json::Value& request)
{

	RPCFuture<JSONRPCResponse> future = mClient.addPendingResponse<JSONRPCBackend, JSONRPCResponse>(callID);

	boost::mutex::scoped_lock lock(mServiceManagementMutex);
	if(mLocalServices.count(service)>0) {
		lock.unlock();
		// we have the requested service locally here, so perform a local
		// call and queue the request immediately
		handleLocalRequest<JSONRPCBackend>(request, service);
	} else {
		// must be a remote service, so find it
		auto it = mRemoteServices.find(service);
		if(it==mRemoteServices.end())
			MIRA_THROW(XRPC, "The requested service '" << service << "' is not known.");

		lock.unlock();

		std::string jsonCall = json::write(request, false, 0);
		// send the request through the RemoteRequestHandler to remote side
		Buffer<uint8> buffer((uint8*)jsonCall.data(), jsonCall.size());
		it->second.handler->onRPCrequested(Buffer<uint8>(buffer)); // need a copy here that can be moved away
	}

	return future;
}

///////////////////////////////////////////////////////////////////////////////

}

