/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ReflectCollection.h
 *    ReflectCollectionCount and ReflectCollectionItems to be
 *    specialized for certain Reflectors.
 *
 * @author Erik Einhorn
 * @date   2010/09/28
 */

#ifndef _MIRA_REFLECTCOLLECTION_H_
#define _MIRA_REFLECTCOLLECTION_H_

#include <serialization/ReflectorInterface.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace serialization {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Can be specialized for a concrete derived RecursiveMemberReflector to
 * reflect the size of collections like vectors, arrays, lists, maps, etc.
 * This default implementation reflects the number of items as "@itemcount".
 */
template<typename Reflector, typename Collection>
struct ReflectCollectionCount
{
	static void reflect(Reflector& r, uint32& ioCount)
	{
		r.member("@itemcount", ioCount, "");
	}
};

/**
 * @ingroup SerializationModule
 *
 * Can be specialized for a concrete derived RecursiveMemberReflector to
 * reflect the items of collections like vectors, arrays, lists, maps, etc.
 * This default implementation reflects the items with name "item" and  id "item[n]".
 */
template<typename Reflector, typename Container>
struct ReflectCollectionItems
{
	typedef typename Container::value_type type;

	static void reflect(Reflector& r, Container& c)
	{
		// store each item
#if __GNUC_PREREQ(4,5)
		using std::begin;
		using std::end;
		auto it = begin(c);
		auto endit = end(c);
#else
		auto it = c.begin();
		auto endit = c.end();
#endif

		for (int id=0; it != endit; ++it, ++id)
		{
			// specifies the id string of the element, since the
			// name "item" is the same for all elements
			MIRA_PROPERTY_WITH_ID(r, "item", "item["+toString(id)+"]", *it, "");
		}
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
