/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MetaSerializer.h
 *    Serializer for creating meta information out of types
 *    using serialization.
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2011/06/18
 */

#ifndef _MIRA_METASERIALIZER_H_
#define _MIRA_METASERIALIZER_H_

#include <iostream>

#include <type_traits>

#include <platform/Platform.h>
#include <utils/MakeString.h>
#include <utils/IsDefaultConstructible.h>

#include <serialization/BinarySerializer.h>
#include <serialization/IsNotMetaSerializable.h>
#include <serialization/adapters/boost/optional.hpp>
#include <serialization/adapters/std/list>
#include <serialization/adapters/std/vector>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace serialization {

///////////////////////////////////////////////////////////////////////////////

// with c++20 this will be part of the language in std
template <typename T>
struct remove_cvref
{
	typedef typename std::remove_cv<typename std::remove_reference<T>::type>::type type;
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

///@endcond

/**
 * Meta Type information. This includes the datatype itself (int, float, ...
 * string or user-defined class), in the case of a class, the class identifier,
 * and optional qualifiers that specify if the type is a pointer, a collection,
 * etc.
 *
 * The TypeMeta is realized as vector of flags. The first item in the vector
 * indicates the data type. This first type item can be followed by multiple
 * qualifier items.
 */
class MIRA_BASE_EXPORT TypeMeta : public std::vector<uint8>
{
	typedef std::vector<uint8> Base;
public:

	/**
	 * The different types (ranging from fundamental types to user-defined class).
	 * Only one such type flag is allowed per TypeMeta.
	 */
	enum Type {
		TYPE_INVALID = 0,
		TYPE_VOID,

		// class type, (typename is stored in mIdentifier)
		TYPE_CLASS,

		// user-defined atomic type (typename is stored in mIdentifier)
		TYPE_ATOMIC,

		// built-in atomic types (mIdentifier stays empty)
		TYPE_UINT8 , TYPE_INT8,
		TYPE_UINT16, TYPE_INT16,
		TYPE_UINT32, TYPE_INT32,
		TYPE_UINT64, TYPE_INT64,
		TYPE_FLOAT , TYPE_DOUBLE,
		TYPE_BOOL,
		TYPE_STRING,
		TYPE_ENUMERATION,
		TYPE_VERSION,

		LAST_TYPE_FIELD
	};

	/**
	 * Additional qualifiers.
	 *
	 */
	enum TypeQualifier {
		NO_QUALIFIER=0,
		// first value must be larger than last value in TypeField
		FIRST_TYPE_QUALIFIER = 0x20,
		TYPE_COLLECTION = FIRST_TYPE_QUALIFIER,
		TYPE_POINTER,
		TYPE_PROPERTY
	};

	// this version is the format version of the TypeMeta
	// only deviate from default if you really know what you are doing!
	TypeMeta(uint8 version = 2) : mVersion(version) {}

	template<typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r,Base);
		if(getType()==TYPE_ATOMIC || getType()==TYPE_CLASS)
			r.member("Identifier", mIdentifier, "The identifier of the class "
			         "or 'complex' atomic type");
		r.member("Version", mVersion, "The metatype format version", 2);
	}

	template<typename BinaryStream>
	void reflect(ConcreteBinaryDeserializer<BinaryStream, 0>& r) {
		MIRA_REFLECT_BASE(r,Base);
		if(getType()==TYPE_ATOMIC || getType()==TYPE_CLASS)
			r.member("Identifier", mPlainIdentifier, "The identifier of the class "
			         "or 'complex' atomic type", REFLECT_CTRLFLAG_TEMP_TRACKING);
		mVersion = 0;
		if (getType()==TYPE_CLASS)
			mIdentifier += mPlainIdentifier + " @v0";
		else
			mIdentifier = mPlainIdentifier;	
	}

	// just in case someone tries to use it...
	template<typename BinaryStream>
	void reflect(ConcreteBinarySerializer<BinaryStream, 0>& r) {
		MIRA_REFLECT_BASE(r,Base);
		if(getType()==TYPE_ATOMIC || getType()==TYPE_CLASS)
			r.member("Identifier", mPlainIdentifier, "The identifier of the class "
			         "or 'complex' atomic type", REFLECT_CTRLFLAG_TEMP_TRACKING);
	}

	/**
	 * Adds a type qualifier. Note, that the data type must have been
	 * set already.
	 */
	void addQualifier(TypeQualifier qualifier) {
		assert(Base::size()>=1);
		push_back(qualifier);
	}

	/**
	 * Sets the datatype to one of the built-in atomic types (TYPE_INT,FLOAT, etc).
	 */
	void setBaseAtomicType(Type type) {
		assert(type!=TYPE_CLASS && type!=TYPE_ATOMIC && type!=TYPE_INVALID);
		assert(Base::size()==0);
		push_back(type);
	}

	/**
	 * Sets an atomic type with the specified typename.
	 * This can be a built-in atomic type like int, float, etc. or a
	 * user defined atomic type.
	 */
	void setAtomicType(const std::string& typenam);

	/**
	 * Same as above, but typename is extracted from T.
	 */
	template <typename T>
	void setAtomicType() { setAtomicType(typeName<T>()); }

	/**
	 * Sets the data type to TYPE_CLASS and sets the class identifier.
	 */
	void setClassType(const std::string& identifier) {
		assert(Base::size()==0);
		push_back(TYPE_CLASS);
		mIdentifier = identifier;
		mPlainIdentifier = mIdentifier;
	}

	/**
	 * Sets the datatype.
	 * This automatically differentiates between atomic types and class types.
	 */
	template <typename T>
	void setType() {
		if(IsAtomicSerializable<T>::value)
			setAtomicType<T>();
		else
			setClassType(typeName<T>());
	}

	/**
	 * Returns the type flag which is the first item in the vector.
	 */
	Type getType() const {
		if(empty() || front() >= LAST_TYPE_FIELD)
			return TYPE_INVALID;
		return (Type)front();
	}

	/**
	 * Returns true, if the type meta contains qualifiers.
	 */
	bool hasQualifier() const {
		return Base::size()>1;
	}

	/**
	 * Returns the number of qualifiers.
	 */
	int qualifierCount() const {
		return std::max(size()-1,0);
	}

	/**
	 * Returns the i-th type qualifier, or NO_QUALIFIER if there is no such
	 * qualifier.
	 */
	TypeQualifier getQualifier(int i=0) const {
		assert(i>=0);
		if(i>=qualifierCount())
			return NO_QUALIFIER;
		return (TypeQualifier)Base::at(size()-i-1);
	}

	/**
	 * Removes the n last qualifiers.
	 * If n is larger than the number of qualifiers, all qualifiers are removed.
	 */
	void removeQualifiers(int n)
	{
		assert(n>=0);
		if(n>qualifierCount())
			n=qualifierCount();
		return Base::resize(size()-n);
	}

	/**
	 * Returns the identifier of the user-defined atomic type or class type.
	 * For built-in atomic types (int,float) this returns an empty string.
	 */
	const std::string& getIdentifier() const {
		return mIdentifier;
	}

	/**
	 * Returns the typename. For user-defined atomic types and class types this
	 * is the previously set identifier. For built-in types this is the typename.
	 */
	std::string getTypename() const;

	/**
	 * Converts the whole type information including qualifiers to a
	 * human-readable string.
	 */
	std::string toString() const;

	/// Returns the number of items (the type item + qualifier items)
	int size() const { return (int)Base::size(); }

	/// Returns the metatype format version
	uint8 version() const { return mVersion; }

private:
	std::string mIdentifier;
	std::string mPlainIdentifier;
	uint8 mVersion;
};

typedef boost::shared_ptr<TypeMeta> TypeMetaPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Meta information for RPC methods.
 */
struct MethodMeta
{
	struct Parameter
	{
		Parameter() {}
		Parameter(TypeMetaPtr iType) :
			type(iType) {}
		Parameter(TypeMetaPtr iType,
		          const std::string& iName,
		          const std::string& iDescription) :
			type(iType),
			name(iName),
			description(iDescription) {}

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("Type",type,"The meta type info of the parameter");
			r.member("Name",name,"The name of the parameter");
			r.member("Description",description,"The description for the parameter");
		}

		/// The meta type info of the return parameter.
		TypeMetaPtr type;
		/// The name of the parameter.
		std::string name;
		/// The description for the parameter.
		std::string description;
	};

	MethodMeta() {}
	MethodMeta(const std::string& iName,
	           const std::string& iComment) :
		name(iName),
		comment(iComment) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Name",name,"The name of the method");
		r.member("Comment",comment,"The user defined comment");
		r.member("ReturnType",returnType,"The meta type info of the return type");
		r.member("Parameters",parameters,"A list with type infos for each parameter");
	}

	/// The name of the method.
	std::string name;
	/// The user defined comment as specified in the r.method() call.
	std::string comment;
	/// The meta type info of the return type.
	TypeMetaPtr returnType;
	/// A list with info for each parameter.
	std::list<Parameter> parameters;
};

typedef boost::shared_ptr<MethodMeta> MethodMetaPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Meta information for complex compounds, like classes and structs.
 * Contains information on version, members, RPC interfaces and RPC methods.
 */
class MIRA_BASE_EXPORT CompoundMeta
{
public:
	/// A single member of the compound
	struct Member
	{
		Member() {}
		Member(TypeMetaPtr iType, const std::string& iName, const std::string& iComment) :
			type(iType), name(iName), comment(iComment) {}

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("Type", type, "The type of the member");
			r.member("Name", name, "The name of the member");
			r.member("Comment", comment, "The user defined comment");
		}

		/// The meta type info of the member
		TypeMetaPtr type;
		/// The name of the member
		std::string name;
		/// The user defined comment as specified in the r.member() call
		std::string comment;
	};

	typedef serialization::VersionType VersionType;

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		VersionType v = r.requireVersion(2, 1, this);

		if (v < 2) {
			boost::optional<int> optVersion; // version was int32 back then
			r.member("Version",optVersion,"The version of the serialization format");

			if (optVersion)
				version[""] = *optVersion;
		} else
			r.member("Version",version,"The version of the serialization format");

		r.member("Members",members,"The members of the compound");
		r.member("Interfaces",interfaces,"The supported interfaces");
		r.member("Methods",methods,"The reflected methods");
	}

	template<typename BinaryStream>
	void reflect(ConcreteBinarySerializer<BinaryStream, 0>& r) {
		r.version(1, this);                        // version 2 of CompoundMeta was unknown back then

		boost::optional<int> optVersion;           // serialization version type was int32 back then
		if (!version.empty())
			optVersion = version.cbegin()->second; // we assume the version of the serialized class did not change
			                                       // from v0, otherwiese it will be unreadable for the recipient anyway
			                                       // --> it can have only one or no version at all now
		r.member("Version",optVersion,"The version of the serialization format");

		r.member("Members",members,"The members of the compound");
		r.member("Interfaces",interfaces,"The supported interfaces");
		r.member("Methods",methods,"The reflected methods");
	}

	void addMember(const ReflectMemberMeta& memberMeta, TypeMetaPtr type)
	{
		addMember(memberMeta.id, memberMeta.comment, type);
	}

	void addMember(const std::string& name, TypeMetaPtr type)
	{
		addMember(name, "", type);
	}

	void addMember(const std::string& name, const std::string& comment, TypeMetaPtr type)
	{
		members.push_back(Member(type, name, comment));
	}

	void insertMember(const std::string& name, TypeMetaPtr type, int index)
	{
		std::list<Member>::iterator pos = members.begin();
		while ((pos != members.end()) && (index-- > 0))
			++pos;
		members.insert(pos, Member(type, name, ""));
	}

	void insertMember(const std::string& name, TypeMetaPtr type, const std::string& before)
	{
		std::list<Member>::iterator pos = members.begin();
		while ((pos != members.end()) && (pos->name != before))
			++pos;
		members.insert(pos, Member(type, name, ""));
	}

	void addInterface(const std::string& iface)
	{
		interfaces.push_back(iface);
	}

	void addMethod(const MethodMetaPtr& method)
	{
		methods.push_back(method);
	}

	void setVersion(VersionType version) {
		this->version[""] = version;
	}

	void setVersion(const std::string& type, VersionType version) {
		this->version[type] = version;
	}

	std::string toString() const;

	/// The version(s) of the described type's serialization format.
	std::map<std::string, VersionType> version;
	/// The members of the compound
	std::list<Member>  members;
	/// The reflected interfaces
	std::list<std::string> interfaces;
	/// The reflected methods (RPC methods)
	std::list<MethodMetaPtr> methods;
};

typedef boost::shared_ptr<CompoundMeta> CompoundMetaPtr;

/**
 * Database that stores all meta type information and provides
 * additional functions for accessing the database
 */
class MIRA_BASE_EXPORT MetaTypeDatabase : public std::map<std::string, CompoundMetaPtr>
{
	typedef std::map<std::string, CompoundMetaPtr> Base;
public:

	MetaTypeDatabase() {}
	MetaTypeDatabase(const MetaTypeDatabase& other) :
		Base(other) {}
	MetaTypeDatabase& operator=(const MetaTypeDatabase& other)
	{
		Base::operator=(other);
		return *this;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r,Base);
	}

	template<typename BinaryStream>
	void reflect(ConcreteBinaryDeserializer<BinaryStream, 0>& r)
	{
		MIRA_REFLECT_BASE(r,Base);

		// mark all loaded type keys as '@v0' (TypeMetas in entries are also marked by TypeMeta::reflect())
		MetaTypeDatabase db;
		foreach (auto p, *this)
			db[p.first+" @v0"] = p.second;

		swap(db);
	}

	// just in case...
	template<typename BinaryStream>
	void reflect(ConcreteBinarySerializer<BinaryStream, 0>& r)
	{
		MetaTypeDatabase db;
		foreach (auto p, *this) {
			// if tagged ' @v0', remove tag, else ignore
			size_t s = p.first.size();
			if (p.first.substr(s-4, std::string::npos) == " @v0")
				db[p.first.substr(0, s-4)] = p.second;	
		}

		r.template reflectBase<Base>(db);
	}

	/**
	 * Merge meta information from other database into THIS
	 */
	void merge(const MetaTypeDatabase& other)
	{
		this->insert(other.begin(), other.end());
	}

	/**
	 * Generates a database which is a subset of THIS database and contains
	 * all types that are necessary to fully describe the specified type.
	 */
	MetaTypeDatabase getDependentTypesDB(TypeMetaPtr type)
	{
		MetaTypeDatabase db;
		generateDependentTypesDB(type, db);
		return db;
	}

private:

	void generateDependentTypesDB(TypeMetaPtr type, MetaTypeDatabase& db);
};

template <>
class IsCollection<MetaTypeDatabase> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

class MIRA_BASE_EXPORT MetaSerializer : public BinarySerializer<MetaSerializer>
{
	typedef BinarySerializer<MetaSerializer> Base;

public:
	typedef boost::mpl::bool_<false> isObjectTrackingSupported;

	typedef boost::mpl::bool_<true> requireReflectBarriers;
	typedef CompoundMeta::Member* ReflectState;

public:

	/**
	 * The meta serializer will work on the database and adds new types there.
	 */
	MetaSerializer(MetaTypeDatabase& database) :
		mMetaDatabase(database) {}

	template<typename T>
	TypeMetaPtr addMeta(T& v)
	{
#ifdef CHECK_FORCE_SERIALIZE_BINARY_VERSION
		int vf = Serializer::forcedSerializeVersion();
		if ((vf >= 0) && (vf < BinaryBufferSerializer::getSerializerFormatVersion()))
			return TypeMetaPtr(); //disable meta serialization if an older binary format is forced
#endif
		try
		{
			mCurrentMeta.reset();
			mParentMeta.reset();
			serialize("top", v);
			return mCurrentMeta;
		}
		catch(XNotImplemented&)
		{
			return TypeMetaPtr(); //return a null ptr as we are not able to extract meta information
		}
	}

	typedef serialization::VersionType VersionType;

private:

	VersionType version(VersionType version, const std::string& type,
	                    const std::string& versionString) {
		if(mParentMeta)
			mParentMeta->setVersion(type, version);

		mCurrentVersion->name = versionString;
		mCurrentVersion->comment = "";

		return version;
	}

protected:

	VersionType version(VersionType version, const std::string& type) {

		return this->version(version, type, "@version[" + type + "]");
	}

public:

	template <typename T>
	VersionType version(VersionType version, const T* caller = NULL) {
		static const std::string t = typeName<T>();
		static const std::string v = "@version[" + t + "]";
		return this->version(version, t, v);
	}

	MIRA_DEPRECATED("Please call as version<MyType>(v) or version(v, this)",
	VersionType version(VersionType version)) {
		if(mParentMeta)
			mParentMeta->setVersion(version);

		mCurrentVersion->name = std::string("@version");

		return version;
	}

	ReflectState preReflect(const char* context = "")
	{
		ReflectState state = mCurrentVersion;

		mCurrentMeta.reset(new TypeMeta);
		mCurrentMeta->setBaseAtomicType(TypeMeta::TYPE_VERSION);
		if(mParentMeta) {
			mParentMeta->addMember("@version_default", std::string("implicit: ") + context, mCurrentMeta);
			mCurrentVersion = &(mParentMeta->members.back());
		}

		return state;
	}

	void postReflect(const ReflectState& prev)
	{
		mCurrentVersion = prev;
	}

	template <typename T>
	void write(const T* data, std::size_t count) {

	}

	/**
	 * Returns true, of there is a codec for the specified type T.
	 * In this case, the encode method can be used to encode the data using
	 * the codec.
	 */
	template <typename T>
	bool hasCodec() const {
		return false;
	}

	/**
	 * Encodes the specified object containing the data using a matching
	 * codec. The encoded data will be written directly into the binary output.
	 * If no codec was found, false is returned, and the caller must serialize
	 * the data manually without codec (in this case "NULL" is written as
	 * codec fourcc into the binary stream).
	 */
	template <typename T>
	bool codec(const T& obj)
	{
		return false;
	}

	void interface(const char* name)
	{
		assert(name != NULL); // check if user has specified NULL as interface name
		if(mParentMeta)
			mParentMeta->addInterface(name);
	}

	void addMethod(MethodMetaPtr method)
	{
		assert(method);
		if (mParentMeta)
			mParentMeta->addMethod(method);
	}

	/*
	The following code uses boost preprocessor for generating the method(name, P1 ... Pn)
	methods. Read documentation of boost preprocessor or use Eclipse to expand
	the BOOST_PP_REPEAT macro below in order to see the code that is generated.
	*/

	// method(name, function<...>, comment)
	#define META_GEN_METHODS(z,n,_)                                                                               \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                          \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), const char* comment)                    \
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment);                                \
	}                                                                                                             \
	                                                                                                              \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                          \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class* This, const char* comment)\
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment);                                \
	}                                                                                                             \
	                                                                                                              \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                          \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class* This, const char* comment) \
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment);                                \
	}                                                                                                             \
	                                                                                                              \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                          \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))> fn, const char* comment)     \
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment);                                \
	}

	BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),META_GEN_METHODS,nil)
	#undef META_GEN_METHODS

	#define META_GEN_METHODS_NAME_DESC(z,n,_) name##n, description##n

	// method(name, function<...>, comment, paramname, paramdescription, ... )
	#define META_GEN_METHODS_PARAMDESC(z,n,_)                                                                     \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                          \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)),                                         \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil))                             \
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment,                                 \
		                                                           BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil)); \
	}                                                                                                             \
	                                                                                                              \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                          \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class* This,                     \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil))                             \
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment,                                 \
		                                                           BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil)); \
	}                                                                                                             \
	                                                                                                              \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                          \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class* This,               \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil))                             \
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment,                                 \
		                                                           BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil)); \
	}                                                                                                             \
	                                                                                                              \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>                                          \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))> fn,                          \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_DECL,nil))                             \
	{                                                                                                             \
		createMethodMeta<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>(name, comment,                                 \
		                                                           BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil)); \
	}

	BOOST_PP_REPEAT_FROM_TO(1,BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),META_GEN_METHODS_PARAMDESC,nil)
	#undef META_GEN_METHODS_PARAMDESC

	// method(name, function<...>, comment, paramname, paramdescription, paramsamplevalue, ... )
	// MetaSerializer ignores the sample params
	#define META_GEN_METHODS_PARAMDESCSAMPLE(z,n,_)                                                               \
	template<typename R, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>                                    \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)),                                         \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil))                      \
	{                                                                                                             \
		method(name, fn, comment, BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil));                               \
	}                                                                                                             \
	                                                                                                              \
	template<typename R, typename Class, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>                    \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class* This,                     \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil))                      \
	{                                                                                                             \
		method(name, fn, This, comment, BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil));                         \
	}                                                                                                             \
	                                                                                                              \
	template<typename R, typename Class, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>                    \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class* This,               \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil))                      \
	{                                                                                                             \
		method(name, fn, This, comment, BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil));                         \
	}                                                                                                             \
	                                                                                                              \
	template<typename R, BOOST_PP_ENUM(n,RPCGEN_CALL_TYPENAME_PARAM_DECL,nil)>                                    \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))> fn,                          \
	            const char* comment, BOOST_PP_ENUM(n,RPCGEN_CALL_NAME_DESC_SAMPLE_DECL,nil))                      \
	{                                                                                                             \
		method(name, fn, comment, BOOST_PP_ENUM(n,META_GEN_METHODS_NAME_DESC,nil));                               \
	}

	BOOST_PP_REPEAT_FROM_TO(1,BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),META_GEN_METHODS_PARAMDESCSAMPLE,nil)
	#undef META_GEN_METHODS_PARAMDESCSAMPLE
	#undef META_GEN_METHODS_NAME_DESC

	// method(name, function<...>, comment, ... )
	// (these should match anything that does not match the more specific definitions above (i.e. it is invalid),
	//  will provide a much more informational error message than just failing to find a matching template instantiation)
	#define META_GEN_METHODS_WRONG_ARGUMENT_NUMBER(z,n,_)                                                         \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args>                        \
	void method(const char* name, R (*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)),                                         \
	            const char* comment, Args...)                                                                     \
	{                                                                                                             \
		invalid_method<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>();                                               \
	}                                                                                                             \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args>        \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)), Class* This,                     \
	            const char* comment, Args...)                                                                     \
	{                                                                                                             \
		invalid_method<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>();                                               \
	}                                                                                                             \
	template<typename R, typename Class BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args>        \
	void method(const char* name, R (Class::*fn)(BOOST_PP_ENUM_PARAMS_Z(z,n,P)) const, Class* This,               \
	            const char* comment, Args...)                                                                     \
	{                                                                                                             \
		invalid_method<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>();                                               \
	}                                                                                                             \
	template<typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P), typename... Args>                        \
	void method(const char* name, boost::function<R (BOOST_PP_ENUM_PARAMS_Z(z,n,P))> fn,                          \
	            const char* comment, Args...)                                                                     \
	{                                                                                                             \
		invalid_method<R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,P)>();                                               \
	}

	BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),META_GEN_METHODS_WRONG_ARGUMENT_NUMBER,nil)
	#undef META_GEN_METHODS_WRONG_ARGUMENT_NUMBER

	template<typename T>
	void atomic(T& )
	{
		if (IsNotMetaSerializable<T>::value)
			MIRA_THROW(XNotImplemented, "Type " << typeName<T>() << " is not meta-serializable");

		mCurrentMeta.reset(new TypeMeta);
		mCurrentMeta->setAtomicType<T>();
		addCurrentAsMemberToParent();
	}

	template<typename T>
	void enumeration(T& )
	{
		if (IsNotMetaSerializable<T>::value)
			MIRA_THROW(XNotImplemented, "Type " << typeName<T>() << " is not meta-serializable");

		mCurrentMeta.reset(new TypeMeta);
		mCurrentMeta->setBaseAtomicType(TypeMeta::TYPE_ENUMERATION);
		addCurrentAsMemberToParent();
	}

	template<typename T>
	void pointer(T* &pointer)
	{
		if (IsNotMetaSerializable<T>::value)
			MIRA_THROW(XNotImplemented, "Type " << typeName<T>() << " is not meta-serializable");

		if(pointer!=NULL)
			Base::pointer(pointer);
		else
			TypeWithoutObjectAbstractHelper<T,std::is_abstract<T>::value>::invoke(this);

		assert(mCurrentMeta);
		mCurrentMeta->addQualifier(TypeMeta::TYPE_POINTER);
	}

	template<typename T>
	void object(T& member)
	{
		if (IsNotMetaSerializable<T>::value)
			MIRA_THROW(XNotImplemented, "Type " << typeName<T>() << " is not meta-serializable");

		mCurrentMeta.reset(new TypeMeta);
		mCurrentMeta->setClassType(typeName<T>());
		addCurrentAsMemberToParent();

		// add us to the global type map
		CompoundMetaPtr compound = createMapEntry(mCurrentMeta);
		if(compound)
		{
			// push on "stack" and create new meta object for first member
			TypeMetaPtr prevCurrentMeta = mCurrentMeta;
			CompoundMetaPtr prevParentMeta = mParentMeta;
			mParentMeta = compound;

			Base::object(member);

			// go "stack" back upwards
			mCurrentMeta = prevCurrentMeta;
			mParentMeta =  prevParentMeta;
		}
		// else we were already reflected, so skip this object
	}

	template<typename T>
	void invokeOverwrite(T& object)
	{

		static const std::string context = "invokeOverwrite " + typeName<T>();
		ReflectState state = preReflect(context.c_str());

		Base::invokeOverwrite(object);

		postReflect(state);
	}

	/**
	 * Specialized for PlainArray, because the BinarySerializer does it...
	 */
	template<typename T>
	void invokeOverwrite(serialization::PlainArray<T>& array)
	{
		static const std::string context = "invokeOverwrite PlainArray<" + typeName<T>() +">";
		ReflectState prevState = preReflect(context.c_str());

		if(this->template isTrackingEnabled<T>() || !IsBitwiseSerializable<T>::value) {
			Base::invokeOverwrite(array);
		} else {
			collectionItems<T>("@itemcount", true);
		}

		postReflect(prevState);
	}


protected:

	CompoundMetaPtr createMapEntry(TypeMetaPtr& type)
	{
		assert(type->getType() == TypeMeta::TYPE_CLASS);
		const std::string& n = type->getIdentifier();

		if(mMetaDatabase.count(n)>0)
			return CompoundMetaPtr();

		CompoundMetaPtr compound(new CompoundMeta);
		mMetaDatabase[n] = compound;
		return compound;
	}

	void addCurrentAsMemberToParent()
	{
		if(mParentMeta)
			mParentMeta->addMember(getCurrentMemberMeta(), mCurrentMeta);
	}

	template <typename T, bool> friend struct TypeWithoutObjectHelper;
	template<typename T, bool>
	struct TypeWithoutObjectHelper {
		static void invoke(MetaSerializer* metaserializer) {
			metaserializer->mCurrentMeta.reset(new TypeMeta);
			metaserializer->mCurrentMeta->setType<T>();
			metaserializer->addCurrentAsMemberToParent();
		}
	};

	template<typename T>
	struct TypeWithoutObjectHelper<T,true> {
		static void invoke(MetaSerializer* metaserializer) {
			T element;
			metaserializer->delegate(element);
		}
	};

	template<typename T>
	struct TypeWithoutObjectHelper<T*,true> {
		static void invoke(MetaSerializer* metaserializer) {
			T* element = NULL;
			metaserializer->delegate(element);
		}
	};

	template <typename T, bool> friend struct TypeWithoutObjectAbstractHelper;
	template<typename T, bool>
	struct TypeWithoutObjectAbstractHelper {
		static void invoke(MetaSerializer* metaserializer) {
			TypeWithoutObjectHelper<T,IsDefaultConstructible<T>::value>::invoke(metaserializer);
		}
	};

	template<typename T>
	struct TypeWithoutObjectAbstractHelper<T,true> {
		static void invoke(MetaSerializer* metaserializer) {
			metaserializer->mCurrentMeta.reset(new TypeMeta);
			metaserializer->mCurrentMeta->setType<T>();
			metaserializer->addCurrentAsMemberToParent();
		}
	};

	template<typename T>
	struct TypeWithoutObjectAbstractHelper<T*,true> {
		static void invoke(MetaSerializer* metaserializer) {
			metaserializer->mCurrentMeta.reset(new TypeMeta);
			metaserializer->mCurrentMeta->setType<T>();
			metaserializer->addCurrentAsMemberToParent();
		}
	};

	template <typename T>
	TypeMetaPtr createMeta()
	{
		// push on "stack" and create new meta object for first member
		TypeMetaPtr prevCurrentMeta = mCurrentMeta;
		CompoundMetaPtr prevParentMeta = mParentMeta;
		mParentMeta.reset();
		if(IsAtomicSerializable<T>::value)
		{
			T* dummy = NULL;
			delegate(*dummy); // is safe, even if element==NULL since atomic never accesses it here
		}
		else
			TypeWithoutObjectAbstractHelper<T,std::is_abstract<T>::value>::invoke(this);

		TypeMetaPtr meta = mCurrentMeta;
		// go "stack" back upwards
		mCurrentMeta = prevCurrentMeta;
		mParentMeta =  prevParentMeta;
		return meta;
	}

	// call this when T could be void, will call the respective overload
	template<typename T>
	TypeMetaPtr createMetaOrVoidHelper()
	{
		typedef typename serialization::remove_cvref<T>::type Type;
		Type* dummy; // just to select the required createMetaHelper() overload
		return createMetaHelper(dummy);
	}

	// can directly call this when not necessary to check for void
	template<typename T>
	TypeMetaPtr createMetaHelper()
	{
		typedef typename serialization::remove_cvref<T>::type Type;
		return createMeta<Type>();
	}

	// different overloads for void/non-void type (can't use void as type here -> use T*/void*)

	template<typename T>
	TypeMetaPtr createMetaHelper(T*)
	{
		return createMeta<T>();
	}

	TypeMetaPtr createMetaHelper(void*)
	{
		TypeMetaPtr meta(new TypeMeta);
		meta->setBaseAtomicType(TypeMeta::TYPE_VOID);
		return meta;
	}

	#define META_MAKE_METHODMETA_PUSH_PARAMETER(z, n, _)                           \
		m->parameters.push_back(MethodMeta::Parameter(createMetaHelper<P##n>()));

	#define META_MAKE_METHODMETA(z,n,_)                                            \
	template <typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>          \
	void createMethodMeta(const std::string& name, const std::string& comment) {   \
		MethodMetaPtr m(new MethodMeta(name, comment));                            \
		m->returnType = createMetaOrVoidHelper<R>();                               \
		BOOST_PP_REPEAT_ ## z(n, META_MAKE_METHODMETA_PUSH_PARAMETER, nil)         \
		addMethod(m);                                                              \
	}

	BOOST_PP_REPEAT(RPC_METHODS_MAX_PARAMS,META_MAKE_METHODMETA,nil)
	#undef META_MAKE_METHODMETA_PUSH_PARAMETER
	#undef META_MAKE_METHODMETA

	#define META_MAKE_METHODMETA_PUSH_PARAMETERDESC(z, n, _)                       \
		m->parameters.push_back(                                                   \
			MethodMeta::Parameter(createMetaHelper<P##n>(),                        \
			                      name##n, description##n));

	#define META_MAKE_METHODMETA_CONSTSTRINGREF_NAME_DESC(z,n,_)                   \
	const std::string& name##n, const std::string& description##n

	#define META_MAKE_METHODMETA_PARAMETERDESC(z,n,_)                              \
	template <typename R BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>          \
	void createMethodMeta(const std::string& name, const std::string& comment,     \
	                      BOOST_PP_ENUM(n,META_MAKE_METHODMETA_CONSTSTRINGREF_NAME_DESC,nil)) { \
		MethodMetaPtr m(new MethodMeta(name, comment));                            \
		m->returnType = createMetaOrVoidHelper<R>();                               \
		BOOST_PP_REPEAT_ ## z(n, META_MAKE_METHODMETA_PUSH_PARAMETERDESC, nil)     \
		addMethod(m);                                                              \
	}

	BOOST_PP_REPEAT_FROM_TO(1,BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),META_MAKE_METHODMETA_PARAMETERDESC,nil)
	#undef META_MAKE_METHODMETA_PARAMETERDESC
	#undef META_MAKE_METHODMETA_CONSTSTRINGREF_NAME_DESC
	#undef META_MAKE_METHODMETA_PUSH_PARAMETERDESC

public:

	// TODO: add the property qualifier just once at the end

	template<typename T>
	void property(const char* name, T& member, const char* comment,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		Base::property(name,member,comment,std::move(hint), flags);
		//mCurrentMeta->addQualifier(TypeMeta::TYPE_PROPERTY);
	}

	template<typename T>
	void property(const char* name, const std::string& id, T& member,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		Base::property(name,id,member,comment,std::move(hint), flags);
		//mCurrentMeta->addQualifier(TypeMeta::TYPE_PROPERTY);
	}

	template<typename T>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		Base::property<T>(name,member,setter,comment,std::move(hint), flags);
		//mCurrentMeta->addQualifier(TypeMeta::TYPE_PROPERTY);
	}

	template<typename T>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		Base::property(name,getter,setter,comment,std::move(hint), flags);
		//mCurrentMeta->addQualifier(TypeMeta::TYPE_PROPERTY);
	}

	template<typename T, typename U>
	void property(const char* name, T& member, const char* comment,
	              const U& defaultValue, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		Base::property(name,member,comment,defaultValue,std::move(hint), flags);
		//mCurrentMeta->addQualifier(TypeMeta::TYPE_PROPERTY);
	}

	template<typename T, typename U>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		Base::property(name,member,setter,comment,defaultValue, std::move(hint), flags);
		//mCurrentMeta->addQualifier(TypeMeta::TYPE_PROPERTY);
	}

	template<typename T, typename U>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		Base::property(name,getter,setter,comment,defaultValue,std::move(hint), flags);
		//mCurrentMeta->addQualifier(TypeMeta::TYPE_PROPERTY);
	}

	template<typename T>
	void delegate(T& member, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		if (flags & REFLECT_CTRLFLAG_TEMP_TRACKING) {
			this->This()->pushObjectTrackingStore();
			this->invokeMember(member, ReflectMemberMeta("@transparent", "@transparent", "delegated") );
			this->This()->popObjectTrackingStore();
		} else
			this->invokeMember(member, ReflectMemberMeta("@transparent", "@transparent", "delegated") );
	}

	template<typename T>
	void delegate(const T& member, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(member, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMember(a, ReflectMemberMeta("@transparent", "@transparent", "delegated") );
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void delegate(Getter<T> getter, Setter<T> setter,
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		auto a = makeAccessor(getter, setter);
		this->This()->pushObjectTrackingStore();
		this->invokeMember(a, ReflectMemberMeta("@transparent", "@transparent", "delegated") );
		this->This()->popObjectTrackingStore();
	}

	template<typename T>
	void collectionItems(const std::string& countName, bool blockdump = false)
	{
		TypeWithoutObjectAbstractHelper<T,std::is_abstract<T>::value>::invoke(this);

		assert(mCurrentMeta);
		mCurrentMeta->addQualifier(TypeMeta::TYPE_COLLECTION);

		assert(mParentMeta);
		if (blockdump) {
			mParentMeta->members.back().name = "@items_blockdump";
			mParentMeta->members.back().comment = countName;
		} else {
			mParentMeta->members.back().name = "@items";
			mParentMeta->members.back().comment = countName;
		}
	}

protected:

	MetaTypeDatabase& mMetaDatabase;
	TypeMetaPtr     mCurrentMeta;
	CompoundMetaPtr mParentMeta;

	CompoundMeta::Member* mCurrentVersion;
};


///@cond INTERNAL

/**
 * Specialization for MetaSerializer.
 * 
 * Nothing is transparent to the metaserializer. Except PlainArray.
 */

template<typename T>
class IsTransparentSerializableHelper<T, MetaSerializer> : public std::false_type {};

template<typename T>
class IsTransparentSerializableHelper<serialization::PlainArray<T>, MetaSerializer> : public std::true_type {};

namespace serialization {

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialization for MetaSerializer.
 * 
 * Specialize the container item reflection so that it only calls collectionItems()
 * on MetaSerializer, just declaring that there is a sequence of items with the
 * specified type (independent of the current number of items in the container).
 */

template<typename Container>
struct ReflectCollectionItems<MetaSerializer, Container>
{
	static void reflect(MetaSerializer& r, Container& c)
	{
		typedef typename Container::value_type rawtype;
		typedef typename remove_cvref<rawtype>::type type;

		r.template collectionItems<type>("@itemcount");
	}
};

template<typename Allocator>
struct ReflectReadBoolVectorItems<MetaSerializer, Allocator>
{
	static void reflect(MetaSerializer& r, std::vector<bool, Allocator>& c)
	{
		r.template collectionItems<bool>("@itemcount");
	}
};

template<typename Container>
struct ReflectReadSetItems<MetaSerializer, Container>
{
	typedef typename Container::value_type rawtype;
	typedef typename remove_cvref<rawtype>::type type;

	static void reflect(MetaSerializer& r, Container& c)
	{
		r.template collectionItems<type>("@itemcount");
	}
};

template<typename Container>
struct ReflectReadMapItems<MetaSerializer, Container>
{
	typedef typename Container::key_type rawkeytype;
	typedef typename Container::mapped_type rawmappedtype;
	typedef typename std::pair<typename remove_cvref<rawkeytype>::type,
	                           typename remove_cvref<rawmappedtype>::type> type;

	static void reflect(MetaSerializer& r, Container& c)
	{
		r.template collectionItems<type>("@itemcount");
	}
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
